/**
 * Copyright 2017 Pivotal Software, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.metrics.boot;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;
import org.springframework.metrics.instrument.MeterRegistry;
import org.springframework.metrics.instrument.TagFormatter;
import org.springframework.metrics.instrument.web.MetricsHandlerInterceptor;
import org.springframework.metrics.instrument.web.WebmvcTagConfigurer;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;

/**
 * Configures instrumentation of Spring Web MVC servlet-based request mappings.
 *
 * We continue to use the deprecated WebMvcConfigurerAdapter for backwards compatibility
 * with Spring Framework 4.X.
 *
 * @author Jon Schneider
 */
@SuppressWarnings("deprecation")
@Configuration
class InstrumentServletRequestConfiguration extends org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter {
    @Autowired
    MeterRegistry registry;

    @Autowired
    TagFormatter formatter;

    @Autowired(required = false)
    WebmvcTagConfigurer tagConfigurer;

    @Bean
    @ConditionalOnMissingBean(WebmvcTagConfigurer.class)
    WebmvcTagConfigurer webmvcTagConfigurer(TagFormatter tagFormatter) {
        if(tagConfigurer != null)
            return tagConfigurer;
        this.tagConfigurer = new WebmvcTagConfigurer(tagFormatter);
        return this.tagConfigurer;
    }

    @Autowired
    Environment environment;

    @Bean
    MetricsHandlerInterceptor webMetricsInterceptor() {
        return new MetricsHandlerInterceptor(registry, webmvcTagConfigurer(formatter),
                environment.getProperty("spring.metrics.web.server_requests.name", "http_server_requests"));
    }

    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        registry.addInterceptor(webMetricsInterceptor());
    }
}