/*
 * Copyright 2002-2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.beans.factory.aot;

import java.util.LinkedHashMap;
import java.util.Map;

import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.RegisteredBean;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

/**
 * {@link BeanFactoryInitializationAotProcessor} that contributes code to
 * register beans.
 *
 * @author Phillip Webb
 * @author Sebastien Deleuze
 * @since 6.0
 */
class BeanRegistrationsAotProcessor implements BeanFactoryInitializationAotProcessor {

	@Override
	public BeanRegistrationsAotContribution processAheadOfTime(ConfigurableListableBeanFactory beanFactory) {
		BeanDefinitionMethodGeneratorFactory beanDefinitionMethodGeneratorFactory =
				new BeanDefinitionMethodGeneratorFactory(beanFactory);
		Map<String, BeanDefinitionMethodGenerator> registrations = new LinkedHashMap<>();
		MultiValueMap<String, String> aliases = new LinkedMultiValueMap<>();
		for (String beanName : beanFactory.getBeanDefinitionNames()) {
			RegisteredBean registeredBean = RegisteredBean.of(beanFactory, beanName);
			BeanDefinitionMethodGenerator beanDefinitionMethodGenerator = beanDefinitionMethodGeneratorFactory
					.getBeanDefinitionMethodGenerator(registeredBean);
			if (beanDefinitionMethodGenerator != null) {
				registrations.put(beanName, beanDefinitionMethodGenerator);
			}
			for (String alias : beanFactory.getAliases(beanName)) {
				aliases.add(beanName, alias);
			}
		}
		if (registrations.isEmpty()) {
			return null;
		}
		return new BeanRegistrationsAotContribution(registrations, aliases);
	}

}
