/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.web.socket.handler;

import java.io.IOException;
import java.util.Queue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.web.socket.CloseStatus;
import org.springframework.web.socket.WebSocketMessage;
import org.springframework.web.socket.WebSocketSession;
import org.springframework.web.socket.handler.SessionLimitExceededException;
import org.springframework.web.socket.handler.WebSocketSessionDecorator;

public class ConcurrentWebSocketSessionDecorator
extends WebSocketSessionDecorator {
    private static final Log logger = LogFactory.getLog(ConcurrentWebSocketSessionDecorator.class);
    private final int sendTimeLimit;
    private final int bufferSizeLimit;
    private final Queue<WebSocketMessage<?>> buffer = new LinkedBlockingQueue();
    private final AtomicInteger bufferSize = new AtomicInteger();
    private volatile long sendStartTime;
    private volatile boolean limitExceeded;
    private volatile boolean closeInProgress;
    private final Lock flushLock = new ReentrantLock();
    private final Lock closeLock = new ReentrantLock();

    public ConcurrentWebSocketSessionDecorator(WebSocketSession delegate, int sendTimeLimit, int bufferSizeLimit) {
        super(delegate);
        this.sendTimeLimit = sendTimeLimit;
        this.bufferSizeLimit = bufferSizeLimit;
    }

    public int getSendTimeLimit() {
        return this.sendTimeLimit;
    }

    public int getBufferSizeLimit() {
        return this.bufferSizeLimit;
    }

    public int getBufferSize() {
        return this.bufferSize.get();
    }

    public long getTimeSinceSendStarted() {
        long start = this.sendStartTime;
        return start > 0L ? System.currentTimeMillis() - start : 0L;
    }

    @Override
    public void sendMessage(WebSocketMessage<?> message) throws IOException {
        if (this.shouldNotSend()) {
            return;
        }
        this.buffer.add(message);
        this.bufferSize.addAndGet(message.getPayloadLength());
        do {
            if (this.tryFlushMessageBuffer()) continue;
            if (logger.isTraceEnabled()) {
                logger.trace((Object)String.format("Another send already in progress: session id '%s':, \"in-progress\" send time %d (ms), buffer size %d bytes", this.getId(), this.getTimeSinceSendStarted(), this.getBufferSize()));
            }
            this.checkSessionLimits();
            break;
        } while (!this.buffer.isEmpty() && !this.shouldNotSend());
    }

    private boolean shouldNotSend() {
        return this.limitExceeded || this.closeInProgress;
    }

    private boolean tryFlushMessageBuffer() throws IOException {
        if (this.flushLock.tryLock()) {
            try {
                WebSocketMessage<?> message;
                while ((message = this.buffer.poll()) != null) {
                    if (this.shouldNotSend()) {
                        break;
                    }
                    this.bufferSize.addAndGet(message.getPayloadLength() * -1);
                    this.sendStartTime = System.currentTimeMillis();
                    this.getDelegate().sendMessage(message);
                    this.sendStartTime = 0L;
                }
            }
            finally {
                this.sendStartTime = 0L;
                this.flushLock.unlock();
            }
            return true;
        }
        return false;
    }

    private void checkSessionLimits() {
        if (!this.shouldNotSend() && this.closeLock.tryLock()) {
            try {
                if (this.getTimeSinceSendStarted() > (long)this.getSendTimeLimit()) {
                    String format = "Message send time %d (ms) for session '%s' exceeded the allowed limit %d";
                    String reason = String.format(format, this.getTimeSinceSendStarted(), this.getId(), this.getSendTimeLimit());
                    this.limitExceeded(reason);
                } else if (this.getBufferSize() > this.getBufferSizeLimit()) {
                    String format = "The send buffer size %d bytes for session '%s' exceeded the allowed limit %d";
                    String reason = String.format(format, this.getBufferSize(), this.getId(), this.getBufferSizeLimit());
                    this.limitExceeded(reason);
                }
            }
            finally {
                this.closeLock.unlock();
            }
        }
    }

    private void limitExceeded(String reason) {
        this.limitExceeded = true;
        throw new SessionLimitExceededException(reason, CloseStatus.SESSION_NOT_RELIABLE);
    }

    @Override
    public void close(CloseStatus status) throws IOException {
        this.closeLock.lock();
        try {
            if (this.closeInProgress) {
                return;
            }
            if (!CloseStatus.SESSION_NOT_RELIABLE.equals(status)) {
                try {
                    this.checkSessionLimits();
                }
                catch (SessionLimitExceededException sessionLimitExceededException) {
                    // empty catch block
                }
                if (this.limitExceeded) {
                    if (logger.isDebugEnabled()) {
                        logger.debug((Object)("Changing close status " + status + " to SESSION_NOT_RELIABLE."));
                    }
                    status = CloseStatus.SESSION_NOT_RELIABLE;
                }
            }
            this.closeInProgress = true;
            super.close(status);
        }
        finally {
            this.closeLock.unlock();
        }
    }

    @Override
    public String toString() {
        return this.getDelegate().toString();
    }
}

