/**
 * <h1>Amazon ECS Construct Library</h1>
 * <p>
 * This package contains constructs for working with <strong>Amazon Elastic Container
 * Service</strong> (Amazon ECS).
 * <p>
 * Amazon Elastic Container Service (Amazon ECS) is a fully managed container orchestration service.
 * <p>
 * For further information on Amazon ECS,
 * see the <a href="https://docs.aws.amazon.com/ecs">Amazon ECS documentation</a>
 * <p>
 * The following example creates an Amazon ECS cluster, adds capacity to it, and
 * runs a service on it:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * // Create an ECS cluster
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * 
 * // Add capacity to it
 * cluster.addCapacity("DefaultAutoScalingGroupCapacity", AddCapacityOptions.builder()
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .desiredCapacity(3)
 *         .build());
 * 
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * 
 * taskDefinition.addContainer("DefaultContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(512)
 *         .build());
 * 
 * // Instantiate an Amazon ECS Service
 * Ec2Service ecsService = Ec2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .build();
 * </pre></blockquote>
 * <p>
 * For a set of constructs defining common ECS architectural patterns, see the <code>&#64;aws-cdk/aws-ecs-patterns</code> package.
 * <p>
 * <h2>Launch Types: AWS Fargate vs Amazon EC2</h2>
 * <p>
 * There are two sets of constructs in this library; one to run tasks on Amazon EC2 and
 * one to run tasks on AWS Fargate.
 * <p>
 * <ul>
 * <li>Use the <code>Ec2TaskDefinition</code> and <code>Ec2Service</code> constructs to run tasks on Amazon EC2 instances running in your account.</li>
 * <li>Use the <code>FargateTaskDefinition</code> and <code>FargateService</code> constructs to run tasks on
 * instances that are managed for you by AWS.</li>
 * <li>Use the <code>ExternalTaskDefinition</code> and <code>ExternalService</code> constructs to run AWS ECS Anywhere tasks on self-managed infrastructure.</li>
 * </ul>
 * <p>
 * Here are the main differences:
 * <p>
 * <ul>
 * <li><strong>Amazon EC2</strong>: instances are under your control. Complete control of task to host
 * allocation. Required to specify at least a memory reservation or limit for
 * every container. Can use Host, Bridge and AwsVpc networking modes. Can attach
 * Classic Load Balancer. Can share volumes between container and host.</li>
 * <li><strong>AWS Fargate</strong>: tasks run on AWS-managed instances, AWS manages task to host
 * allocation for you. Requires specification of memory and cpu sizes at the
 * taskdefinition level. Only supports AwsVpc networking modes and
 * Application/Network Load Balancers. Only the AWS log driver is supported.
 * Many host features are not supported such as adding kernel capabilities
 * and mounting host devices/volumes inside the container.</li>
 * <li><strong>AWS ECSAnywhere</strong>: tasks are run and managed by AWS ECS Anywhere on infrastructure owned by the customer. Only Bridge networking mode is supported. Does not support autoscaling, load balancing, cloudmap or attachment of volumes.</li>
 * </ul>
 * <p>
 * For more information on Amazon EC2 vs AWS Fargate, networking and ECS Anywhere see the AWS Documentation:
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate</a>,
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task Networking</a>,
 * <a href="https://aws.amazon.com/ecs/anywhere/">ECS Anywhere</a>
 * <p>
 * <h2>Clusters</h2>
 * <p>
 * A <code>Cluster</code> defines the infrastructure to run your
 * tasks on. You can run many tasks on a single cluster.
 * <p>
 * The following code creates a cluster that can run AWS Fargate tasks:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following code imports an existing cluster using the ARN which can be used to
 * import an Amazon ECS service either EC2 or Fargate.
 * <p>
 * <blockquote><pre>
 * String clusterArn = "arn:aws:ecs:us-east-1:012345678910:cluster/clusterName";
 * 
 * ICluster cluster = Cluster.fromClusterArn(this, "Cluster", clusterArn);
 * </pre></blockquote>
 * <p>
 * To use tasks with Amazon EC2 launch-type, you have to add capacity to
 * the cluster in order for tasks to be scheduled on your instances.  Typically,
 * you add an AutoScalingGroup with instances running the latest
 * Amazon ECS-optimized AMI to the cluster. There is a method to build and add such an
 * AutoScalingGroup automatically, or you can supply a customized AutoScalingGroup
 * that you construct yourself. It's possible to add multiple AutoScalingGroups
 * with various instance types.
 * <p>
 * The following example creates an Amazon ECS cluster and adds capacity to it:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * 
 * // Either add default capacity
 * cluster.addCapacity("DefaultAutoScalingGroupCapacity", AddCapacityOptions.builder()
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .desiredCapacity(3)
 *         .build());
 * 
 * // Or add customized capacity. Be sure to start the Amazon ECS-optimized AMI.
 * AutoScalingGroup autoScalingGroup = AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .machineImage(EcsOptimizedImage.amazonLinux())
 *         // Or use Amazon ECS-Optimized Amazon Linux 2 AMI
 *         // machineImage: EcsOptimizedImage.amazonLinux2(),
 *         .desiredCapacity(3)
 *         .build();
 * 
 * cluster.addAutoScalingGroup(autoScalingGroup);
 * </pre></blockquote>
 * <p>
 * If you omit the property <code>vpc</code>, the construct will create a new VPC with two AZs.
 * <p>
 * By default, all machine images will auto-update to the latest version
 * on each deployment, causing a replacement of the instances in your AutoScalingGroup
 * if the AMI has been updated since the last deployment.
 * <p>
 * If task draining is enabled, ECS will transparently reschedule tasks on to the new
 * instances before terminating your old instances. If you have disabled task draining,
 * the tasks will be terminated along with the instance. To prevent that, you
 * can pick a non-updating AMI by passing <code>cacheInContext: true</code>, but be sure
 * to periodically update to the latest AMI manually by using the <a href="https://docs.aws.amazon.com/cdk/latest/guide/context.html">CDK CLI
 * context management commands</a>:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * AutoScalingGroup autoScalingGroup = AutoScalingGroup.Builder.create(this, "ASG")
 *         .machineImage(EcsOptimizedImage.amazonLinux(EcsOptimizedImageOptions.builder().cachedInContext(true).build()))
 *         .vpc(vpc)
 *         .instanceType(new InstanceType("t2.micro"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Bottlerocket</h3>
 * <p>
 * <a href="https://aws.amazon.com/bottlerocket/">Bottlerocket</a> is a Linux-based open source operating system that is
 * purpose-built by AWS for running containers. You can launch Amazon ECS container instances with the Bottlerocket AMI.
 * <p>
 * The following example will create a capacity with self-managed Amazon EC2 capacity of 2 <code>c5.large</code> Linux instances running with <code>Bottlerocket</code> AMI.
 * <p>
 * The following example adds Bottlerocket capacity to the cluster:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * 
 * cluster.addCapacity("bottlerocket-asg", AddCapacityOptions.builder()
 *         .minCapacity(2)
 *         .instanceType(new InstanceType("c5.large"))
 *         .machineImage(new BottleRocketImage())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>ARM64 (Graviton) Instances</h3>
 * <p>
 * To launch instances with ARM64 hardware, you can use the Amazon ECS-optimized
 * Amazon Linux 2 (arm64) AMI. Based on Amazon Linux 2, this AMI is recommended
 * for use when launching your EC2 instances that are powered by Arm-based AWS
 * Graviton Processors.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * 
 * cluster.addCapacity("graviton-cluster", AddCapacityOptions.builder()
 *         .minCapacity(2)
 *         .instanceType(new InstanceType("c6g.large"))
 *         .machineImage(EcsOptimizedImage.amazonLinux2(AmiHardwareType.ARM))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Bottlerocket is also supported:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * 
 * cluster.addCapacity("graviton-cluster", AddCapacityOptions.builder()
 *         .minCapacity(2)
 *         .instanceType(new InstanceType("c6g.large"))
 *         .machineImageType(MachineImageType.BOTTLEROCKET)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Spot Instances</h3>
 * <p>
 * To add spot instances into the cluster, you must specify the <code>spotPrice</code> in the <code>ecs.AddCapacityOptions</code> and optionally enable the <code>spotInstanceDraining</code> property.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * 
 * // Add an AutoScalingGroup with spot instances to the existing cluster
 * cluster.addCapacity("AsgSpot", AddCapacityOptions.builder()
 *         .maxCapacity(2)
 *         .minCapacity(2)
 *         .desiredCapacity(2)
 *         .instanceType(new InstanceType("c5.xlarge"))
 *         .spotPrice("0.0735")
 *         // Enable the Automated Spot Draining support for Amazon ECS
 *         .spotInstanceDraining(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>SNS Topic Encryption</h3>
 * <p>
 * When the <code>ecs.AddCapacityOptions</code> that you provide has a non-zero <code>taskDrainTime</code> (the default) then an SNS topic and Lambda are created to ensure that the
 * cluster's instances have been properly drained of tasks before terminating. The SNS Topic is sent the instance-terminating lifecycle event from the AutoScalingGroup,
 * and the Lambda acts on that event. If you wish to engage <a href="https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html">server-side encryption</a> for this SNS Topic
 * then you may do so by providing a KMS key for the <code>topicEncryptionKey</code> property of <code>ecs.AddCapacityOptions</code>.
 * <p>
 * <blockquote><pre>
 * // Given
 * Cluster cluster;
 * Key key;
 * 
 * // Then, use that key to encrypt the lifecycle-event SNS Topic.
 * cluster.addCapacity("ASGEncryptedSNS", AddCapacityOptions.builder()
 *         .instanceType(new InstanceType("t2.xlarge"))
 *         .desiredCapacity(3)
 *         .topicEncryptionKey(key)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Task definitions</h2>
 * <p>
 * A task definition describes what a single copy of a <strong>task</strong> should look like.
 * A task definition has one or more containers; typically, it has one
 * main container (the <em>default container</em> is the first one that's added
 * to the task definition, and it is marked <em>essential</em>) and optionally
 * some supporting containers which are used to support the main container,
 * doings things like upload logs or metrics to monitoring services.
 * <p>
 * To run a task or service with Amazon EC2 launch type, use the <code>Ec2TaskDefinition</code>. For AWS Fargate tasks/services, use the
 * <code>FargateTaskDefinition</code>. For AWS ECS Anywhere use the <code>ExternalTaskDefinition</code>. These classes
 * provide simplified APIs that only contain properties relevant for each specific launch type.
 * <p>
 * For a <code>FargateTaskDefinition</code>, specify the task size (<code>memoryLimitMiB</code> and <code>cpu</code>):
 * <p>
 * <blockquote><pre>
 * FargateTaskDefinition fargateTaskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .build();
 * </pre></blockquote>
 * <p>
 * On Fargate Platform Version 1.4.0 or later, you may specify up to 200GiB of
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/fargate-task-storage.html#fargate-task-storage-pv14">ephemeral storage</a>:
 * <p>
 * <blockquote><pre>
 * FargateTaskDefinition fargateTaskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .ephemeralStorageGiB(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add containers to a task definition, call <code>addContainer()</code>:
 * <p>
 * <blockquote><pre>
 * FargateTaskDefinition fargateTaskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .build();
 * ContainerDefinition container = fargateTaskDefinition.addContainer("WebContainer", ContainerDefinitionOptions.builder()
 *         // Use an image from DockerHub
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * For a <code>Ec2TaskDefinition</code>:
 * <p>
 * <blockquote><pre>
 * Ec2TaskDefinition ec2TaskDefinition = Ec2TaskDefinition.Builder.create(this, "TaskDef")
 *         .networkMode(NetworkMode.BRIDGE)
 *         .build();
 * 
 * ContainerDefinition container = ec2TaskDefinition.addContainer("WebContainer", ContainerDefinitionOptions.builder()
 *         // Use an image from DockerHub
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .build());
 * </pre></blockquote>
 * <p>
 * For an <code>ExternalTaskDefinition</code>:
 * <p>
 * <blockquote><pre>
 * ExternalTaskDefinition externalTaskDefinition = new ExternalTaskDefinition(this, "TaskDef");
 * 
 * ContainerDefinition container = externalTaskDefinition.addContainer("WebContainer", ContainerDefinitionOptions.builder()
 *         // Use an image from DockerHub
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can specify container properties when you add them to the task definition, or with various methods, e.g.:
 * <p>
 * To add a port mapping when adding a container to the task definition, specify the <code>portMappings</code> option:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * 
 * 
 * taskDefinition.addContainer("WebContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .portMappings(List.of(PortMapping.builder().containerPort(3000).build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * To add port mappings directly to a container definition, call <code>addPortMappings()</code>:
 * <p>
 * <blockquote><pre>
 * ContainerDefinition container;
 * 
 * 
 * container.addPortMappings(PortMapping.builder()
 *         .containerPort(3000)
 *         .build());
 * </pre></blockquote>
 * <p>
 * To add data volumes to a task definition, call <code>addVolume()</code>:
 * <p>
 * <blockquote><pre>
 * FargateTaskDefinition fargateTaskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryLimitMiB(512)
 *         .cpu(256)
 *         .build();
 * Map&lt;String, Object&gt; volume = Map.of(
 *         // Use an Elastic FileSystem
 *         "name", "mydatavolume",
 *         "efsVolumeConfiguration", Map.of(
 *                 "fileSystemId", "EFS"));
 * 
 * void container = fargateTaskDefinition.addVolume(volume);
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: ECS Anywhere doesn't support volume attachments in the task definition.
 * <p>
 * </blockquote>
 * <p>
 * To use a TaskDefinition that can be used with either Amazon EC2 or
 * AWS Fargate launch types, use the <code>TaskDefinition</code> construct.
 * <p>
 * When creating a task definition you have to specify what kind of
 * tasks you intend to run: Amazon EC2, AWS Fargate, or both.
 * The following example uses both:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition = TaskDefinition.Builder.create(this, "TaskDef")
 *         .memoryMiB("512")
 *         .cpu("256")
 *         .networkMode(NetworkMode.AWS_VPC)
 *         .compatibility(Compatibility.EC2_AND_FARGATE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Images</h3>
 * <p>
 * Images supply the software that runs inside the container. Images can be
 * obtained from either DockerHub or from ECR repositories, built directly from a local Dockerfile, or use an existing tarball.
 * <p>
 * <ul>
 * <li><code>ecs.ContainerImage.fromRegistry(imageName)</code>: use a public image.</li>
 * <li><code>ecs.ContainerImage.fromRegistry(imageName, { credentials: mySecret })</code>: use a private image that requires credentials.</li>
 * <li><code>ecs.ContainerImage.fromEcrRepository(repo, tag)</code>: use the given ECR repository as the image
 * to start. If no tag is provided, "latest" is assumed.</li>
 * <li><code>ecs.ContainerImage.fromAsset('./image')</code>: build and upload an
 * image directly from a <code>Dockerfile</code> in your source directory.</li>
 * <li><code>ecs.ContainerImage.fromDockerImageAsset(asset)</code>: uses an existing
 * <code>&#64;aws-cdk/aws-ecr-assets.DockerImageAsset</code> as a container image.</li>
 * <li><code>ecs.ContainerImage.fromTarball(file)</code>: use an existing tarball.</li>
 * <li><code>new ecs.TagParameterContainerImage(repository)</code>: use the given ECR repository as the image
 * but a CloudFormation parameter as the tag.</li>
 * </ul>
 * <p>
 * <h3>Environment variables</h3>
 * <p>
 * To pass environment variables to the container, you can use the <code>environment</code>, <code>environmentFiles</code>, and <code>secrets</code> props.
 * <p>
 * <blockquote><pre>
 * Secret secret;
 * Secret dbSecret;
 * StringParameter parameter;
 * TaskDefinition taskDefinition;
 * Bucket s3Bucket;
 * 
 * 
 * ContainerDefinition newContainer = taskDefinition.addContainer("container", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .environment(Map.of( // clear text, not for sensitive data
 *                 "STAGE", "prod"))
 *         .environmentFiles(List.of(EnvironmentFile.fromAsset("./demo-env-file.env"), EnvironmentFile.fromBucket(s3Bucket, "assets/demo-env-file.env")))
 *         .secrets(Map.of( // Retrieved from AWS Secrets Manager or AWS Systems Manager Parameter Store at container start-up.
 *                 "SECRET", Secret.fromSecretsManager(secret),
 *                 "DB_PASSWORD", Secret.fromSecretsManager(dbSecret, "password"),  // Reference a specific JSON field, (requires platform version 1.4.0 or later for Fargate tasks)
 *                 "API_KEY", Secret.fromSecretsManagerVersion(secret, SecretVersionInfo.builder().versionId("12345").build(), "apiKey"),  // Reference a specific version of the secret by its version id or version stage (requires platform version 1.4.0 or later for Fargate tasks)
 *                 "PARAMETER", Secret.fromSsmParameter(parameter)))
 *         .build());
 * newContainer.addEnvironment("QUEUE_NAME", "MyQueue");
 * </pre></blockquote>
 * <p>
 * The task execution role is automatically granted read permissions on the secrets/parameters. Support for environment
 * files is restricted to the EC2 launch type for files hosted on S3. Further details provided in the AWS documentation
 * about <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html">specifying environment variables</a>.
 * <p>
 * <h3>System controls</h3>
 * <p>
 * To set system controls (kernel parameters) on the container, use the <code>systemControls</code> prop:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * 
 * 
 * taskDefinition.addContainer("container", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryLimitMiB(1024)
 *         .systemControls(List.of(SystemControl.builder()
 *                 .namespace("net")
 *                 .value("ipv4.tcp_tw_recycle")
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Using Windows containers on Fargate</h3>
 * <p>
 * AWS Fargate supports Amazon ECS Windows containers. For more details, please see this <a href="https://aws.amazon.com/tw/blogs/containers/running-windows-containers-with-amazon-ecs-on-aws-fargate/">blog post</a>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the Windows container to start
 * FargateTaskDefinition taskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .runtimePlatform(RuntimePlatform.builder()
 *                 .operatingSystemFamily(OperatingSystemFamily.WINDOWS_SERVER_2019_CORE)
 *                 .cpuArchitecture(CpuArchitecture.X86_64)
 *                 .build())
 *         .cpu(1024)
 *         .memoryLimitMiB(2048)
 *         .build();
 * 
 * taskDefinition.addContainer("windowsservercore", ContainerDefinitionOptions.builder()
 *         .logging(LogDriver.awsLogs(AwsLogDriverProps.builder().streamPrefix("win-iis-on-fargate").build()))
 *         .portMappings(List.of(PortMapping.builder().containerPort(80).build()))
 *         .image(ContainerImage.fromRegistry("mcr.microsoft.com/windows/servercore/iis:windowsservercore-ltsc2019"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Using Graviton2 with Fargate</h3>
 * <p>
 * AWS Graviton2 supports AWS Fargate. For more details, please see this <a href="https://aws.amazon.com/blogs/aws/announcing-aws-graviton2-support-for-aws-fargate-get-up-to-40-better-price-performance-for-your-serverless-containers/">blog post</a>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for running container on Graviton Runtime.
 * FargateTaskDefinition taskDefinition = FargateTaskDefinition.Builder.create(this, "TaskDef")
 *         .runtimePlatform(RuntimePlatform.builder()
 *                 .operatingSystemFamily(OperatingSystemFamily.LINUX)
 *                 .cpuArchitecture(CpuArchitecture.ARM64)
 *                 .build())
 *         .cpu(1024)
 *         .memoryLimitMiB(2048)
 *         .build();
 * 
 * taskDefinition.addContainer("webarm64", ContainerDefinitionOptions.builder()
 *         .logging(LogDriver.awsLogs(AwsLogDriverProps.builder().streamPrefix("graviton2-on-fargate").build()))
 *         .portMappings(List.of(PortMapping.builder().containerPort(80).build()))
 *         .image(ContainerImage.fromRegistry("public.ecr.aws/nginx/nginx:latest-arm64v8"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Service</h2>
 * <p>
 * A <code>Service</code> instantiates a <code>TaskDefinition</code> on a <code>Cluster</code> a given number of
 * times, optionally associating them with a load balancer.
 * If a task fails,
 * Amazon ECS automatically restarts the task.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .desiredCount(5)
 *         .build();
 * </pre></blockquote>
 * <p>
 * ECS Anywhere service definition looks like:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * ExternalService service = ExternalService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .desiredCount(5)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <code>Services</code> by default will create a security group if not provided.
 * If you'd like to specify which security groups to use you can override the <code>securityGroups</code> property.
 * <p>
 * <h3>Deployment circuit breaker and rollback</h3>
 * <p>
 * Amazon ECS <a href="https://aws.amazon.com/tw/blogs/containers/announcing-amazon-ecs-deployment-circuit-breaker/">deployment circuit breaker</a>
 * automatically rolls back unhealthy service deployments without the need for manual intervention. Use <code>circuitBreaker</code> to enable
 * deployment circuit breaker and optionally enable <code>rollback</code> for automatic rollback. See <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Using the deployment circuit breaker</a>
 * for more details.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * FargateService service = FargateService.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .circuitBreaker(DeploymentCircuitBreaker.builder().rollback(true).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: ECS Anywhere doesn't support deployment circuit breakers and rollback.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Include an application/network load balancer</h3>
 * <p>
 * <code>Services</code> are load balancing targets and can be added to a target group, which will be attached to an application/network load balancers:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * FargateService service = FargateService.Builder.create(this, "Service").cluster(cluster).taskDefinition(taskDefinition).build();
 * 
 * ApplicationLoadBalancer lb = ApplicationLoadBalancer.Builder.create(this, "LB").vpc(vpc).internetFacing(true).build();
 * ApplicationListener listener = lb.addListener("Listener", BaseApplicationListenerProps.builder().port(80).build());
 * ApplicationTargetGroup targetGroup1 = listener.addTargets("ECS1", AddApplicationTargetsProps.builder()
 *         .port(80)
 *         .targets(List.of(service))
 *         .build());
 * ApplicationTargetGroup targetGroup2 = listener.addTargets("ECS2", AddApplicationTargetsProps.builder()
 *         .port(80)
 *         .targets(List.of(service.loadBalancerTarget(LoadBalancerTargetOptions.builder()
 *                 .containerName("MyContainer")
 *                 .containerPort(8080)
 *                 .build())))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: ECS Anywhere doesn't support application/network load balancers.
 * <p>
 * </blockquote>
 * <p>
 * Note that in the example above, the default <code>service</code> only allows you to register the first essential container or the first mapped port on the container as a target and add it to a new target group. To have more control over which container and port to register as targets, you can use <code>service.loadBalancerTarget()</code> to return a load balancing target for a specific container and port.
 * <p>
 * Alternatively, you can also create all load balancer targets to be registered in this service, add them to target groups, and attach target groups to listeners accordingly.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * Vpc vpc;
 * 
 * FargateService service = FargateService.Builder.create(this, "Service").cluster(cluster).taskDefinition(taskDefinition).build();
 * 
 * ApplicationLoadBalancer lb = ApplicationLoadBalancer.Builder.create(this, "LB").vpc(vpc).internetFacing(true).build();
 * ApplicationListener listener = lb.addListener("Listener", BaseApplicationListenerProps.builder().port(80).build());
 * service.registerLoadBalancerTargets(EcsTarget.builder()
 *         .containerName("web")
 *         .containerPort(80)
 *         .newTargetGroupId("ECS")
 *         .listener(ListenerConfig.applicationListener(listener, AddApplicationTargetsProps.builder()
 *                 .protocol(ApplicationProtocol.HTTPS)
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Using a Load Balancer from a different Stack</h3>
 * <p>
 * If you want to put your Load Balancer and the Service it is load balancing to in
 * different stacks, you may not be able to use the convenience methods
 * <code>loadBalancer.addListener()</code> and <code>listener.addTargets()</code>.
 * <p>
 * The reason is that these methods will create resources in the same Stack as the
 * object they're called on, which may lead to cyclic references between stacks.
 * Instead, you will have to create an <code>ApplicationListener</code> in the service stack,
 * or an empty <code>TargetGroup</code> in the load balancer stack that you attach your
 * service to.
 * <p>
 * See the <a href="https://github.com/aws-samples/aws-cdk-examples/tree/master/typescript/ecs/cross-stack-load-balancer/">ecs/cross-stack-load-balancer example</a>
 * for the alternatives.
 * <p>
 * <h3>Include a classic load balancer</h3>
 * <p>
 * <code>Services</code> can also be directly attached to a classic load balancer as targets:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * Vpc vpc;
 * 
 * Ec2Service service = Ec2Service.Builder.create(this, "Service").cluster(cluster).taskDefinition(taskDefinition).build();
 * 
 * LoadBalancer lb = LoadBalancer.Builder.create(this, "LB").vpc(vpc).build();
 * lb.addListener(LoadBalancerListener.builder().externalPort(80).build());
 * lb.addTarget(service);
 * </pre></blockquote>
 * <p>
 * Similarly, if you want to have more control over load balancer targeting:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * Vpc vpc;
 * 
 * Ec2Service service = Ec2Service.Builder.create(this, "Service").cluster(cluster).taskDefinition(taskDefinition).build();
 * 
 * LoadBalancer lb = LoadBalancer.Builder.create(this, "LB").vpc(vpc).build();
 * lb.addListener(LoadBalancerListener.builder().externalPort(80).build());
 * lb.addTarget(service.loadBalancerTarget(LoadBalancerTargetOptions.builder()
 *         .containerName("MyContainer")
 *         .containerPort(80)
 *         .build()));
 * </pre></blockquote>
 * <p>
 * There are two higher-level constructs available which include a load balancer for you that can be found in the aws-ecs-patterns module:
 * <p>
 * <ul>
 * <li><code>LoadBalancedFargateService</code></li>
 * <li><code>LoadBalancedEc2Service</code></li>
 * </ul>
 * <p>
 * <h2>Task Auto-Scaling</h2>
 * <p>
 * You can configure the task count of a service to match demand. Task auto-scaling is
 * configured by calling <code>autoScaleTaskCount()</code>:
 * <p>
 * <blockquote><pre>
 * ApplicationTargetGroup target;
 * BaseService service;
 * 
 * ScalableTaskCount scaling = service.autoScaleTaskCount(EnableScalingProps.builder().maxCapacity(10).build());
 * scaling.scaleOnCpuUtilization("CpuScaling", CpuUtilizationScalingProps.builder()
 *         .targetUtilizationPercent(50)
 *         .build());
 * 
 * scaling.scaleOnRequestCount("RequestScaling", RequestCountScalingProps.builder()
 *         .requestsPerTarget(10000)
 *         .targetGroup(target)
 *         .build());
 * </pre></blockquote>
 * <p>
 * Task auto-scaling is powered by <em>Application Auto-Scaling</em>.
 * See that section for details.
 * <p>
 * <h2>Integration with CloudWatch Events</h2>
 * <p>
 * To start an Amazon ECS task on an Amazon EC2-backed Cluster, instantiate an
 * <code>&#64;aws-cdk/aws-events-targets.EcsTask</code> instead of an <code>Ec2Service</code>:
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * 
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromAsset(resolve(__dirname, "..", "eventhandler-image")))
 *         .memoryLimitMiB(256)
 *         .logging(AwsLogDriver.Builder.create().streamPrefix("EventDemo").mode(AwsLogDriverMode.NON_BLOCKING).build())
 *         .build());
 * 
 * // An Rule that describes the event trigger (in this case a scheduled run)
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.expression("rate(1 min)"))
 *         .build();
 * 
 * // Pass an environment variable to the container 'TheContainer' in the task
 * rule.addTarget(EcsTask.Builder.create()
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .taskCount(1)
 *         .containerOverrides(List.of(ContainerOverride.builder()
 *                 .containerName("TheContainer")
 *                 .environment(List.of(TaskEnvironmentVariable.builder()
 *                         .name("I_WAS_TRIGGERED")
 *                         .value("From CloudWatch Events")
 *                         .build()))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Log Drivers</h2>
 * <p>
 * Currently Supported Log Drivers:
 * <p>
 * <ul>
 * <li>awslogs</li>
 * <li>fluentd</li>
 * <li>gelf</li>
 * <li>journald</li>
 * <li>json-file</li>
 * <li>splunk</li>
 * <li>syslog</li>
 * <li>awsfirelens</li>
 * <li>Generic</li>
 * </ul>
 * <p>
 * <h3>awslogs Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.awsLogs(AwsLogDriverProps.builder().streamPrefix("EventDemo").build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>fluentd Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.fluentd())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>gelf Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.gelf(GelfLogDriverProps.builder().address("my-gelf-address").build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>journald Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.journald())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>json-file Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.jsonFile())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>splunk Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.splunk(SplunkLogDriverProps.builder()
 *                 .token(SecretValue.secretsManager("my-splunk-token"))
 *                 .url("my-splunk-url")
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>syslog Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.syslog())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>firelens Log Driver</h3>
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.firelens(FireLensLogDriverProps.builder()
 *                 .options(Map.of(
 *                         "Name", "firehose",
 *                         "region", "us-west-2",
 *                         "delivery_stream", "my-stream"))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * To pass secrets to the log configuration, use the <code>secretOptions</code> property of the log configuration. The task execution role is automatically granted read permissions on the secrets/parameters.
 * <p>
 * <blockquote><pre>
 * Secret secret;
 * StringParameter parameter;
 * 
 * 
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(LogDrivers.firelens(FireLensLogDriverProps.builder()
 *                 .options(Map.of())
 *                 .secretOptions(Map.of( // Retrieved from AWS Secrets Manager or AWS Systems Manager Parameter Store
 *                         "apikey", Secret.fromSecretsManager(secret),
 *                         "host", Secret.fromSsmParameter(parameter)))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Generic Log Driver</h3>
 * <p>
 * A generic log driver object exists to provide a lower level abstraction of the log driver configuration.
 * <p>
 * <blockquote><pre>
 * // Create a Task Definition for the container to start
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * taskDefinition.addContainer("TheContainer", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("example-image"))
 *         .memoryLimitMiB(256)
 *         .logging(GenericLogDriver.Builder.create()
 *                 .logDriver("fluentd")
 *                 .options(Map.of(
 *                         "tag", "example-tag"))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>CloudMap Service Discovery</h2>
 * <p>
 * To register your ECS service with a CloudMap Service Registry, you may add the
 * <code>cloudMapOptions</code> property to your service:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * Cluster cluster;
 * 
 * 
 * Ec2Service service = Ec2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .cloudMapOptions(CloudMapOptions.builder()
 *                 // Create A records - useful for AWSVPC network mode.
 *                 .dnsRecordType(DnsRecordType.A)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * With <code>bridge</code> or <code>host</code> network modes, only <code>SRV</code> DNS record types are supported.
 * By default, <code>SRV</code> DNS record types will target the default container and default
 * port. However, you may target a different container and port on the same ECS task:
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * Cluster cluster;
 * 
 * 
 * // Add a container to the task definition
 * ContainerDefinition specificContainer = taskDefinition.addContainer("Container", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("/aws/aws-example-app"))
 *         .memoryLimitMiB(2048)
 *         .build());
 * 
 * // Add a port mapping
 * specificContainer.addPortMappings(PortMapping.builder()
 *         .containerPort(7600)
 *         .protocol(Protocol.TCP)
 *         .build());
 * 
 * Ec2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .cloudMapOptions(CloudMapOptions.builder()
 *                 // Create SRV records - useful for bridge networking
 *                 .dnsRecordType(DnsRecordType.SRV)
 *                 // Targets port TCP port 7600 `specificContainer`
 *                 .container(specificContainer)
 *                 .containerPort(7600)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Associate With a Specific CloudMap Service</h3>
 * <p>
 * You may associate an ECS service with a specific CloudMap service. To do
 * this, use the service's <code>associateCloudMapService</code> method:
 * <p>
 * <blockquote><pre>
 * Service cloudMapService;
 * FargateService ecsService;
 * 
 * 
 * ecsService.associateCloudMapService(AssociateCloudMapServiceOptions.builder()
 *         .service(cloudMapService)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Capacity Providers</h2>
 * <p>
 * There are two major families of Capacity Providers: <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/fargate-capacity-providers.html">AWS
 * Fargate</a>
 * (including Fargate Spot) and EC2 <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html">Auto Scaling
 * Group</a>
 * Capacity Providers. Both are supported.
 * <p>
 * <h3>Fargate Capacity Providers</h3>
 * <p>
 * To enable Fargate capacity providers, you can either set
 * <code>enableFargateCapacityProviders</code> to <code>true</code> when creating your cluster, or by
 * invoking the <code>enableFargateCapacityProviders()</code> method after creating your
 * cluster. This will add both <code>FARGATE</code> and <code>FARGATE_SPOT</code> as available capacity
 * providers on your cluster.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "FargateCPCluster")
 *         .vpc(vpc)
 *         .enableFargateCapacityProviders(true)
 *         .build();
 * 
 * FargateTaskDefinition taskDefinition = new FargateTaskDefinition(this, "TaskDef");
 * 
 * taskDefinition.addContainer("web", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .build());
 * 
 * FargateService.Builder.create(this, "FargateService")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .capacityProviderStrategies(List.of(CapacityProviderStrategy.builder()
 *                 .capacityProvider("FARGATE_SPOT")
 *                 .weight(2)
 *                 .build(), CapacityProviderStrategy.builder()
 *                 .capacityProvider("FARGATE")
 *                 .weight(1)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Auto Scaling Group Capacity Providers</h3>
 * <p>
 * To add an Auto Scaling Group Capacity Provider, first create an EC2 Auto Scaling
 * Group. Then, create an <code>AsgCapacityProvider</code> and pass the Auto Scaling Group to
 * it in the constructor. Then add the Capacity Provider to the cluster. Finally,
 * you can refer to the Provider by its name in your service's or task's Capacity
 * Provider strategy.
 * <p>
 * By default, an Auto Scaling Group Capacity Provider will manage the Auto Scaling
 * Group's size for you. It will also enable managed termination protection, in
 * order to prevent EC2 Auto Scaling from terminating EC2 instances that have tasks
 * running on them. If you want to disable this behavior, set both
 * <code>enableManagedScaling</code> to and <code>enableManagedTerminationProtection</code> to <code>false</code>.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .build();
 * 
 * AutoScalingGroup autoScalingGroup = AutoScalingGroup.Builder.create(this, "ASG")
 *         .vpc(vpc)
 *         .instanceType(new InstanceType("t2.micro"))
 *         .machineImage(EcsOptimizedImage.amazonLinux2())
 *         .minCapacity(0)
 *         .maxCapacity(100)
 *         .build();
 * 
 * AsgCapacityProvider capacityProvider = AsgCapacityProvider.Builder.create(this, "AsgCapacityProvider")
 *         .autoScalingGroup(autoScalingGroup)
 *         .build();
 * cluster.addAsgCapacityProvider(capacityProvider);
 * 
 * Ec2TaskDefinition taskDefinition = new Ec2TaskDefinition(this, "TaskDef");
 * 
 * taskDefinition.addContainer("web", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *         .memoryReservationMiB(256)
 *         .build());
 * 
 * Ec2Service.Builder.create(this, "EC2Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .capacityProviderStrategies(List.of(CapacityProviderStrategy.builder()
 *                 .capacityProvider(capacityProvider.getCapacityProviderName())
 *                 .weight(1)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Elastic Inference Accelerators</h2>
 * <p>
 * Currently, this feature is only supported for services with EC2 launch types.
 * <p>
 * To add elastic inference accelerators to your EC2 instance, first add
 * <code>inferenceAccelerators</code> field to the Ec2TaskDefinition and set the <code>deviceName</code>
 * and <code>deviceType</code> properties.
 * <p>
 * <blockquote><pre>
 * Map&lt;String, String&gt;[] inferenceAccelerators = List.of(Map.of(
 *         "deviceName", "device1",
 *         "deviceType", "eia2.medium"));
 * 
 * Ec2TaskDefinition taskDefinition = Ec2TaskDefinition.Builder.create(this, "Ec2TaskDef")
 *         .inferenceAccelerators(inferenceAccelerators)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To enable using the inference accelerators in the containers, add <code>inferenceAcceleratorResources</code>
 * field and set it to a list of device names used for the inference accelerators. Each value in the
 * list should match a <code>DeviceName</code> for an <code>InferenceAccelerator</code> specified in the task definition.
 * <p>
 * <blockquote><pre>
 * TaskDefinition taskDefinition;
 * 
 * String[] inferenceAcceleratorResources = List.of("device1");
 * 
 * taskDefinition.addContainer("cont", ContainerDefinitionOptions.builder()
 *         .image(ContainerImage.fromRegistry("test"))
 *         .memoryLimitMiB(1024)
 *         .inferenceAcceleratorResources(inferenceAcceleratorResources)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>ECS Exec command</h2>
 * <p>
 * Please note, ECS Exec leverages AWS Systems Manager (SSM). So as a prerequisite for the exec command
 * to work, you need to have the SSM plugin for the AWS CLI installed locally. For more information, see
 * <a href="https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager-working-with-install-plugin.html">Install Session Manager plugin for AWS CLI</a>.
 * <p>
 * To enable the ECS Exec feature for your containers, set the boolean flag <code>enableExecuteCommand</code> to <code>true</code> in
 * your <code>Ec2Service</code> or <code>FargateService</code>.
 * <p>
 * <blockquote><pre>
 * Cluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * Ec2Service service = Ec2Service.Builder.create(this, "Service")
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .enableExecuteCommand(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Enabling logging</h3>
 * <p>
 * You can enable sending logs of your execute session commands to a CloudWatch log group or S3 bucket by configuring
 * the <code>executeCommandConfiguration</code> property for your cluster. The default configuration will send the
 * logs to the CloudWatch Logs using the <code>awslogs</code> log driver that is configured in your task definition. Please note,
 * when using your own <code>logConfiguration</code> the log group or S3 Bucket specified must already be created.
 * <p>
 * To encrypt data using your own KMS Customer Key (CMK), you must create a CMK and provide the key in the <code>kmsKey</code> field
 * of the <code>executeCommandConfiguration</code>. To use this key for encrypting CloudWatch log data or S3 bucket, make sure to associate the key
 * to these resources on creation.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * Key kmsKey = new Key(this, "KmsKey");
 * 
 * // Pass the KMS key in the `encryptionKey` field to associate the key to the log group
 * LogGroup logGroup = LogGroup.Builder.create(this, "LogGroup")
 *         .encryptionKey(kmsKey)
 *         .build();
 * 
 * // Pass the KMS key in the `encryptionKey` field to associate the key to the S3 bucket
 * Bucket execBucket = Bucket.Builder.create(this, "EcsExecBucket")
 *         .encryptionKey(kmsKey)
 *         .build();
 * 
 * Cluster cluster = Cluster.Builder.create(this, "Cluster")
 *         .vpc(vpc)
 *         .executeCommandConfiguration(ExecuteCommandConfiguration.builder()
 *                 .kmsKey(kmsKey)
 *                 .logConfiguration(ExecuteCommandLogConfiguration.builder()
 *                         .cloudWatchLogGroup(logGroup)
 *                         .cloudWatchEncryptionEnabled(true)
 *                         .s3Bucket(execBucket)
 *                         .s3EncryptionEnabled(true)
 *                         .s3KeyPrefix("exec-command-output")
 *                         .build())
 *                 .logging(ExecuteCommandLogging.OVERRIDE)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.ecs;
