/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The summary information for the workgroup, which includes its name, state, description, and the date and time it was
 * created.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkGroupSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<WorkGroupSummary.Builder, WorkGroupSummary> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(WorkGroupSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(WorkGroupSummary::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(WorkGroupSummary::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(WorkGroupSummary::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<EngineVersion> ENGINE_VERSION_FIELD = SdkField
            .<EngineVersion> builder(MarshallingType.SDK_POJO).memberName("EngineVersion")
            .getter(getter(WorkGroupSummary::engineVersion)).setter(setter(Builder::engineVersion))
            .constructor(EngineVersion::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, STATE_FIELD,
            DESCRIPTION_FIELD, CREATION_TIME_FIELD, ENGINE_VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String state;

    private final String description;

    private final Instant creationTime;

    private final EngineVersion engineVersion;

    private WorkGroupSummary(BuilderImpl builder) {
        this.name = builder.name;
        this.state = builder.state;
        this.description = builder.description;
        this.creationTime = builder.creationTime;
        this.engineVersion = builder.engineVersion;
    }

    /**
     * <p>
     * The name of the workgroup.
     * </p>
     * 
     * @return The name of the workgroup.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The state of the workgroup.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link WorkGroupState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the workgroup.
     * @see WorkGroupState
     */
    public final WorkGroupState state() {
        return WorkGroupState.fromValue(state);
    }

    /**
     * <p>
     * The state of the workgroup.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link WorkGroupState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the workgroup.
     * @see WorkGroupState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The workgroup description.
     * </p>
     * 
     * @return The workgroup description.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The workgroup creation date and time.
     * </p>
     * 
     * @return The workgroup creation date and time.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The engine version setting for all queries on the workgroup. Queries on the
     * <code>AmazonAthenaPreviewFunctionality</code> workgroup run on the preview engine regardless of this setting.
     * </p>
     * 
     * @return The engine version setting for all queries on the workgroup. Queries on the
     *         <code>AmazonAthenaPreviewFunctionality</code> workgroup run on the preview engine regardless of this
     *         setting.
     */
    public final EngineVersion engineVersion() {
        return engineVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkGroupSummary)) {
            return false;
        }
        WorkGroupSummary other = (WorkGroupSummary) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(description(), other.description()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(engineVersion(), other.engineVersion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WorkGroupSummary").add("Name", name()).add("State", stateAsString())
                .add("Description", description()).add("CreationTime", creationTime()).add("EngineVersion", engineVersion())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WorkGroupSummary, T> g) {
        return obj -> g.apply((WorkGroupSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkGroupSummary> {
        /**
         * <p>
         * The name of the workgroup.
         * </p>
         * 
         * @param name
         *        The name of the workgroup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The state of the workgroup.
         * </p>
         * 
         * @param state
         *        The state of the workgroup.
         * @see WorkGroupState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkGroupState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the workgroup.
         * </p>
         * 
         * @param state
         *        The state of the workgroup.
         * @see WorkGroupState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkGroupState
         */
        Builder state(WorkGroupState state);

        /**
         * <p>
         * The workgroup description.
         * </p>
         * 
         * @param description
         *        The workgroup description.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The workgroup creation date and time.
         * </p>
         * 
         * @param creationTime
         *        The workgroup creation date and time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The engine version setting for all queries on the workgroup. Queries on the
         * <code>AmazonAthenaPreviewFunctionality</code> workgroup run on the preview engine regardless of this setting.
         * </p>
         * 
         * @param engineVersion
         *        The engine version setting for all queries on the workgroup. Queries on the
         *        <code>AmazonAthenaPreviewFunctionality</code> workgroup run on the preview engine regardless of this
         *        setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(EngineVersion engineVersion);

        /**
         * <p>
         * The engine version setting for all queries on the workgroup. Queries on the
         * <code>AmazonAthenaPreviewFunctionality</code> workgroup run on the preview engine regardless of this setting.
         * </p>
         * This is a convenience that creates an instance of the {@link EngineVersion.Builder} avoiding the need to
         * create one manually via {@link EngineVersion#builder()}.
         *
         * When the {@link Consumer} completes, {@link EngineVersion.Builder#build()} is called immediately and its
         * result is passed to {@link #engineVersion(EngineVersion)}.
         * 
         * @param engineVersion
         *        a consumer that will call methods on {@link EngineVersion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #engineVersion(EngineVersion)
         */
        default Builder engineVersion(Consumer<EngineVersion.Builder> engineVersion) {
            return engineVersion(EngineVersion.builder().applyMutation(engineVersion).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String state;

        private String description;

        private Instant creationTime;

        private EngineVersion engineVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(WorkGroupSummary model) {
            name(model.name);
            state(model.state);
            description(model.description);
            creationTime(model.creationTime);
            engineVersion(model.engineVersion);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        @Transient
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        @Transient
        public final Builder state(WorkGroupState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        @Transient
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        @Transient
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final EngineVersion.Builder getEngineVersion() {
            return engineVersion != null ? engineVersion.toBuilder() : null;
        }

        public final void setEngineVersion(EngineVersion.BuilderImpl engineVersion) {
            this.engineVersion = engineVersion != null ? engineVersion.build() : null;
        }

        @Override
        @Transient
        public final Builder engineVersion(EngineVersion engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        @Override
        public WorkGroupSummary build() {
            return new WorkGroupSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
