/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.cloudfront.CloudFrontClient;
import software.amazon.awssdk.services.cloudfront.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.cloudfront.model.CloudFrontRequest;
import software.amazon.awssdk.services.cloudfront.model.GetDistributionRequest;
import software.amazon.awssdk.services.cloudfront.model.GetDistributionResponse;
import software.amazon.awssdk.services.cloudfront.model.GetInvalidationForDistributionTenantRequest;
import software.amazon.awssdk.services.cloudfront.model.GetInvalidationForDistributionTenantResponse;
import software.amazon.awssdk.services.cloudfront.model.GetInvalidationRequest;
import software.amazon.awssdk.services.cloudfront.model.GetInvalidationResponse;
import software.amazon.awssdk.services.cloudfront.model.GetStreamingDistributionRequest;
import software.amazon.awssdk.services.cloudfront.model.GetStreamingDistributionResponse;
import software.amazon.awssdk.services.cloudfront.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultCloudFrontWaiter implements CloudFrontWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final CloudFrontClient client;

    private final AttributeMap managedResources;

    private final Waiter<GetDistributionResponse> distributionDeployedWaiter;

    private final Waiter<GetInvalidationResponse> invalidationCompletedWaiter;

    private final Waiter<GetInvalidationForDistributionTenantResponse> invalidationForDistributionTenantCompletedWaiter;

    private final Waiter<GetStreamingDistributionResponse> streamingDistributionDeployedWaiter;

    private DefaultCloudFrontWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = CloudFrontClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.distributionDeployedWaiter = Waiter.builder(GetDistributionResponse.class)
                .acceptors(distributionDeployedWaiterAcceptors())
                .overrideConfiguration(distributionDeployedWaiterConfig(builder.overrideConfiguration)).build();
        this.invalidationCompletedWaiter = Waiter.builder(GetInvalidationResponse.class)
                .acceptors(invalidationCompletedWaiterAcceptors())
                .overrideConfiguration(invalidationCompletedWaiterConfig(builder.overrideConfiguration)).build();
        this.invalidationForDistributionTenantCompletedWaiter = Waiter
                .builder(GetInvalidationForDistributionTenantResponse.class)
                .acceptors(invalidationForDistributionTenantCompletedWaiterAcceptors())
                .overrideConfiguration(invalidationForDistributionTenantCompletedWaiterConfig(builder.overrideConfiguration))
                .build();
        this.streamingDistributionDeployedWaiter = Waiter.builder(GetStreamingDistributionResponse.class)
                .acceptors(streamingDistributionDeployedWaiterAcceptors())
                .overrideConfiguration(streamingDistributionDeployedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<GetDistributionResponse> waitUntilDistributionDeployed(GetDistributionRequest getDistributionRequest) {
        return distributionDeployedWaiter.run(() -> client.getDistribution(applyWaitersUserAgent(getDistributionRequest)));
    }

    @Override
    public WaiterResponse<GetDistributionResponse> waitUntilDistributionDeployed(GetDistributionRequest getDistributionRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return distributionDeployedWaiter.run(() -> client.getDistribution(applyWaitersUserAgent(getDistributionRequest)),
                distributionDeployedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetInvalidationResponse> waitUntilInvalidationCompleted(GetInvalidationRequest getInvalidationRequest) {
        return invalidationCompletedWaiter.run(() -> client.getInvalidation(applyWaitersUserAgent(getInvalidationRequest)));
    }

    @Override
    public WaiterResponse<GetInvalidationResponse> waitUntilInvalidationCompleted(GetInvalidationRequest getInvalidationRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return invalidationCompletedWaiter.run(() -> client.getInvalidation(applyWaitersUserAgent(getInvalidationRequest)),
                invalidationCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetInvalidationForDistributionTenantResponse> waitUntilInvalidationForDistributionTenantCompleted(
            GetInvalidationForDistributionTenantRequest getInvalidationForDistributionTenantRequest) {
        return invalidationForDistributionTenantCompletedWaiter.run(() -> client
                .getInvalidationForDistributionTenant(applyWaitersUserAgent(getInvalidationForDistributionTenantRequest)));
    }

    @Override
    public WaiterResponse<GetInvalidationForDistributionTenantResponse> waitUntilInvalidationForDistributionTenantCompleted(
            GetInvalidationForDistributionTenantRequest getInvalidationForDistributionTenantRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return invalidationForDistributionTenantCompletedWaiter.run(() -> client
                .getInvalidationForDistributionTenant(applyWaitersUserAgent(getInvalidationForDistributionTenantRequest)),
                invalidationForDistributionTenantCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetStreamingDistributionResponse> waitUntilStreamingDistributionDeployed(
            GetStreamingDistributionRequest getStreamingDistributionRequest) {
        return streamingDistributionDeployedWaiter.run(() -> client
                .getStreamingDistribution(applyWaitersUserAgent(getStreamingDistributionRequest)));
    }

    @Override
    public WaiterResponse<GetStreamingDistributionResponse> waitUntilStreamingDistributionDeployed(
            GetStreamingDistributionRequest getStreamingDistributionRequest, WaiterOverrideConfiguration overrideConfig) {
        return streamingDistributionDeployedWaiter.run(
                () -> client.getStreamingDistribution(applyWaitersUserAgent(getStreamingDistributionRequest)),
                streamingDistributionDeployedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetDistributionResponse>> distributionDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetDistributionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Distribution").field("Status").value(), "Deployed");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetInvalidationResponse>> invalidationCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetInvalidationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Invalidation").field("Status").value(), "Completed");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetInvalidationForDistributionTenantResponse>> invalidationForDistributionTenantCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetInvalidationForDistributionTenantResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Invalidation").field("Status").value(), "Completed");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStreamingDistributionResponse>> streamingDistributionDeployedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStreamingDistributionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("StreamingDistribution").field("Status").value(), "Deployed");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration distributionDeployedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(35);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration invalidationCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(30);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration invalidationForDistributionTenantCompletedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(30);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration streamingDistributionDeployedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(25);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static CloudFrontWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends CloudFrontRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements CloudFrontWaiter.Builder {
        private CloudFrontClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public CloudFrontWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public CloudFrontWaiter.Builder client(CloudFrontClient client) {
            this.client = client;
            return this;
        }

        public CloudFrontWaiter build() {
            return new DefaultCloudFrontWaiter(this);
        }
    }
}
