/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchevents;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloudwatchevents.model.ActivateEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ActivateEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.CloudWatchEventsException;
import software.amazon.awssdk.services.cloudwatchevents.model.ConcurrentModificationException;
import software.amazon.awssdk.services.cloudwatchevents.model.CreateEventBusRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.CreateEventBusResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.CreatePartnerEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.CreatePartnerEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeactivateEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeactivateEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteEventBusRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteEventBusResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeletePartnerEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeletePartnerEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeEventBusRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeEventBusResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribePartnerEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribePartnerEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DisableRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DisableRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.EnableRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.EnableRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.InternalException;
import software.amazon.awssdk.services.cloudwatchevents.model.InvalidEventPatternException;
import software.amazon.awssdk.services.cloudwatchevents.model.InvalidStateException;
import software.amazon.awssdk.services.cloudwatchevents.model.LimitExceededException;
import software.amazon.awssdk.services.cloudwatchevents.model.ListEventBusesRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListEventBusesResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListEventSourcesRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListEventSourcesResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListPartnerEventSourceAccountsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListPartnerEventSourceAccountsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListPartnerEventSourcesRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListPartnerEventSourcesResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRuleNamesByTargetRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRuleNamesByTargetResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRulesRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRulesResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListTargetsByRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListTargetsByRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ManagedRuleException;
import software.amazon.awssdk.services.cloudwatchevents.model.PolicyLengthExceededException;
import software.amazon.awssdk.services.cloudwatchevents.model.PutEventsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutEventsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PutPartnerEventsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutPartnerEventsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PutPermissionRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutPermissionResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PutRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PutTargetsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutTargetsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.RemovePermissionRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.RemovePermissionResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.RemoveTargetsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.RemoveTargetsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.cloudwatchevents.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudwatchevents.model.TagResourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.TagResourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.TestEventPatternRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.TestEventPatternResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.transform.ActivateEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.CreateEventBusRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.CreatePartnerEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeactivateEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeleteEventBusRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeletePartnerEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeleteRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeEventBusRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribePartnerEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DisableRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.EnableRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListEventBusesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListEventSourcesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListPartnerEventSourceAccountsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListPartnerEventSourcesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListRuleNamesByTargetRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListTargetsByRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutEventsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutPartnerEventsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutPermissionRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutTargetsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.RemovePermissionRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.RemoveTargetsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.TestEventPatternRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CloudWatchEventsAsyncClient}.
 *
 * @see CloudWatchEventsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudWatchEventsAsyncClient implements CloudWatchEventsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCloudWatchEventsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudWatchEventsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Activates a partner event source that has been deactivated. Once activated, your matching event bus will start
     * receiving events from the event source.
     * </p>
     * <note>
     * <p>
     * This operation is performed by AWS customers, not by SaaS partners.
     * </p>
     * </note>
     *
     * @param activateEventSourceRequest
     * @return A Java Future containing the result of the ActivateEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InvalidStateException The specified state isn't a valid state for an event source.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ActivateEventSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ActivateEventSource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ActivateEventSourceResponse> activateEventSource(
            ActivateEventSourceRequest activateEventSourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ActivateEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ActivateEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ActivateEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ActivateEventSourceRequest, ActivateEventSourceResponse>()
                            .withOperationName("ActivateEventSource")
                            .withMarshaller(new ActivateEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(activateEventSourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new event bus within your account. This can be a custom event bus which you can use to receive events
     * from your own custom applications and services, or it can be a partner event bus which can be matched to a
     * partner event source.
     * </p>
     * <note>
     * <p>
     * This operation is used by AWS customers, not by SaaS partners.
     * </p>
     * </note>
     *
     * @param createEventBusRequest
     * @return A Java Future containing the result of the CreateEventBus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceAlreadyExistsException The resource that you're trying to create already exists.</li>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InvalidStateException The specified state isn't a valid state for an event source.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>LimitExceededException You tried to create more resources than is allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.CreateEventBus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/CreateEventBus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEventBusResponse> createEventBus(CreateEventBusRequest createEventBusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEventBusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEventBusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEventBusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEventBusRequest, CreateEventBusResponse>()
                            .withOperationName("CreateEventBus")
                            .withMarshaller(new CreateEventBusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createEventBusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Called by an SaaS partner to create a partner event source.
     * </p>
     * <note>
     * <p>
     * This operation is not used by AWS customers.
     * </p>
     * </note>
     * <p>
     * Each partner event source can be used by one AWS account to create a matching partner event bus in that AWS
     * account. A SaaS partner must create one partner event source for each AWS account that wants to receive those
     * event types.
     * </p>
     * <p>
     * A partner event source creates events based on resources in the SaaS partner's service or application.
     * </p>
     * <p>
     * An AWS account that creates a partner event bus that matches the partner event source can use that event bus to
     * receive events from the partner, and then process them using AWS Events rules and targets.
     * </p>
     * <p>
     * Partner event source names follow this format:
     * </p>
     * <p>
     * <code>aws.partner/<i>partner_name</i>/<i>event_namespace</i>/<i>event_name</i> </code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>partner_name</i> is determined during partner registration and identifies the partner to AWS customers.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <i>event_namespace</i>, we recommend that partners use a string that identifies the AWS customer within the
     * partner's system. This should not be the customer's AWS account ID.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>event_name</i> is determined by the partner, and should uniquely identify an event-generating resource within
     * the partner system. This should help AWS customers decide whether to create an event bus to receive these events.
     * </p>
     * </li>
     * </ul>
     *
     * @param createPartnerEventSourceRequest
     * @return A Java Future containing the result of the CreatePartnerEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceAlreadyExistsException The resource that you're trying to create already exists.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>LimitExceededException You tried to create more resources than is allowed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.CreatePartnerEventSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/CreatePartnerEventSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePartnerEventSourceResponse> createPartnerEventSource(
            CreatePartnerEventSourceRequest createPartnerEventSourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePartnerEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePartnerEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePartnerEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePartnerEventSourceRequest, CreatePartnerEventSourceResponse>()
                            .withOperationName("CreatePartnerEventSource")
                            .withMarshaller(new CreatePartnerEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createPartnerEventSourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * An AWS customer uses this operation to temporarily stop receiving events from the specified partner event source.
     * The matching event bus isn't deleted.
     * </p>
     * <p>
     * When you deactivate a partner event source, the source goes into <code>PENDING</code> state. If it remains in
     * <code>PENDING</code> state for more than two weeks, it's deleted.
     * </p>
     * <p>
     * To activate a deactivated partner event source, use <a>ActivateEventSource</a>.
     * </p>
     *
     * @param deactivateEventSourceRequest
     * @return A Java Future containing the result of the DeactivateEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InvalidStateException The specified state isn't a valid state for an event source.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeactivateEventSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeactivateEventSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeactivateEventSourceResponse> deactivateEventSource(
            DeactivateEventSourceRequest deactivateEventSourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeactivateEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeactivateEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeactivateEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeactivateEventSourceRequest, DeactivateEventSourceResponse>()
                            .withOperationName("DeactivateEventSource")
                            .withMarshaller(new DeactivateEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deactivateEventSourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified custom event bus or partner event bus. All rules associated with this event bus are also
     * deleted. You can't delete your account's default event bus.
     * </p>
     * <note>
     * <p>
     * This operation is performed by AWS customers, not by SaaS partners.
     * </p>
     * </note>
     *
     * @param deleteEventBusRequest
     * @return A Java Future containing the result of the DeleteEventBus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeleteEventBus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeleteEventBus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEventBusResponse> deleteEventBus(DeleteEventBusRequest deleteEventBusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEventBusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEventBusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEventBusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEventBusRequest, DeleteEventBusResponse>()
                            .withOperationName("DeleteEventBus")
                            .withMarshaller(new DeleteEventBusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteEventBusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation is used by SaaS partners to delete a partner event source. AWS customers don't use this operation.
     * </p>
     * <p>
     * When you delete an event source, the status of the corresponding partner event bus in the AWS customer account
     * becomes <code>DELETED</code>.
     * </p>
     *
     * @param deletePartnerEventSourceRequest
     * @return A Java Future containing the result of the DeletePartnerEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeletePartnerEventSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeletePartnerEventSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePartnerEventSourceResponse> deletePartnerEventSource(
            DeletePartnerEventSourceRequest deletePartnerEventSourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePartnerEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePartnerEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePartnerEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePartnerEventSourceRequest, DeletePartnerEventSourceResponse>()
                            .withOperationName("DeletePartnerEventSource")
                            .withMarshaller(new DeletePartnerEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deletePartnerEventSourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified rule.
     * </p>
     * <p>
     * Before you can delete the rule, you must remove all targets, using <a>RemoveTargets</a>.
     * </p>
     * <p>
     * When you delete a rule, incoming events might continue to match to the deleted rule. Allow a short period of time
     * for changes to take effect.
     * </p>
     * <p>
     * Managed rules are rules created and managed by another AWS service on your behalf. These rules are created by
     * those other AWS services to support functionality in those services. You can delete these rules using the
     * <code>Force</code> option, but you should do so only if you're sure that the other service isn't still using that
     * rule.
     * </p>
     *
     * @param deleteRuleRequest
     * @return A Java Future containing the result of the DeleteRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>ManagedRuleException An AWS service created this rule on behalf of your account. That service manages
     *         it. If you see this error in response to <code>DeleteRule</code> or <code>RemoveTargets</code>, you can
     *         use the <code>Force</code> parameter in those calls to delete the rule or remove targets from the rule.
     *         You can't modify these managed rules by using <code>DisableRule</code>, <code>EnableRule</code>,
     *         <code>PutTargets</code>, <code>PutRule</code>, <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeleteRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeleteRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRuleResponse> deleteRule(DeleteRuleRequest deleteRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRuleRequest, DeleteRuleResponse>().withOperationName("DeleteRule")
                            .withMarshaller(new DeleteRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays details about an event bus in your account. This can include the external AWS accounts that are
     * permitted to write events to your default event bus, and the associated policy. For custom event buses and
     * partner event buses, it displays the name, ARN, policy, state, and creation time.
     * </p>
     * <p>
     * To enable your account to receive events from other accounts on its default event bus, use <a>PutPermission</a>.
     * </p>
     * <p>
     * For more information about partner event buses, see <a>CreateEventBus</a>.
     * </p>
     *
     * @param describeEventBusRequest
     * @return A Java Future containing the result of the DescribeEventBus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribeEventBus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeEventBus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventBusResponse> describeEventBus(DescribeEventBusRequest describeEventBusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventBusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventBusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventBusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventBusRequest, DescribeEventBusResponse>()
                            .withOperationName("DescribeEventBus")
                            .withMarshaller(new DescribeEventBusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEventBusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation lists details about a partner event source that is shared with your account.
     * </p>
     * <note>
     * <p>
     * This operation is run by AWS customers, not by SaaS partners.
     * </p>
     * </note>
     *
     * @param describeEventSourceRequest
     * @return A Java Future containing the result of the DescribeEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribeEventSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeEventSource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventSourceResponse> describeEventSource(
            DescribeEventSourceRequest describeEventSourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventSourceRequest, DescribeEventSourceResponse>()
                            .withOperationName("DescribeEventSource")
                            .withMarshaller(new DescribeEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEventSourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * An SaaS partner can use this operation to list details about a partner event source that they have created.
     * </p>
     * <note>
     * <p>
     * AWS customers do not use this operation. Instead, AWS customers can use <a>DescribeEventSource</a> to see details
     * about a partner event source that is shared with them.
     * </p>
     * </note>
     *
     * @param describePartnerEventSourceRequest
     * @return A Java Future containing the result of the DescribePartnerEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribePartnerEventSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribePartnerEventSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePartnerEventSourceResponse> describePartnerEventSource(
            DescribePartnerEventSourceRequest describePartnerEventSourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePartnerEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePartnerEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePartnerEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePartnerEventSourceRequest, DescribePartnerEventSourceResponse>()
                            .withOperationName("DescribePartnerEventSource")
                            .withMarshaller(new DescribePartnerEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describePartnerEventSourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified rule.
     * </p>
     * <p>
     * <code>DescribeRule</code> doesn't list the targets of a rule. To see the targets associated with a rule, use
     * <a>ListTargetsByRule</a>.
     * </p>
     *
     * @param describeRuleRequest
     * @return A Java Future containing the result of the DescribeRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribeRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRuleResponse> describeRule(DescribeRuleRequest describeRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRuleRequest, DescribeRuleResponse>()
                            .withOperationName("DescribeRule").withMarshaller(new DescribeRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables the specified rule. A disabled rule won't match any events and won't self-trigger if it has a schedule
     * expression.
     * </p>
     * <p>
     * When you disable a rule, incoming events might continue to match to the disabled rule. Allow a short period of
     * time for changes to take effect.
     * </p>
     *
     * @param disableRuleRequest
     * @return A Java Future containing the result of the DisableRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>ManagedRuleException An AWS service created this rule on behalf of your account. That service manages
     *         it. If you see this error in response to <code>DeleteRule</code> or <code>RemoveTargets</code>, you can
     *         use the <code>Force</code> parameter in those calls to delete the rule or remove targets from the rule.
     *         You can't modify these managed rules by using <code>DisableRule</code>, <code>EnableRule</code>,
     *         <code>PutTargets</code>, <code>PutRule</code>, <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DisableRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DisableRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisableRuleResponse> disableRule(DisableRuleRequest disableRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DisableRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableRuleRequest, DisableRuleResponse>()
                            .withOperationName("DisableRule").withMarshaller(new DisableRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disableRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables the specified rule. If the rule doesn't exist, the operation fails.
     * </p>
     * <p>
     * When you enable a rule, incoming events might not immediately start matching to a newly enabled rule. Allow a
     * short period of time for changes to take effect.
     * </p>
     *
     * @param enableRuleRequest
     * @return A Java Future containing the result of the EnableRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>ManagedRuleException An AWS service created this rule on behalf of your account. That service manages
     *         it. If you see this error in response to <code>DeleteRule</code> or <code>RemoveTargets</code>, you can
     *         use the <code>Force</code> parameter in those calls to delete the rule or remove targets from the rule.
     *         You can't modify these managed rules by using <code>DisableRule</code>, <code>EnableRule</code>,
     *         <code>PutTargets</code>, <code>PutRule</code>, <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.EnableRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/EnableRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EnableRuleResponse> enableRule(EnableRuleRequest enableRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    EnableRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableRuleRequest, EnableRuleResponse>().withOperationName("EnableRule")
                            .withMarshaller(new EnableRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(enableRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the event buses in your account, including the default event bus, custom event buses, and partner event
     * buses.
     * </p>
     * <note>
     * <p>
     * This operation is run by AWS customers, not by SaaS partners.
     * </p>
     * </note>
     *
     * @param listEventBusesRequest
     * @return A Java Future containing the result of the ListEventBuses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListEventBuses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListEventBuses" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventBusesResponse> listEventBuses(ListEventBusesRequest listEventBusesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventBusesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEventBusesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEventBusesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventBusesRequest, ListEventBusesResponse>()
                            .withOperationName("ListEventBuses")
                            .withMarshaller(new ListEventBusesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listEventBusesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You can use this to see all the partner event sources that have been shared with your AWS account. For more
     * information about partner event sources, see <a>CreateEventBus</a>.
     * </p>
     * <note>
     * <p>
     * This operation is run by AWS customers, not by SaaS partners.
     * </p>
     * </note>
     *
     * @param listEventSourcesRequest
     * @return A Java Future containing the result of the ListEventSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListEventSources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListEventSources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventSourcesResponse> listEventSources(ListEventSourcesRequest listEventSourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEventSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEventSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventSourcesRequest, ListEventSourcesResponse>()
                            .withOperationName("ListEventSources")
                            .withMarshaller(new ListEventSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listEventSourcesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * An SaaS partner can use this operation to display the AWS account ID that a particular partner event source name
     * is associated with.
     * </p>
     * <note>
     * <p>
     * This operation is used by SaaS partners, not by AWS customers.
     * </p>
     * </note>
     *
     * @param listPartnerEventSourceAccountsRequest
     * @return A Java Future containing the result of the ListPartnerEventSourceAccounts operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListPartnerEventSourceAccounts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListPartnerEventSourceAccounts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPartnerEventSourceAccountsResponse> listPartnerEventSourceAccounts(
            ListPartnerEventSourceAccountsRequest listPartnerEventSourceAccountsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPartnerEventSourceAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPartnerEventSourceAccountsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPartnerEventSourceAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPartnerEventSourceAccountsRequest, ListPartnerEventSourceAccountsResponse>()
                            .withOperationName("ListPartnerEventSourceAccounts")
                            .withMarshaller(new ListPartnerEventSourceAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listPartnerEventSourceAccountsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * An SaaS partner can use this operation to list all the partner event source names that they have created.
     * </p>
     * <note>
     * <p>
     * This operation is not used by AWS customers.
     * </p>
     * </note>
     *
     * @param listPartnerEventSourcesRequest
     * @return A Java Future containing the result of the ListPartnerEventSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListPartnerEventSources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListPartnerEventSources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPartnerEventSourcesResponse> listPartnerEventSources(
            ListPartnerEventSourcesRequest listPartnerEventSourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPartnerEventSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPartnerEventSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPartnerEventSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPartnerEventSourcesRequest, ListPartnerEventSourcesResponse>()
                            .withOperationName("ListPartnerEventSources")
                            .withMarshaller(new ListPartnerEventSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listPartnerEventSourcesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the rules for the specified target. You can see which rules can invoke a specific target in your account.
     * </p>
     *
     * @param listRuleNamesByTargetRequest
     * @return A Java Future containing the result of the ListRuleNamesByTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListRuleNamesByTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListRuleNamesByTarget" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRuleNamesByTargetResponse> listRuleNamesByTarget(
            ListRuleNamesByTargetRequest listRuleNamesByTargetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRuleNamesByTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRuleNamesByTargetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRuleNamesByTargetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRuleNamesByTargetRequest, ListRuleNamesByTargetResponse>()
                            .withOperationName("ListRuleNamesByTarget")
                            .withMarshaller(new ListRuleNamesByTargetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listRuleNamesByTargetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your EventBridge rules. You can either list all the rules or provide a prefix to match to the rule names.
     * </p>
     * <p>
     * <code>ListRules</code> doesn't list the targets of a rule. To see the targets associated with a rule, use
     * <a>ListTargetsByRule</a>.
     * </p>
     *
     * @param listRulesRequest
     * @return A Java Future containing the result of the ListRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListRulesResponse> listRules(ListRulesRequest listRulesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRulesRequest, ListRulesResponse>().withOperationName("ListRules")
                            .withMarshaller(new ListRulesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listRulesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays the tags associated with an EventBridge resource. In EventBridge, rules can be tagged.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the targets assigned to the specified rule.
     * </p>
     *
     * @param listTargetsByRuleRequest
     * @return A Java Future containing the result of the ListTargetsByRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListTargetsByRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListTargetsByRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTargetsByRuleResponse> listTargetsByRule(ListTargetsByRuleRequest listTargetsByRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTargetsByRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTargetsByRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTargetsByRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTargetsByRuleRequest, ListTargetsByRuleResponse>()
                            .withOperationName("ListTargetsByRule")
                            .withMarshaller(new ListTargetsByRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTargetsByRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends custom events to EventBridge so that they can be matched to rules. These events can be from your custom
     * applications and services.
     * </p>
     *
     * @param putEventsRequest
     * @return A Java Future containing the result of the PutEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.PutEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutEventsResponse> putEvents(PutEventsRequest putEventsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEventsRequest, PutEventsResponse>().withOperationName("PutEvents")
                            .withMarshaller(new PutEventsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putEventsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This is used by SaaS partners to write events to a customer's partner event bus.
     * </p>
     * <note>
     * <p>
     * AWS customers do not use this operation. Instead, AWS customers can use <a>PutEvents</a> to write custom events
     * from their own applications to an event bus.
     * </p>
     * </note>
     *
     * @param putPartnerEventsRequest
     * @return A Java Future containing the result of the PutPartnerEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.PutPartnerEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutPartnerEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutPartnerEventsResponse> putPartnerEvents(PutPartnerEventsRequest putPartnerEventsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPartnerEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutPartnerEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPartnerEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPartnerEventsRequest, PutPartnerEventsResponse>()
                            .withOperationName("PutPartnerEvents")
                            .withMarshaller(new PutPartnerEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putPartnerEventsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Running <code>PutPermission</code> permits the specified AWS account or AWS organization to put events to the
     * specified <i>event bus</i>. Rules in your account are triggered by these events arriving to an event bus in your
     * account.
     * </p>
     * <p>
     * For another account to send events to your account, that external account must have a rule with your account's
     * event bus as a target.
     * </p>
     * <p>
     * To enable multiple AWS accounts to put events to an event bus, run <code>PutPermission</code> once for each of
     * these accounts. Or, if all the accounts are members of the same AWS organization, you can run
     * <code>PutPermission</code> once specifying <code>Principal</code> as "*" and specifying the AWS organization ID
     * in <code>Condition</code>, to grant permissions to all accounts in that organization.
     * </p>
     * <p>
     * If you grant permissions using an organization, then accounts in that organization must specify a
     * <code>RoleArn</code> with proper permissions when they use <code>PutTarget</code> to add your account's event bus
     * as a target. For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html"
     * >Sending and Receiving Events Between AWS Accounts</a> in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * <p>
     * The permission policy on an event bus can't exceed 10 KB in size.
     * </p>
     *
     * @param putPermissionRequest
     * @return A Java Future containing the result of the PutPermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>PolicyLengthExceededException The event bus policy is too long. For more information, see the limits.
     *         </li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.PutPermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutPermissionResponse> putPermission(PutPermissionRequest putPermissionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutPermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPermissionRequest, PutPermissionResponse>()
                            .withOperationName("PutPermission")
                            .withMarshaller(new PutPermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putPermissionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates the specified rule. Rules are enabled by default or based on value of the state. You can
     * disable a rule using <a>DisableRule</a>.
     * </p>
     * <p>
     * A single rule watches for events from a single event bus. Events generated by AWS services go to your account's
     * default event bus. Events generated by SaaS partner services or applications go to the matching partner event
     * bus. If you have custom applications or services, you can specify whether their events go to your default event
     * bus or a custom event bus that you have created. For more information, see <a>CreateEventBus</a>.
     * </p>
     * <p>
     * If you're updating an existing rule, the rule is replaced with what you specify in this <code>PutRule</code>
     * command. If you omit arguments in <code>PutRule</code>, the old values for those arguments aren't kept. Instead,
     * they're replaced with null values.
     * </p>
     * <p>
     * When you create or update a rule, incoming events might not immediately start matching to new or updated rules.
     * Allow a short period of time for changes to take effect.
     * </p>
     * <p>
     * A rule must contain at least an <code>EventPattern</code> or <code>ScheduleExpression</code>. Rules with
     * <code>EventPatterns</code> are triggered when a matching event is observed. Rules with
     * <code>ScheduleExpressions</code> self-trigger based on the given schedule. A rule can have both an
     * <code>EventPattern</code> and a <code>ScheduleExpression</code>, in which case the rule triggers on matching
     * events as well as on a schedule.
     * </p>
     * <p>
     * When you initially create a rule, you can optionally assign one or more tags to the rule. Tags can help you
     * organize and categorize your resources. You can also use them to scope user permissions, by granting a user
     * permission to access or change only rules with certain tag values. To use the <code>PutRule</code> operation and
     * assign tags, you must have both the <code>events:PutRule</code> and <code>events:TagResource</code> permissions.
     * </p>
     * <p>
     * If you are updating an existing rule, any tags you specify in the <code>PutRule</code> operation are ignored. To
     * update the tags of an existing rule, use <a>TagResource</a> and <a>UntagResource</a>.
     * </p>
     * <p>
     * Most services in AWS treat <code>:</code> or <code>/</code> as the same character in Amazon Resource Names
     * (ARNs). However, EventBridge uses an exact match in event patterns and rules. Be sure to use the correct ARN
     * characters when creating event patterns so that they match the ARN syntax in the event that you want to match.
     * </p>
     * <p>
     * In EventBridge, you could create rules that lead to infinite loops, where a rule is fired repeatedly. For
     * example, a rule might detect that ACLs have changed on an S3 bucket, and trigger software to change them to the
     * desired state. If you don't write the rule carefully, the subsequent change to the ACLs fires the rule again,
     * creating an infinite loop.
     * </p>
     * <p>
     * To prevent this, write the rules so that the triggered actions don't refire the same rule. For example, your rule
     * could fire only if ACLs are found to be in a bad state, instead of after any change.
     * </p>
     * <p>
     * An infinite loop can quickly cause higher than expected charges. We recommend that you use budgeting, which
     * alerts you when charges exceed your specified limit. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/budgets-managing-costs.html">Managing Your
     * Costs with Budgets</a>.
     * </p>
     *
     * @param putRuleRequest
     * @return A Java Future containing the result of the PutRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidEventPatternException The event pattern isn't valid.</li>
     *         <li>LimitExceededException You tried to create more resources than is allowed.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>ManagedRuleException An AWS service created this rule on behalf of your account. That service manages
     *         it. If you see this error in response to <code>DeleteRule</code> or <code>RemoveTargets</code>, you can
     *         use the <code>Force</code> parameter in those calls to delete the rule or remove targets from the rule.
     *         You can't modify these managed rules by using <code>DisableRule</code>, <code>EnableRule</code>,
     *         <code>PutTargets</code>, <code>PutRule</code>, <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.PutRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutRuleResponse> putRule(PutRuleRequest putRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRuleRequest, PutRuleResponse>().withOperationName("PutRule")
                            .withMarshaller(new PutRuleRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the specified targets to the specified rule, or updates the targets if they're already associated with the
     * rule.
     * </p>
     * <p>
     * Targets are the resources that are invoked when a rule is triggered.
     * </p>
     * <p>
     * You can configure the following as targets in EventBridge:
     * </p>
     * <ul>
     * <li>
     * <p>
     * EC2 instances
     * </p>
     * </li>
     * <li>
     * <p>
     * SSM Run Command
     * </p>
     * </li>
     * <li>
     * <p>
     * SSM Automation
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS Lambda functions
     * </p>
     * </li>
     * <li>
     * <p>
     * Data streams in Amazon Kinesis Data Streams
     * </p>
     * </li>
     * <li>
     * <p>
     * Data delivery streams in Amazon Kinesis Data Firehose
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon ECS tasks
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS Step Functions state machines
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS Batch jobs
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS CodeBuild projects
     * </p>
     * </li>
     * <li>
     * <p>
     * Pipelines in AWS CodePipeline
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Inspector assessment templates
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon SNS topics
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon SQS queues, including FIFO queues
     * </p>
     * </li>
     * <li>
     * <p>
     * The default event bus of another AWS account
     * </p>
     * </li>
     * </ul>
     * <p>
     * Creating rules with built-in targets is supported only on the AWS Management Console. The built-in targets are
     * <code>EC2 CreateSnapshot API call</code>, <code>EC2 RebootInstances API call</code>,
     * <code>EC2 StopInstances API call</code>, and <code>EC2 TerminateInstances API call</code>.
     * </p>
     * <p>
     * For some target types, <code>PutTargets</code> provides target-specific parameters. If the target is a Kinesis
     * data stream, you can optionally specify which shard the event goes to by using the <code>KinesisParameters</code>
     * argument. To invoke a command on multiple EC2 instances with one rule, you can use the
     * <code>RunCommandParameters</code> field.
     * </p>
     * <p>
     * To be able to make API calls against the resources that you own, Amazon EventBridge needs the appropriate
     * permissions. For AWS Lambda and Amazon SNS resources, EventBridge relies on resource-based policies. For EC2
     * instances, Kinesis data streams, and AWS Step Functions state machines, EventBridge relies on IAM roles that you
     * specify in the <code>RoleARN</code> argument in <code>PutTargets</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/auth-and-access-control-eventbridge.html"
     * >Authentication and Access Control</a> in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * <p>
     * If another AWS account is in the same Region and has granted you permission (using <code>PutPermission</code>),
     * you can send events to that account. Set that account's event bus as a target of the rules in your account. To
     * send the matched events to the other account, specify that account's event bus as the <code>Arn</code> value when
     * you run <code>PutTargets</code>. If your account sends events to another account, your account is charged for
     * each sent event. Each event sent to another account is charged as a custom event. The account receiving the event
     * isn't charged. For more information, see <a href="https://aws.amazon.com/eventbridge/pricing/">Amazon EventBridge
     * Pricing</a>.
     * </p>
     * <p>
     * If you're setting an event bus in another account as the target and that account granted permission to your
     * account through an organization instead of directly by the account ID, you must specify a <code>RoleArn</code>
     * with proper permissions in the <code>Target</code> structure. For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html"
     * >Sending and Receiving Events Between AWS Accounts</a> in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * <p>
     * For more information about enabling cross-account events, see <a>PutPermission</a>.
     * </p>
     * <p>
     * <code>Input</code>, <code>InputPath</code>, and <code>InputTransformer</code> are mutually exclusive and optional
     * parameters of a target. When a rule is triggered due to a matched event:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If none of the following arguments are specified for a target, the entire event is passed to the target in JSON
     * format (unless the target is Amazon EC2 Run Command or Amazon ECS task, in which case nothing from the event is
     * passed to the target).
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>Input</code> is specified in the form of valid JSON, then the matched event is overridden with this
     * constant.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>InputPath</code> is specified in the form of JSONPath (for example, <code>&#36.detail</code>), only the
     * part of the event specified in the path is passed to the target (for example, only the detail part of the event
     * is passed).
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>InputTransformer</code> is specified, one or more specified JSONPaths are extracted from the event and
     * used as values in a template that you specify as the input to the target.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you specify <code>InputPath</code> or <code>InputTransformer</code>, you must use JSON dot notation, not
     * bracket notation.
     * </p>
     * <p>
     * When you add targets to a rule and the associated rule triggers soon after, new or updated targets might not be
     * immediately invoked. Allow a short period of time for changes to take effect.
     * </p>
     * <p>
     * This action can partially fail if too many requests are made at the same time. If that happens,
     * <code>FailedEntryCount</code> is nonzero in the response, and each entry in <code>FailedEntries</code> provides
     * the ID of the failed target and the error code.
     * </p>
     *
     * @param putTargetsRequest
     * @return A Java Future containing the result of the PutTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>LimitExceededException You tried to create more resources than is allowed.</li>
     *         <li>ManagedRuleException An AWS service created this rule on behalf of your account. That service manages
     *         it. If you see this error in response to <code>DeleteRule</code> or <code>RemoveTargets</code>, you can
     *         use the <code>Force</code> parameter in those calls to delete the rule or remove targets from the rule.
     *         You can't modify these managed rules by using <code>DisableRule</code>, <code>EnableRule</code>,
     *         <code>PutTargets</code>, <code>PutRule</code>, <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.PutTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutTargets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutTargetsResponse> putTargets(PutTargetsRequest putTargetsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTargetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutTargetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutTargetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTargetsRequest, PutTargetsResponse>().withOperationName("PutTargets")
                            .withMarshaller(new PutTargetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putTargetsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Revokes the permission of another AWS account to be able to put events to the specified event bus. Specify the
     * account to revoke by the <code>StatementId</code> value that you associated with the account when you granted it
     * permission with <code>PutPermission</code>. You can find the <code>StatementId</code> by using
     * <a>DescribeEventBus</a>.
     * </p>
     *
     * @param removePermissionRequest
     * @return A Java Future containing the result of the RemovePermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.RemovePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/RemovePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemovePermissionResponse> removePermission(RemovePermissionRequest removePermissionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemovePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemovePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemovePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemovePermissionRequest, RemovePermissionResponse>()
                            .withOperationName("RemovePermission")
                            .withMarshaller(new RemovePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(removePermissionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified targets from the specified rule. When the rule is triggered, those targets are no longer be
     * invoked.
     * </p>
     * <p>
     * When you remove a target, when the associated rule triggers, removed targets might continue to be invoked. Allow
     * a short period of time for changes to take effect.
     * </p>
     * <p>
     * This action can partially fail if too many requests are made at the same time. If that happens,
     * <code>FailedEntryCount</code> is non-zero in the response and each entry in <code>FailedEntries</code> provides
     * the ID of the failed target and the error code.
     * </p>
     *
     * @param removeTargetsRequest
     * @return A Java Future containing the result of the RemoveTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>ManagedRuleException An AWS service created this rule on behalf of your account. That service manages
     *         it. If you see this error in response to <code>DeleteRule</code> or <code>RemoveTargets</code>, you can
     *         use the <code>Force</code> parameter in those calls to delete the rule or remove targets from the rule.
     *         You can't modify these managed rules by using <code>DisableRule</code>, <code>EnableRule</code>,
     *         <code>PutTargets</code>, <code>PutRule</code>, <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.RemoveTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/RemoveTargets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveTargetsResponse> removeTargets(RemoveTargetsRequest removeTargetsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveTargetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RemoveTargetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveTargetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveTargetsRequest, RemoveTargetsResponse>()
                            .withOperationName("RemoveTargets")
                            .withMarshaller(new RemoveTargetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(removeTargetsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified EventBridge resource. Tags can help you organize and
     * categorize your resources. You can also use them to scope user permissions by granting a user permission to
     * access or change only resources with certain tag values. In EventBridge, rules can be tagged.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to AWS and are interpreted strictly as strings of characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a rule that already has tags. If you specify a new tag key
     * for the rule, this tag is appended to the list of tags associated with the rule. If you specify a tag key that is
     * already associated with the rule, the new tag value that you specify replaces the previous value for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ManagedRuleException An AWS service created this rule on behalf of your account. That service manages
     *         it. If you see this error in response to <code>DeleteRule</code> or <code>RemoveTargets</code>, you can
     *         use the <code>Force</code> parameter in those calls to delete the rule or remove targets from the rule.
     *         You can't modify these managed rules by using <code>DisableRule</code>, <code>EnableRule</code>,
     *         <code>PutTargets</code>, <code>PutRule</code>, <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tests whether the specified event pattern matches the provided event.
     * </p>
     * <p>
     * Most services in AWS treat <code>:</code> or <code>/</code> as the same character in Amazon Resource Names
     * (ARNs). However, EventBridge uses an exact match in event patterns and rules. Be sure to use the correct ARN
     * characters when creating event patterns so that they match the ARN syntax in the event that you want to match.
     * </p>
     *
     * @param testEventPatternRequest
     * @return A Java Future containing the result of the TestEventPattern operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidEventPatternException The event pattern isn't valid.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.TestEventPattern
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/TestEventPattern" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TestEventPatternResponse> testEventPattern(TestEventPatternRequest testEventPatternRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TestEventPatternResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TestEventPatternResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TestEventPatternResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TestEventPatternRequest, TestEventPatternResponse>()
                            .withOperationName("TestEventPattern")
                            .withMarshaller(new TestEventPatternRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(testEventPatternRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified EventBridge resource. In EventBridge, rules can be tagged.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified doesn't exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a resource.</li>
     *         <li>ManagedRuleException An AWS service created this rule on behalf of your account. That service manages
     *         it. If you see this error in response to <code>DeleteRule</code> or <code>RemoveTargets</code>, you can
     *         use the <code>Force</code> parameter in those calls to delete the rule or remove targets from the rule.
     *         You can't modify these managed rules by using <code>DisableRule</code>, <code>EnableRule</code>,
     *         <code>PutTargets</code>, <code>PutRule</code>, <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudWatchEventsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyLengthExceededException")
                                .exceptionBuilderSupplier(PolicyLengthExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateException")
                                .exceptionBuilderSupplier(InvalidStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEventPatternException")
                                .exceptionBuilderSupplier(InvalidEventPatternException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ManagedRuleException")
                                .exceptionBuilderSupplier(ManagedRuleException::builder).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
