/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeguruprofiler;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codeguruprofiler.model.CodeGuruProfilerException;
import software.amazon.awssdk.services.codeguruprofiler.model.CodeGuruProfilerRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.ConfigureAgentRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.ConfigureAgentResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.ConflictException;
import software.amazon.awssdk.services.codeguruprofiler.model.CreateProfilingGroupRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.CreateProfilingGroupResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.DeleteProfilingGroupRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.DeleteProfilingGroupResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.DescribeProfilingGroupRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.DescribeProfilingGroupResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.GetProfileRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.GetProfileResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.InternalServerException;
import software.amazon.awssdk.services.codeguruprofiler.model.ListProfileTimesRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.ListProfileTimesResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.ListProfilingGroupsRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.ListProfilingGroupsResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.PostAgentProfileRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.PostAgentProfileResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codeguruprofiler.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.codeguruprofiler.model.ThrottlingException;
import software.amazon.awssdk.services.codeguruprofiler.model.UpdateProfilingGroupRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.UpdateProfilingGroupResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.ValidationException;
import software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfileTimesPublisher;
import software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfilingGroupsPublisher;
import software.amazon.awssdk.services.codeguruprofiler.transform.ConfigureAgentRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.CreateProfilingGroupRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.DeleteProfilingGroupRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.DescribeProfilingGroupRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.GetProfileRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.ListProfileTimesRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.ListProfilingGroupsRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.PostAgentProfileRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.UpdateProfilingGroupRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CodeGuruProfilerAsyncClient}.
 *
 * @see CodeGuruProfilerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodeGuruProfilerAsyncClient implements CodeGuruProfilerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCodeGuruProfilerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodeGuruProfilerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Provides the configuration to use for an agent of the profiling group.
     * </p>
     *
     * @param configureAgentRequest
     *        Request for ConfigureAgent operation.
     * @return A Java Future containing the result of the ConfigureAgent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the API.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeGuruProfilerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeGuruProfilerAsyncClient.ConfigureAgent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/ConfigureAgent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ConfigureAgentResponse> configureAgent(ConfigureAgentRequest configureAgentRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ConfigureAgentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ConfigureAgentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ConfigureAgentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ConfigureAgentRequest, ConfigureAgentResponse>()
                            .withOperationName("ConfigureAgent")
                            .withMarshaller(new ConfigureAgentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(configureAgentRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a profiling group.
     * </p>
     *
     * @param createProfilingGroupRequest
     *        Request for CreateProfilingGroup operation.
     * @return A Java Future containing the result of the CreateProfilingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Request can can cause an inconsistent state for the resource.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the API.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeGuruProfilerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeGuruProfilerAsyncClient.CreateProfilingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/CreateProfilingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProfilingGroupResponse> createProfilingGroup(
            CreateProfilingGroupRequest createProfilingGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProfilingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateProfilingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProfilingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProfilingGroupRequest, CreateProfilingGroupResponse>()
                            .withOperationName("CreateProfilingGroup")
                            .withMarshaller(new CreateProfilingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createProfilingGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a profiling group.
     * </p>
     *
     * @param deleteProfilingGroupRequest
     *        Request for DeleteProfilingGroup operation.
     * @return A Java Future containing the result of the DeleteProfilingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the API.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeGuruProfilerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeGuruProfilerAsyncClient.DeleteProfilingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/DeleteProfilingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProfilingGroupResponse> deleteProfilingGroup(
            DeleteProfilingGroupRequest deleteProfilingGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProfilingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteProfilingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProfilingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProfilingGroupRequest, DeleteProfilingGroupResponse>()
                            .withOperationName("DeleteProfilingGroup")
                            .withMarshaller(new DeleteProfilingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteProfilingGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describe a profiling group.
     * </p>
     *
     * @param describeProfilingGroupRequest
     *        Request for DescribeProfilingGroup operation.
     * @return A Java Future containing the result of the DescribeProfilingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the API.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeGuruProfilerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeGuruProfilerAsyncClient.DescribeProfilingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/DescribeProfilingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProfilingGroupResponse> describeProfilingGroup(
            DescribeProfilingGroupRequest describeProfilingGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProfilingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProfilingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProfilingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProfilingGroupRequest, DescribeProfilingGroupResponse>()
                            .withOperationName("DescribeProfilingGroup")
                            .withMarshaller(new DescribeProfilingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeProfilingGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the aggregated profile of a profiling group for the specified time range. If the requested time range does
     * not align with the available aggregated profiles, it will be expanded to attain alignment. If aggregated profiles
     * are available only for part of the period requested, the profile is returned from the earliest available to the
     * latest within the requested time range. For instance, if the requested time range is from 00:00 to 00:20 and the
     * available profiles are from 00:15 to 00:25, then the returned profile will be from 00:15 to 00:20.
     * </p>
     *
     * @param getProfileRequest
     *        Request for GetProfile operation.
     * @return A Java Future containing the result of the GetProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the API.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeGuruProfilerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeGuruProfilerAsyncClient.GetProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/GetProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetProfileResponse> getProfile(GetProfileRequest getProfileRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProfileRequest, GetProfileResponse>().withOperationName("GetProfile")
                            .withMarshaller(new GetProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getProfileRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the start times of the available aggregated profiles of a profiling group for an aggregation period within
     * the specified time range.
     * </p>
     *
     * @param listProfileTimesRequest
     *        Request for ListProfileTimes operation.
     * @return A Java Future containing the result of the ListProfileTimes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the API.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeGuruProfilerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeGuruProfilerAsyncClient.ListProfileTimes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/ListProfileTimes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProfileTimesResponse> listProfileTimes(ListProfileTimesRequest listProfileTimesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProfileTimesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProfileTimesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProfileTimesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProfileTimesRequest, ListProfileTimesResponse>()
                            .withOperationName("ListProfileTimes")
                            .withMarshaller(new ListProfileTimesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listProfileTimesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the start times of the available aggregated profiles of a profiling group for an aggregation period within
     * the specified time range.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProfileTimes(software.amazon.awssdk.services.codeguruprofiler.model.ListProfileTimesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfileTimesPublisher publisher = client.listProfileTimesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfileTimesPublisher publisher = client.listProfileTimesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codeguruprofiler.model.ListProfileTimesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codeguruprofiler.model.ListProfileTimesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProfileTimes(software.amazon.awssdk.services.codeguruprofiler.model.ListProfileTimesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProfileTimesRequest
     *        Request for ListProfileTimes operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the API.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeGuruProfilerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeGuruProfilerAsyncClient.ListProfileTimes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/ListProfileTimes"
     *      target="_top">AWS API Documentation</a>
     */
    public ListProfileTimesPublisher listProfileTimesPaginator(ListProfileTimesRequest listProfileTimesRequest) {
        return new ListProfileTimesPublisher(this, applyPaginatorUserAgent(listProfileTimesRequest));
    }

    /**
     * <p>
     * List profiling groups in the account.
     * </p>
     *
     * @param listProfilingGroupsRequest
     *        Request for ListProfilingGroups operation.
     * @return A Java Future containing the result of the ListProfilingGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeGuruProfilerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeGuruProfilerAsyncClient.ListProfilingGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/ListProfilingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProfilingGroupsResponse> listProfilingGroups(
            ListProfilingGroupsRequest listProfilingGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProfilingGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProfilingGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProfilingGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProfilingGroupsRequest, ListProfilingGroupsResponse>()
                            .withOperationName("ListProfilingGroups")
                            .withMarshaller(new ListProfilingGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listProfilingGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List profiling groups in the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProfilingGroups(software.amazon.awssdk.services.codeguruprofiler.model.ListProfilingGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfilingGroupsPublisher publisher = client.listProfilingGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfilingGroupsPublisher publisher = client.listProfilingGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codeguruprofiler.model.ListProfilingGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codeguruprofiler.model.ListProfilingGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProfilingGroups(software.amazon.awssdk.services.codeguruprofiler.model.ListProfilingGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProfilingGroupsRequest
     *        Request for ListProfilingGroups operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeGuruProfilerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeGuruProfilerAsyncClient.ListProfilingGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/ListProfilingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    public ListProfilingGroupsPublisher listProfilingGroupsPaginator(ListProfilingGroupsRequest listProfilingGroupsRequest) {
        return new ListProfilingGroupsPublisher(this, applyPaginatorUserAgent(listProfilingGroupsRequest));
    }

    /**
     * <p>
     * Submit profile collected by an agent belonging to a profiling group for aggregation.
     * </p>
     *
     * @param postAgentProfileRequest
     *        Request for PostAgentProfile operation.
     * @return A Java Future containing the result of the PostAgentProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the API.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeGuruProfilerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeGuruProfilerAsyncClient.PostAgentProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/PostAgentProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PostAgentProfileResponse> postAgentProfile(PostAgentProfileRequest postAgentProfileRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PostAgentProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PostAgentProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PostAgentProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PostAgentProfileRequest, PostAgentProfileResponse>()
                            .withOperationName("PostAgentProfile")
                            .withMarshaller(new PostAgentProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(postAgentProfileRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a profiling group.
     * </p>
     *
     * @param updateProfilingGroupRequest
     *        Request for UpdateProfilingGroup operation.
     * @return A Java Future containing the result of the UpdateProfilingGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ConflictException Request can can cause an inconsistent state for the resource.</li>
     *         <li>ValidationException The input fails to satisfy the constraints of the API.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeGuruProfilerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeGuruProfilerAsyncClient.UpdateProfilingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/UpdateProfilingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProfilingGroupResponse> updateProfilingGroup(
            UpdateProfilingGroupRequest updateProfilingGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProfilingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateProfilingGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProfilingGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProfilingGroupRequest, UpdateProfilingGroupResponse>()
                            .withOperationName("UpdateProfilingGroup")
                            .withMarshaller(new UpdateProfilingGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateProfilingGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodeGuruProfilerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    private <T extends CodeGuruProfilerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
