/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon ECS service recommendation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ECSServiceRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<ECSServiceRecommendation.Builder, ECSServiceRecommendation> {
    private static final SdkField<String> SERVICE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceArn").getter(getter(ECSServiceRecommendation::serviceArn)).setter(setter(Builder::serviceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceArn").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(ECSServiceRecommendation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<ServiceConfiguration> CURRENT_SERVICE_CONFIGURATION_FIELD = SdkField
            .<ServiceConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("currentServiceConfiguration")
            .getter(getter(ECSServiceRecommendation::currentServiceConfiguration))
            .setter(setter(Builder::currentServiceConfiguration))
            .constructor(ServiceConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentServiceConfiguration")
                    .build()).build();

    private static final SdkField<List<ECSServiceUtilizationMetric>> UTILIZATION_METRICS_FIELD = SdkField
            .<List<ECSServiceUtilizationMetric>> builder(MarshallingType.LIST)
            .memberName("utilizationMetrics")
            .getter(getter(ECSServiceRecommendation::utilizationMetrics))
            .setter(setter(Builder::utilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ECSServiceUtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(ECSServiceUtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> LOOKBACK_PERIOD_IN_DAYS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("lookbackPeriodInDays").getter(getter(ECSServiceRecommendation::lookbackPeriodInDays))
            .setter(setter(Builder::lookbackPeriodInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookbackPeriodInDays").build())
            .build();

    private static final SdkField<String> LAUNCH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("launchType").getter(getter(ECSServiceRecommendation::launchTypeAsString))
            .setter(setter(Builder::launchType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchType").build()).build();

    private static final SdkField<Instant> LAST_REFRESH_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastRefreshTimestamp").getter(getter(ECSServiceRecommendation::lastRefreshTimestamp))
            .setter(setter(Builder::lastRefreshTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTimestamp").build())
            .build();

    private static final SdkField<String> FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("finding")
            .getter(getter(ECSServiceRecommendation::findingAsString)).setter(setter(Builder::finding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finding").build()).build();

    private static final SdkField<List<String>> FINDING_REASON_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("findingReasonCodes")
            .getter(getter(ECSServiceRecommendation::findingReasonCodesAsStrings))
            .setter(setter(Builder::findingReasonCodesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("findingReasonCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ECSServiceRecommendationOption>> SERVICE_RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<ECSServiceRecommendationOption>> builder(MarshallingType.LIST)
            .memberName("serviceRecommendationOptions")
            .getter(getter(ECSServiceRecommendation::serviceRecommendationOptions))
            .setter(setter(Builder::serviceRecommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceRecommendationOptions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ECSServiceRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(ECSServiceRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CURRENT_PERFORMANCE_RISK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("currentPerformanceRisk").getter(getter(ECSServiceRecommendation::currentPerformanceRiskAsString))
            .setter(setter(Builder::currentPerformanceRisk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentPerformanceRisk").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(ECSServiceRecommendation::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVICE_ARN_FIELD,
            ACCOUNT_ID_FIELD, CURRENT_SERVICE_CONFIGURATION_FIELD, UTILIZATION_METRICS_FIELD, LOOKBACK_PERIOD_IN_DAYS_FIELD,
            LAUNCH_TYPE_FIELD, LAST_REFRESH_TIMESTAMP_FIELD, FINDING_FIELD, FINDING_REASON_CODES_FIELD,
            SERVICE_RECOMMENDATION_OPTIONS_FIELD, CURRENT_PERFORMANCE_RISK_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String serviceArn;

    private final String accountId;

    private final ServiceConfiguration currentServiceConfiguration;

    private final List<ECSServiceUtilizationMetric> utilizationMetrics;

    private final Double lookbackPeriodInDays;

    private final String launchType;

    private final Instant lastRefreshTimestamp;

    private final String finding;

    private final List<String> findingReasonCodes;

    private final List<ECSServiceRecommendationOption> serviceRecommendationOptions;

    private final String currentPerformanceRisk;

    private final List<Tag> tags;

    private ECSServiceRecommendation(BuilderImpl builder) {
        this.serviceArn = builder.serviceArn;
        this.accountId = builder.accountId;
        this.currentServiceConfiguration = builder.currentServiceConfiguration;
        this.utilizationMetrics = builder.utilizationMetrics;
        this.lookbackPeriodInDays = builder.lookbackPeriodInDays;
        this.launchType = builder.launchType;
        this.lastRefreshTimestamp = builder.lastRefreshTimestamp;
        this.finding = builder.finding;
        this.findingReasonCodes = builder.findingReasonCodes;
        this.serviceRecommendationOptions = builder.serviceRecommendationOptions;
        this.currentPerformanceRisk = builder.currentPerformanceRisk;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the current Amazon ECS service.
     * </p>
     * <p>
     * The following is the format of the ARN:
     * </p>
     * <p>
     * <code>arn:aws:ecs:region:aws_account_id:service/cluster-name/service-name</code>
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the current Amazon ECS service. </p>
     *         <p>
     *         The following is the format of the ARN:
     *         </p>
     *         <p>
     *         <code>arn:aws:ecs:region:aws_account_id:service/cluster-name/service-name</code>
     */
    public final String serviceArn() {
        return serviceArn;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the Amazon ECS service.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the Amazon ECS service.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The configuration of the current Amazon ECS service.
     * </p>
     * 
     * @return The configuration of the current Amazon ECS service.
     */
    public final ServiceConfiguration currentServiceConfiguration() {
        return currentServiceConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the UtilizationMetrics property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUtilizationMetrics() {
        return utilizationMetrics != null && !(utilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the utilization metrics of the Amazon ECS service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUtilizationMetrics} method.
     * </p>
     * 
     * @return An array of objects that describe the utilization metrics of the Amazon ECS service.
     */
    public final List<ECSServiceUtilizationMetric> utilizationMetrics() {
        return utilizationMetrics;
    }

    /**
     * <p>
     * The number of days the Amazon ECS service utilization metrics were analyzed.
     * </p>
     * 
     * @return The number of days the Amazon ECS service utilization metrics were analyzed.
     */
    public final Double lookbackPeriodInDays() {
        return lookbackPeriodInDays;
    }

    /**
     * <p>
     * The launch type the Amazon ECS service is using.
     * </p>
     * <note>
     * <p>
     * Compute Optimizer only supports the Fargate launch type.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link ECSServiceLaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #launchTypeAsString}.
     * </p>
     * 
     * @return The launch type the Amazon ECS service is using. </p> <note>
     *         <p>
     *         Compute Optimizer only supports the Fargate launch type.
     *         </p>
     * @see ECSServiceLaunchType
     */
    public final ECSServiceLaunchType launchType() {
        return ECSServiceLaunchType.fromValue(launchType);
    }

    /**
     * <p>
     * The launch type the Amazon ECS service is using.
     * </p>
     * <note>
     * <p>
     * Compute Optimizer only supports the Fargate launch type.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link ECSServiceLaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #launchTypeAsString}.
     * </p>
     * 
     * @return The launch type the Amazon ECS service is using. </p> <note>
     *         <p>
     *         Compute Optimizer only supports the Fargate launch type.
     *         </p>
     * @see ECSServiceLaunchType
     */
    public final String launchTypeAsString() {
        return launchType;
    }

    /**
     * <p>
     * The timestamp of when the Amazon ECS service recommendation was last generated.
     * </p>
     * 
     * @return The timestamp of when the Amazon ECS service recommendation was last generated.
     */
    public final Instant lastRefreshTimestamp() {
        return lastRefreshTimestamp;
    }

    /**
     * <p>
     * The finding classification of an Amazon ECS service.
     * </p>
     * <p>
     * Findings for Amazon ECS services include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough memory or CPU, an
     * Amazon ECS service is considered under-provisioned. An under-provisioned service might result in poor application
     * performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive memory or CPU, an
     * Amazon ECS service is considered over-provisioned. An over-provisioned service might result in additional
     * infrastructure costs.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b> — When both the CPU and memory of your Amazon ECS service meet the performance
     * requirements of your workload, the service is considered optimized.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link ECSServiceRecommendationFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification of an Amazon ECS service. </p>
     *         <p>
     *         Findings for Amazon ECS services include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough memory or
     *         CPU, an Amazon ECS service is considered under-provisioned. An under-provisioned service might result in
     *         poor application performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive memory or
     *         CPU, an Amazon ECS service is considered over-provisioned. An over-provisioned service might result in
     *         additional infrastructure costs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b> — When both the CPU and memory of your Amazon ECS service meet the
     *         performance requirements of your workload, the service is considered optimized.
     *         </p>
     *         </li>
     * @see ECSServiceRecommendationFinding
     */
    public final ECSServiceRecommendationFinding finding() {
        return ECSServiceRecommendationFinding.fromValue(finding);
    }

    /**
     * <p>
     * The finding classification of an Amazon ECS service.
     * </p>
     * <p>
     * Findings for Amazon ECS services include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough memory or CPU, an
     * Amazon ECS service is considered under-provisioned. An under-provisioned service might result in poor application
     * performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive memory or CPU, an
     * Amazon ECS service is considered over-provisioned. An over-provisioned service might result in additional
     * infrastructure costs.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b> — When both the CPU and memory of your Amazon ECS service meet the performance
     * requirements of your workload, the service is considered optimized.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link ECSServiceRecommendationFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification of an Amazon ECS service. </p>
     *         <p>
     *         Findings for Amazon ECS services include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough memory or
     *         CPU, an Amazon ECS service is considered under-provisioned. An under-provisioned service might result in
     *         poor application performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive memory or
     *         CPU, an Amazon ECS service is considered over-provisioned. An over-provisioned service might result in
     *         additional infrastructure costs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b> — When both the CPU and memory of your Amazon ECS service meet the
     *         performance requirements of your workload, the service is considered optimized.
     *         </p>
     *         </li>
     * @see ECSServiceRecommendationFinding
     */
    public final String findingAsString() {
        return finding;
    }

    /**
     * <p>
     * The reason for the finding classification of an Amazon ECS service.
     * </p>
     * <p>
     * Finding reason codes for Amazon ECS services include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance the
     * performance of your workload. This is identified by analyzing the <code>CPUUtilization</code> metric of the
     * current service during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still meeting
     * the performance requirements of your workload. This is identified by analyzing the <code>CPUUtilization</code>
     * metric of the current service during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to enhance the
     * performance of your workload. This is identified by analyzing the <code>MemoryUtilization</code> metric of the
     * current service during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while still
     * meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>MemoryUtilization</code> metric of the current service during the look-back period.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification of an Amazon ECS service. </p>
     *         <p>
     *         Finding reason codes for Amazon ECS services include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance the
     *         performance of your workload. This is identified by analyzing the <code>CPUUtilization</code> metric of
     *         the current service during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still
     *         meeting the performance requirements of your workload. This is identified by analyzing the
     *         <code>CPUUtilization</code> metric of the current service during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to
     *         enhance the performance of your workload. This is identified by analyzing the
     *         <code>MemoryUtilization</code> metric of the current service during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while
     *         still meeting the performance requirements of your workload. This is identified by analyzing the
     *         <code>MemoryUtilization</code> metric of the current service during the look-back period.
     *         </p>
     *         </li>
     */
    public final List<ECSServiceRecommendationFindingReasonCode> findingReasonCodes() {
        return ECSServiceRecommendationFindingReasonCodesCopier.copyStringToEnum(findingReasonCodes);
    }

    /**
     * For responses, this returns true if the service returned a value for the FindingReasonCodes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFindingReasonCodes() {
        return findingReasonCodes != null && !(findingReasonCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The reason for the finding classification of an Amazon ECS service.
     * </p>
     * <p>
     * Finding reason codes for Amazon ECS services include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance the
     * performance of your workload. This is identified by analyzing the <code>CPUUtilization</code> metric of the
     * current service during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still meeting
     * the performance requirements of your workload. This is identified by analyzing the <code>CPUUtilization</code>
     * metric of the current service during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to enhance the
     * performance of your workload. This is identified by analyzing the <code>MemoryUtilization</code> metric of the
     * current service during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while still
     * meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>MemoryUtilization</code> metric of the current service during the look-back period.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification of an Amazon ECS service. </p>
     *         <p>
     *         Finding reason codes for Amazon ECS services include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance the
     *         performance of your workload. This is identified by analyzing the <code>CPUUtilization</code> metric of
     *         the current service during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still
     *         meeting the performance requirements of your workload. This is identified by analyzing the
     *         <code>CPUUtilization</code> metric of the current service during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to
     *         enhance the performance of your workload. This is identified by analyzing the
     *         <code>MemoryUtilization</code> metric of the current service during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while
     *         still meeting the performance requirements of your workload. This is identified by analyzing the
     *         <code>MemoryUtilization</code> metric of the current service during the look-back period.
     *         </p>
     *         </li>
     */
    public final List<String> findingReasonCodesAsStrings() {
        return findingReasonCodes;
    }

    /**
     * For responses, this returns true if the service returned a value for the ServiceRecommendationOptions property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasServiceRecommendationOptions() {
        return serviceRecommendationOptions != null && !(serviceRecommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the recommendation options for the Amazon ECS service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasServiceRecommendationOptions} method.
     * </p>
     * 
     * @return An array of objects that describe the recommendation options for the Amazon ECS service.
     */
    public final List<ECSServiceRecommendationOption> serviceRecommendationOptions() {
        return serviceRecommendationOptions;
    }

    /**
     * <p>
     * The risk of the current Amazon ECS service not meeting the performance needs of its workloads. The higher the
     * risk, the more likely the current service can't meet the performance requirements of its workload.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentPerformanceRisk} will return {@link CurrentPerformanceRisk#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #currentPerformanceRiskAsString}.
     * </p>
     * 
     * @return The risk of the current Amazon ECS service not meeting the performance needs of its workloads. The higher
     *         the risk, the more likely the current service can't meet the performance requirements of its workload.
     * @see CurrentPerformanceRisk
     */
    public final CurrentPerformanceRisk currentPerformanceRisk() {
        return CurrentPerformanceRisk.fromValue(currentPerformanceRisk);
    }

    /**
     * <p>
     * The risk of the current Amazon ECS service not meeting the performance needs of its workloads. The higher the
     * risk, the more likely the current service can't meet the performance requirements of its workload.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentPerformanceRisk} will return {@link CurrentPerformanceRisk#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #currentPerformanceRiskAsString}.
     * </p>
     * 
     * @return The risk of the current Amazon ECS service not meeting the performance needs of its workloads. The higher
     *         the risk, the more likely the current service can't meet the performance requirements of its workload.
     * @see CurrentPerformanceRisk
     */
    public final String currentPerformanceRiskAsString() {
        return currentPerformanceRisk;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags assigned to your Amazon ECS service recommendations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags assigned to your Amazon ECS service recommendations.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(serviceArn());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(currentServiceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasUtilizationMetrics() ? utilizationMetrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lookbackPeriodInDays());
        hashCode = 31 * hashCode + Objects.hashCode(launchTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(findingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasFindingReasonCodes() ? findingReasonCodesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasServiceRecommendationOptions() ? serviceRecommendationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(currentPerformanceRiskAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ECSServiceRecommendation)) {
            return false;
        }
        ECSServiceRecommendation other = (ECSServiceRecommendation) obj;
        return Objects.equals(serviceArn(), other.serviceArn()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(currentServiceConfiguration(), other.currentServiceConfiguration())
                && hasUtilizationMetrics() == other.hasUtilizationMetrics()
                && Objects.equals(utilizationMetrics(), other.utilizationMetrics())
                && Objects.equals(lookbackPeriodInDays(), other.lookbackPeriodInDays())
                && Objects.equals(launchTypeAsString(), other.launchTypeAsString())
                && Objects.equals(lastRefreshTimestamp(), other.lastRefreshTimestamp())
                && Objects.equals(findingAsString(), other.findingAsString())
                && hasFindingReasonCodes() == other.hasFindingReasonCodes()
                && Objects.equals(findingReasonCodesAsStrings(), other.findingReasonCodesAsStrings())
                && hasServiceRecommendationOptions() == other.hasServiceRecommendationOptions()
                && Objects.equals(serviceRecommendationOptions(), other.serviceRecommendationOptions())
                && Objects.equals(currentPerformanceRiskAsString(), other.currentPerformanceRiskAsString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ECSServiceRecommendation").add("ServiceArn", serviceArn()).add("AccountId", accountId())
                .add("CurrentServiceConfiguration", currentServiceConfiguration())
                .add("UtilizationMetrics", hasUtilizationMetrics() ? utilizationMetrics() : null)
                .add("LookbackPeriodInDays", lookbackPeriodInDays()).add("LaunchType", launchTypeAsString())
                .add("LastRefreshTimestamp", lastRefreshTimestamp()).add("Finding", findingAsString())
                .add("FindingReasonCodes", hasFindingReasonCodes() ? findingReasonCodesAsStrings() : null)
                .add("ServiceRecommendationOptions", hasServiceRecommendationOptions() ? serviceRecommendationOptions() : null)
                .add("CurrentPerformanceRisk", currentPerformanceRiskAsString()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "serviceArn":
            return Optional.ofNullable(clazz.cast(serviceArn()));
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "currentServiceConfiguration":
            return Optional.ofNullable(clazz.cast(currentServiceConfiguration()));
        case "utilizationMetrics":
            return Optional.ofNullable(clazz.cast(utilizationMetrics()));
        case "lookbackPeriodInDays":
            return Optional.ofNullable(clazz.cast(lookbackPeriodInDays()));
        case "launchType":
            return Optional.ofNullable(clazz.cast(launchTypeAsString()));
        case "lastRefreshTimestamp":
            return Optional.ofNullable(clazz.cast(lastRefreshTimestamp()));
        case "finding":
            return Optional.ofNullable(clazz.cast(findingAsString()));
        case "findingReasonCodes":
            return Optional.ofNullable(clazz.cast(findingReasonCodesAsStrings()));
        case "serviceRecommendationOptions":
            return Optional.ofNullable(clazz.cast(serviceRecommendationOptions()));
        case "currentPerformanceRisk":
            return Optional.ofNullable(clazz.cast(currentPerformanceRiskAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ECSServiceRecommendation, T> g) {
        return obj -> g.apply((ECSServiceRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ECSServiceRecommendation> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the current Amazon ECS service.
         * </p>
         * <p>
         * The following is the format of the ARN:
         * </p>
         * <p>
         * <code>arn:aws:ecs:region:aws_account_id:service/cluster-name/service-name</code>
         * </p>
         * 
         * @param serviceArn
         *        The Amazon Resource Name (ARN) of the current Amazon ECS service. </p>
         *        <p>
         *        The following is the format of the ARN:
         *        </p>
         *        <p>
         *        <code>arn:aws:ecs:region:aws_account_id:service/cluster-name/service-name</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceArn(String serviceArn);

        /**
         * <p>
         * The Amazon Web Services account ID of the Amazon ECS service.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID of the Amazon ECS service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The configuration of the current Amazon ECS service.
         * </p>
         * 
         * @param currentServiceConfiguration
         *        The configuration of the current Amazon ECS service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentServiceConfiguration(ServiceConfiguration currentServiceConfiguration);

        /**
         * <p>
         * The configuration of the current Amazon ECS service.
         * </p>
         * This is a convenience method that creates an instance of the {@link ServiceConfiguration.Builder} avoiding
         * the need to create one manually via {@link ServiceConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ServiceConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #currentServiceConfiguration(ServiceConfiguration)}.
         * 
         * @param currentServiceConfiguration
         *        a consumer that will call methods on {@link ServiceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #currentServiceConfiguration(ServiceConfiguration)
         */
        default Builder currentServiceConfiguration(Consumer<ServiceConfiguration.Builder> currentServiceConfiguration) {
            return currentServiceConfiguration(ServiceConfiguration.builder().applyMutation(currentServiceConfiguration).build());
        }

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Amazon ECS service.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the Amazon ECS service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(Collection<ECSServiceUtilizationMetric> utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Amazon ECS service.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the Amazon ECS service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(ECSServiceUtilizationMetric... utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Amazon ECS service.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.ECSServiceUtilizationMetric.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.ECSServiceUtilizationMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.ECSServiceUtilizationMetric.Builder#build()} is
         * called immediately and its result is passed to {@link #utilizationMetrics(List<ECSServiceUtilizationMetric>)}.
         * 
         * @param utilizationMetrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.ECSServiceUtilizationMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationMetrics(java.util.Collection<ECSServiceUtilizationMetric>)
         */
        Builder utilizationMetrics(Consumer<ECSServiceUtilizationMetric.Builder>... utilizationMetrics);

        /**
         * <p>
         * The number of days the Amazon ECS service utilization metrics were analyzed.
         * </p>
         * 
         * @param lookbackPeriodInDays
         *        The number of days the Amazon ECS service utilization metrics were analyzed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lookbackPeriodInDays(Double lookbackPeriodInDays);

        /**
         * <p>
         * The launch type the Amazon ECS service is using.
         * </p>
         * <note>
         * <p>
         * Compute Optimizer only supports the Fargate launch type.
         * </p>
         * </note>
         * 
         * @param launchType
         *        The launch type the Amazon ECS service is using. </p> <note>
         *        <p>
         *        Compute Optimizer only supports the Fargate launch type.
         *        </p>
         * @see ECSServiceLaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ECSServiceLaunchType
         */
        Builder launchType(String launchType);

        /**
         * <p>
         * The launch type the Amazon ECS service is using.
         * </p>
         * <note>
         * <p>
         * Compute Optimizer only supports the Fargate launch type.
         * </p>
         * </note>
         * 
         * @param launchType
         *        The launch type the Amazon ECS service is using. </p> <note>
         *        <p>
         *        Compute Optimizer only supports the Fargate launch type.
         *        </p>
         * @see ECSServiceLaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ECSServiceLaunchType
         */
        Builder launchType(ECSServiceLaunchType launchType);

        /**
         * <p>
         * The timestamp of when the Amazon ECS service recommendation was last generated.
         * </p>
         * 
         * @param lastRefreshTimestamp
         *        The timestamp of when the Amazon ECS service recommendation was last generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTimestamp(Instant lastRefreshTimestamp);

        /**
         * <p>
         * The finding classification of an Amazon ECS service.
         * </p>
         * <p>
         * Findings for Amazon ECS services include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough memory or
         * CPU, an Amazon ECS service is considered under-provisioned. An under-provisioned service might result in poor
         * application performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive memory or CPU,
         * an Amazon ECS service is considered over-provisioned. An over-provisioned service might result in additional
         * infrastructure costs.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b> — When both the CPU and memory of your Amazon ECS service meet the
         * performance requirements of your workload, the service is considered optimized.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification of an Amazon ECS service. </p>
         *        <p>
         *        Findings for Amazon ECS services include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough memory
         *        or CPU, an Amazon ECS service is considered under-provisioned. An under-provisioned service might
         *        result in poor application performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive memory
         *        or CPU, an Amazon ECS service is considered over-provisioned. An over-provisioned service might result
         *        in additional infrastructure costs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b> — When both the CPU and memory of your Amazon ECS service meet the
         *        performance requirements of your workload, the service is considered optimized.
         *        </p>
         *        </li>
         * @see ECSServiceRecommendationFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ECSServiceRecommendationFinding
         */
        Builder finding(String finding);

        /**
         * <p>
         * The finding classification of an Amazon ECS service.
         * </p>
         * <p>
         * Findings for Amazon ECS services include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough memory or
         * CPU, an Amazon ECS service is considered under-provisioned. An under-provisioned service might result in poor
         * application performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive memory or CPU,
         * an Amazon ECS service is considered over-provisioned. An over-provisioned service might result in additional
         * infrastructure costs.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b> — When both the CPU and memory of your Amazon ECS service meet the
         * performance requirements of your workload, the service is considered optimized.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification of an Amazon ECS service. </p>
         *        <p>
         *        Findings for Amazon ECS services include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Underprovisioned</code> </b> — When Compute Optimizer detects that there’s not enough memory
         *        or CPU, an Amazon ECS service is considered under-provisioned. An under-provisioned service might
         *        result in poor application performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Overprovisioned</code> </b> — When Compute Optimizer detects that there’s excessive memory
         *        or CPU, an Amazon ECS service is considered over-provisioned. An over-provisioned service might result
         *        in additional infrastructure costs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b> — When both the CPU and memory of your Amazon ECS service meet the
         *        performance requirements of your workload, the service is considered optimized.
         *        </p>
         *        </li>
         * @see ECSServiceRecommendationFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ECSServiceRecommendationFinding
         */
        Builder finding(ECSServiceRecommendationFinding finding);

        /**
         * <p>
         * The reason for the finding classification of an Amazon ECS service.
         * </p>
         * <p>
         * Finding reason codes for Amazon ECS services include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance the
         * performance of your workload. This is identified by analyzing the <code>CPUUtilization</code> metric of the
         * current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>CPUUtilization</code> metric of the current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to enhance
         * the performance of your workload. This is identified by analyzing the <code>MemoryUtilization</code> metric
         * of the current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>MemoryUtilization</code> metric of the current service during the look-back period.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of an Amazon ECS service. </p>
         *        <p>
         *        Finding reason codes for Amazon ECS services include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance
         *        the performance of your workload. This is identified by analyzing the <code>CPUUtilization</code>
         *        metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still
         *        meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>CPUUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to
         *        enhance the performance of your workload. This is identified by analyzing the
         *        <code>MemoryUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while
         *        still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>MemoryUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodesWithStrings(Collection<String> findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of an Amazon ECS service.
         * </p>
         * <p>
         * Finding reason codes for Amazon ECS services include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance the
         * performance of your workload. This is identified by analyzing the <code>CPUUtilization</code> metric of the
         * current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>CPUUtilization</code> metric of the current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to enhance
         * the performance of your workload. This is identified by analyzing the <code>MemoryUtilization</code> metric
         * of the current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>MemoryUtilization</code> metric of the current service during the look-back period.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of an Amazon ECS service. </p>
         *        <p>
         *        Finding reason codes for Amazon ECS services include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance
         *        the performance of your workload. This is identified by analyzing the <code>CPUUtilization</code>
         *        metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still
         *        meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>CPUUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to
         *        enhance the performance of your workload. This is identified by analyzing the
         *        <code>MemoryUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while
         *        still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>MemoryUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodesWithStrings(String... findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of an Amazon ECS service.
         * </p>
         * <p>
         * Finding reason codes for Amazon ECS services include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance the
         * performance of your workload. This is identified by analyzing the <code>CPUUtilization</code> metric of the
         * current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>CPUUtilization</code> metric of the current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to enhance
         * the performance of your workload. This is identified by analyzing the <code>MemoryUtilization</code> metric
         * of the current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>MemoryUtilization</code> metric of the current service during the look-back period.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of an Amazon ECS service. </p>
         *        <p>
         *        Finding reason codes for Amazon ECS services include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance
         *        the performance of your workload. This is identified by analyzing the <code>CPUUtilization</code>
         *        metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still
         *        meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>CPUUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to
         *        enhance the performance of your workload. This is identified by analyzing the
         *        <code>MemoryUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while
         *        still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>MemoryUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodes(Collection<ECSServiceRecommendationFindingReasonCode> findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of an Amazon ECS service.
         * </p>
         * <p>
         * Finding reason codes for Amazon ECS services include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance the
         * performance of your workload. This is identified by analyzing the <code>CPUUtilization</code> metric of the
         * current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>CPUUtilization</code> metric of the current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to enhance
         * the performance of your workload. This is identified by analyzing the <code>MemoryUtilization</code> metric
         * of the current service during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>MemoryUtilization</code> metric of the current service during the look-back period.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of an Amazon ECS service. </p>
         *        <p>
         *        Finding reason codes for Amazon ECS services include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>CPUUnderprovisioned</code> </b> — The service CPU configuration can be sized up to enhance
         *        the performance of your workload. This is identified by analyzing the <code>CPUUtilization</code>
         *        metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUOverprovisioned</code> </b> — The service CPU configuration can be sized down while still
         *        meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>CPUUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The service memory configuration can be sized up to
         *        enhance the performance of your workload. This is identified by analyzing the
         *        <code>MemoryUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The service memory configuration can be sized down while
         *        still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>MemoryUtilization</code> metric of the current service during the look-back period.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodes(ECSServiceRecommendationFindingReasonCode... findingReasonCodes);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Amazon ECS service.
         * </p>
         * 
         * @param serviceRecommendationOptions
         *        An array of objects that describe the recommendation options for the Amazon ECS service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRecommendationOptions(Collection<ECSServiceRecommendationOption> serviceRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Amazon ECS service.
         * </p>
         * 
         * @param serviceRecommendationOptions
         *        An array of objects that describe the recommendation options for the Amazon ECS service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRecommendationOptions(ECSServiceRecommendationOption... serviceRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Amazon ECS service.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.ECSServiceRecommendationOption.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.ECSServiceRecommendationOption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.ECSServiceRecommendationOption.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #serviceRecommendationOptions(List<ECSServiceRecommendationOption>)}.
         * 
         * @param serviceRecommendationOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.ECSServiceRecommendationOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serviceRecommendationOptions(java.util.Collection<ECSServiceRecommendationOption>)
         */
        Builder serviceRecommendationOptions(Consumer<ECSServiceRecommendationOption.Builder>... serviceRecommendationOptions);

        /**
         * <p>
         * The risk of the current Amazon ECS service not meeting the performance needs of its workloads. The higher the
         * risk, the more likely the current service can't meet the performance requirements of its workload.
         * </p>
         * 
         * @param currentPerformanceRisk
         *        The risk of the current Amazon ECS service not meeting the performance needs of its workloads. The
         *        higher the risk, the more likely the current service can't meet the performance requirements of its
         *        workload.
         * @see CurrentPerformanceRisk
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CurrentPerformanceRisk
         */
        Builder currentPerformanceRisk(String currentPerformanceRisk);

        /**
         * <p>
         * The risk of the current Amazon ECS service not meeting the performance needs of its workloads. The higher the
         * risk, the more likely the current service can't meet the performance requirements of its workload.
         * </p>
         * 
         * @param currentPerformanceRisk
         *        The risk of the current Amazon ECS service not meeting the performance needs of its workloads. The
         *        higher the risk, the more likely the current service can't meet the performance requirements of its
         *        workload.
         * @see CurrentPerformanceRisk
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CurrentPerformanceRisk
         */
        Builder currentPerformanceRisk(CurrentPerformanceRisk currentPerformanceRisk);

        /**
         * <p>
         * A list of tags assigned to your Amazon ECS service recommendations.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to your Amazon ECS service recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags assigned to your Amazon ECS service recommendations.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to your Amazon ECS service recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags assigned to your Amazon ECS service recommendations.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.computeoptimizer.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String serviceArn;

        private String accountId;

        private ServiceConfiguration currentServiceConfiguration;

        private List<ECSServiceUtilizationMetric> utilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private Double lookbackPeriodInDays;

        private String launchType;

        private Instant lastRefreshTimestamp;

        private String finding;

        private List<String> findingReasonCodes = DefaultSdkAutoConstructList.getInstance();

        private List<ECSServiceRecommendationOption> serviceRecommendationOptions = DefaultSdkAutoConstructList.getInstance();

        private String currentPerformanceRisk;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ECSServiceRecommendation model) {
            serviceArn(model.serviceArn);
            accountId(model.accountId);
            currentServiceConfiguration(model.currentServiceConfiguration);
            utilizationMetrics(model.utilizationMetrics);
            lookbackPeriodInDays(model.lookbackPeriodInDays);
            launchType(model.launchType);
            lastRefreshTimestamp(model.lastRefreshTimestamp);
            finding(model.finding);
            findingReasonCodesWithStrings(model.findingReasonCodes);
            serviceRecommendationOptions(model.serviceRecommendationOptions);
            currentPerformanceRisk(model.currentPerformanceRisk);
            tags(model.tags);
        }

        public final String getServiceArn() {
            return serviceArn;
        }

        public final void setServiceArn(String serviceArn) {
            this.serviceArn = serviceArn;
        }

        @Override
        public final Builder serviceArn(String serviceArn) {
            this.serviceArn = serviceArn;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final ServiceConfiguration.Builder getCurrentServiceConfiguration() {
            return currentServiceConfiguration != null ? currentServiceConfiguration.toBuilder() : null;
        }

        public final void setCurrentServiceConfiguration(ServiceConfiguration.BuilderImpl currentServiceConfiguration) {
            this.currentServiceConfiguration = currentServiceConfiguration != null ? currentServiceConfiguration.build() : null;
        }

        @Override
        public final Builder currentServiceConfiguration(ServiceConfiguration currentServiceConfiguration) {
            this.currentServiceConfiguration = currentServiceConfiguration;
            return this;
        }

        public final List<ECSServiceUtilizationMetric.Builder> getUtilizationMetrics() {
            List<ECSServiceUtilizationMetric.Builder> result = ECSServiceUtilizationMetricsCopier
                    .copyToBuilder(this.utilizationMetrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUtilizationMetrics(Collection<ECSServiceUtilizationMetric.BuilderImpl> utilizationMetrics) {
            this.utilizationMetrics = ECSServiceUtilizationMetricsCopier.copyFromBuilder(utilizationMetrics);
        }

        @Override
        public final Builder utilizationMetrics(Collection<ECSServiceUtilizationMetric> utilizationMetrics) {
            this.utilizationMetrics = ECSServiceUtilizationMetricsCopier.copy(utilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(ECSServiceUtilizationMetric... utilizationMetrics) {
            utilizationMetrics(Arrays.asList(utilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(Consumer<ECSServiceUtilizationMetric.Builder>... utilizationMetrics) {
            utilizationMetrics(Stream.of(utilizationMetrics)
                    .map(c -> ECSServiceUtilizationMetric.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Double getLookbackPeriodInDays() {
            return lookbackPeriodInDays;
        }

        public final void setLookbackPeriodInDays(Double lookbackPeriodInDays) {
            this.lookbackPeriodInDays = lookbackPeriodInDays;
        }

        @Override
        public final Builder lookbackPeriodInDays(Double lookbackPeriodInDays) {
            this.lookbackPeriodInDays = lookbackPeriodInDays;
            return this;
        }

        public final String getLaunchType() {
            return launchType;
        }

        public final void setLaunchType(String launchType) {
            this.launchType = launchType;
        }

        @Override
        public final Builder launchType(String launchType) {
            this.launchType = launchType;
            return this;
        }

        @Override
        public final Builder launchType(ECSServiceLaunchType launchType) {
            this.launchType(launchType == null ? null : launchType.toString());
            return this;
        }

        public final Instant getLastRefreshTimestamp() {
            return lastRefreshTimestamp;
        }

        public final void setLastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
        }

        @Override
        public final Builder lastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
            return this;
        }

        public final String getFinding() {
            return finding;
        }

        public final void setFinding(String finding) {
            this.finding = finding;
        }

        @Override
        public final Builder finding(String finding) {
            this.finding = finding;
            return this;
        }

        @Override
        public final Builder finding(ECSServiceRecommendationFinding finding) {
            this.finding(finding == null ? null : finding.toString());
            return this;
        }

        public final Collection<String> getFindingReasonCodes() {
            if (findingReasonCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return findingReasonCodes;
        }

        public final void setFindingReasonCodes(Collection<String> findingReasonCodes) {
            this.findingReasonCodes = ECSServiceRecommendationFindingReasonCodesCopier.copy(findingReasonCodes);
        }

        @Override
        public final Builder findingReasonCodesWithStrings(Collection<String> findingReasonCodes) {
            this.findingReasonCodes = ECSServiceRecommendationFindingReasonCodesCopier.copy(findingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingReasonCodesWithStrings(String... findingReasonCodes) {
            findingReasonCodesWithStrings(Arrays.asList(findingReasonCodes));
            return this;
        }

        @Override
        public final Builder findingReasonCodes(Collection<ECSServiceRecommendationFindingReasonCode> findingReasonCodes) {
            this.findingReasonCodes = ECSServiceRecommendationFindingReasonCodesCopier.copyEnumToString(findingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingReasonCodes(ECSServiceRecommendationFindingReasonCode... findingReasonCodes) {
            findingReasonCodes(Arrays.asList(findingReasonCodes));
            return this;
        }

        public final List<ECSServiceRecommendationOption.Builder> getServiceRecommendationOptions() {
            List<ECSServiceRecommendationOption.Builder> result = ECSServiceRecommendationOptionsCopier
                    .copyToBuilder(this.serviceRecommendationOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setServiceRecommendationOptions(
                Collection<ECSServiceRecommendationOption.BuilderImpl> serviceRecommendationOptions) {
            this.serviceRecommendationOptions = ECSServiceRecommendationOptionsCopier
                    .copyFromBuilder(serviceRecommendationOptions);
        }

        @Override
        public final Builder serviceRecommendationOptions(Collection<ECSServiceRecommendationOption> serviceRecommendationOptions) {
            this.serviceRecommendationOptions = ECSServiceRecommendationOptionsCopier.copy(serviceRecommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serviceRecommendationOptions(ECSServiceRecommendationOption... serviceRecommendationOptions) {
            serviceRecommendationOptions(Arrays.asList(serviceRecommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serviceRecommendationOptions(
                Consumer<ECSServiceRecommendationOption.Builder>... serviceRecommendationOptions) {
            serviceRecommendationOptions(Stream.of(serviceRecommendationOptions)
                    .map(c -> ECSServiceRecommendationOption.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getCurrentPerformanceRisk() {
            return currentPerformanceRisk;
        }

        public final void setCurrentPerformanceRisk(String currentPerformanceRisk) {
            this.currentPerformanceRisk = currentPerformanceRisk;
        }

        @Override
        public final Builder currentPerformanceRisk(String currentPerformanceRisk) {
            this.currentPerformanceRisk = currentPerformanceRisk;
            return this;
        }

        @Override
        public final Builder currentPerformanceRisk(CurrentPerformanceRisk currentPerformanceRisk) {
            this.currentPerformanceRisk(currentPerformanceRisk == null ? null : currentPerformanceRisk.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ECSServiceRecommendation build() {
            return new ECSServiceRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
