/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon EC2 instance recommendation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceRecommendation.Builder, InstanceRecommendation> {
    private static final SdkField<String> INSTANCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceArn").getter(getter(InstanceRecommendation::instanceArn)).setter(setter(Builder::instanceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceArn").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(InstanceRecommendation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<String> INSTANCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceName").getter(getter(InstanceRecommendation::instanceName))
            .setter(setter(Builder::instanceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceName").build()).build();

    private static final SdkField<String> CURRENT_INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("currentInstanceType").getter(getter(InstanceRecommendation::currentInstanceType))
            .setter(setter(Builder::currentInstanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentInstanceType").build())
            .build();

    private static final SdkField<String> FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("finding")
            .getter(getter(InstanceRecommendation::findingAsString)).setter(setter(Builder::finding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finding").build()).build();

    private static final SdkField<List<String>> FINDING_REASON_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("findingReasonCodes")
            .getter(getter(InstanceRecommendation::findingReasonCodesAsStrings))
            .setter(setter(Builder::findingReasonCodesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("findingReasonCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<UtilizationMetric>> UTILIZATION_METRICS_FIELD = SdkField
            .<List<UtilizationMetric>> builder(MarshallingType.LIST)
            .memberName("utilizationMetrics")
            .getter(getter(InstanceRecommendation::utilizationMetrics))
            .setter(setter(Builder::utilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(UtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> LOOK_BACK_PERIOD_IN_DAYS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("lookBackPeriodInDays").getter(getter(InstanceRecommendation::lookBackPeriodInDays))
            .setter(setter(Builder::lookBackPeriodInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookBackPeriodInDays").build())
            .build();

    private static final SdkField<List<InstanceRecommendationOption>> RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<InstanceRecommendationOption>> builder(MarshallingType.LIST)
            .memberName("recommendationOptions")
            .getter(getter(InstanceRecommendation::recommendationOptions))
            .setter(setter(Builder::recommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InstanceRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<RecommendationSource>> RECOMMENDATION_SOURCES_FIELD = SdkField
            .<List<RecommendationSource>> builder(MarshallingType.LIST)
            .memberName("recommendationSources")
            .getter(getter(InstanceRecommendation::recommendationSources))
            .setter(setter(Builder::recommendationSources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationSources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RecommendationSource> builder(MarshallingType.SDK_POJO)
                                            .constructor(RecommendationSource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_REFRESH_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastRefreshTimestamp").getter(getter(InstanceRecommendation::lastRefreshTimestamp))
            .setter(setter(Builder::lastRefreshTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTimestamp").build())
            .build();

    private static final SdkField<String> CURRENT_PERFORMANCE_RISK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("currentPerformanceRisk").getter(getter(InstanceRecommendation::currentPerformanceRiskAsString))
            .setter(setter(Builder::currentPerformanceRisk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentPerformanceRisk").build())
            .build();

    private static final SdkField<EffectiveRecommendationPreferences> EFFECTIVE_RECOMMENDATION_PREFERENCES_FIELD = SdkField
            .<EffectiveRecommendationPreferences> builder(MarshallingType.SDK_POJO)
            .memberName("effectiveRecommendationPreferences")
            .getter(getter(InstanceRecommendation::effectiveRecommendationPreferences))
            .setter(setter(Builder::effectiveRecommendationPreferences))
            .constructor(EffectiveRecommendationPreferences::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("effectiveRecommendationPreferences")
                    .build()).build();

    private static final SdkField<List<String>> INFERRED_WORKLOAD_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("inferredWorkloadTypes")
            .getter(getter(InstanceRecommendation::inferredWorkloadTypesAsStrings))
            .setter(setter(Builder::inferredWorkloadTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inferredWorkloadTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INSTANCE_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceState").getter(getter(InstanceRecommendation::instanceStateAsString))
            .setter(setter(Builder::instanceState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceState").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(InstanceRecommendation::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_ARN_FIELD,
            ACCOUNT_ID_FIELD, INSTANCE_NAME_FIELD, CURRENT_INSTANCE_TYPE_FIELD, FINDING_FIELD, FINDING_REASON_CODES_FIELD,
            UTILIZATION_METRICS_FIELD, LOOK_BACK_PERIOD_IN_DAYS_FIELD, RECOMMENDATION_OPTIONS_FIELD,
            RECOMMENDATION_SOURCES_FIELD, LAST_REFRESH_TIMESTAMP_FIELD, CURRENT_PERFORMANCE_RISK_FIELD,
            EFFECTIVE_RECOMMENDATION_PREFERENCES_FIELD, INFERRED_WORKLOAD_TYPES_FIELD, INSTANCE_STATE_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String instanceArn;

    private final String accountId;

    private final String instanceName;

    private final String currentInstanceType;

    private final String finding;

    private final List<String> findingReasonCodes;

    private final List<UtilizationMetric> utilizationMetrics;

    private final Double lookBackPeriodInDays;

    private final List<InstanceRecommendationOption> recommendationOptions;

    private final List<RecommendationSource> recommendationSources;

    private final Instant lastRefreshTimestamp;

    private final String currentPerformanceRisk;

    private final EffectiveRecommendationPreferences effectiveRecommendationPreferences;

    private final List<String> inferredWorkloadTypes;

    private final String instanceState;

    private final List<Tag> tags;

    private InstanceRecommendation(BuilderImpl builder) {
        this.instanceArn = builder.instanceArn;
        this.accountId = builder.accountId;
        this.instanceName = builder.instanceName;
        this.currentInstanceType = builder.currentInstanceType;
        this.finding = builder.finding;
        this.findingReasonCodes = builder.findingReasonCodes;
        this.utilizationMetrics = builder.utilizationMetrics;
        this.lookBackPeriodInDays = builder.lookBackPeriodInDays;
        this.recommendationOptions = builder.recommendationOptions;
        this.recommendationSources = builder.recommendationSources;
        this.lastRefreshTimestamp = builder.lastRefreshTimestamp;
        this.currentPerformanceRisk = builder.currentPerformanceRisk;
        this.effectiveRecommendationPreferences = builder.effectiveRecommendationPreferences;
        this.inferredWorkloadTypes = builder.inferredWorkloadTypes;
        this.instanceState = builder.instanceState;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the current instance.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the current instance.
     */
    public final String instanceArn() {
        return instanceArn;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the instance.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the instance.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The name of the current instance.
     * </p>
     * 
     * @return The name of the current instance.
     */
    public final String instanceName() {
        return instanceName;
    }

    /**
     * <p>
     * The instance type of the current instance.
     * </p>
     * 
     * @return The instance type of the current instance.
     */
    public final String currentInstanceType() {
        return currentInstanceType;
    }

    /**
     * <p>
     * The finding classification of the instance.
     * </p>
     * <p>
     * Findings for instances include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
     * specification of your instance, such as CPU, memory, or network, does not meet the performance requirements of
     * your workload. Under-provisioned instances may lead to poor application performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one specification
     * of your instance, such as CPU, memory, or network, can be sized down while still meeting the performance
     * requirements of your workload, and no specification is under-provisioned. Over-provisioned instances may lead to
     * unnecessary infrastructure cost.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your instance,
     * such as CPU, memory, and network, meet the performance requirements of your workload and is not over provisioned.
     * For optimized resources, Compute Optimizer might recommend a new generation instance type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link Finding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification of the instance.</p>
     *         <p>
     *         Findings for instances include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
     *         specification of your instance, such as CPU, memory, or network, does not meet the performance
     *         requirements of your workload. Under-provisioned instances may lead to poor application performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
     *         specification of your instance, such as CPU, memory, or network, can be sized down while still meeting
     *         the performance requirements of your workload, and no specification is under-provisioned.
     *         Over-provisioned instances may lead to unnecessary infrastructure cost.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your
     *         instance, such as CPU, memory, and network, meet the performance requirements of your workload and is not
     *         over provisioned. For optimized resources, Compute Optimizer might recommend a new generation instance
     *         type.
     *         </p>
     *         </li>
     * @see Finding
     */
    public final Finding finding() {
        return Finding.fromValue(finding);
    }

    /**
     * <p>
     * The finding classification of the instance.
     * </p>
     * <p>
     * Findings for instances include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
     * specification of your instance, such as CPU, memory, or network, does not meet the performance requirements of
     * your workload. Under-provisioned instances may lead to poor application performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one specification
     * of your instance, such as CPU, memory, or network, can be sized down while still meeting the performance
     * requirements of your workload, and no specification is under-provisioned. Over-provisioned instances may lead to
     * unnecessary infrastructure cost.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your instance,
     * such as CPU, memory, and network, meet the performance requirements of your workload and is not over provisioned.
     * For optimized resources, Compute Optimizer might recommend a new generation instance type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link Finding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification of the instance.</p>
     *         <p>
     *         Findings for instances include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
     *         specification of your instance, such as CPU, memory, or network, does not meet the performance
     *         requirements of your workload. Under-provisioned instances may lead to poor application performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
     *         specification of your instance, such as CPU, memory, or network, can be sized down while still meeting
     *         the performance requirements of your workload, and no specification is under-provisioned.
     *         Over-provisioned instances may lead to unnecessary infrastructure cost.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your
     *         instance, such as CPU, memory, and network, meet the performance requirements of your workload and is not
     *         over provisioned. For optimized resources, Compute Optimizer might recommend a new generation instance
     *         type.
     *         </p>
     *         </li>
     * @see Finding
     */
    public final String findingAsString() {
        return finding;
    }

    /**
     * <p>
     * The reason for the finding classification of the instance.
     * </p>
     * <p>
     * Finding reason codes for instances include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still meeting
     * the performance requirements of your workload. This is identified by analyzing the <code>CPUUtilization</code>
     * metric of the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
     * requirements of your workload and there is an alternative instance type that provides better CPU performance.
     * This is identified by analyzing the <code>CPUUtilization</code> metric of the current instance during the
     * look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while still
     * meeting the performance requirements of your workload. This is identified by analyzing the memory utilization
     * metric of the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the performance
     * requirements of your workload and there is an alternative instance type that provides better memory performance.
     * This is identified by analyzing the memory utilization metric of the current instance during the look-back
     * period.
     * </p>
     * <note>
     * <p>
     * Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on them. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory utilization
     * with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux instances, Compute
     * Optimizer analyses the <code>mem_used_percent</code> metric in the <code>CWAgent</code> namespace, or the legacy
     * <code>MemoryUtilization</code> metric in the <code>System/Linux</code> namespace. On Windows instances, Compute
     * Optimizer analyses the <code>Memory % Committed Bytes In Use</code> metric in the <code>CWAgent</code> namespace.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be sized
     * down while still meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current
     * instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't meet
     * the performance requirements of your workload and there is an alternative instance type that provides better EBS
     * throughput performance. This is identified by analyzing the <code>VolumeReadBytes</code> and
     * <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current instance during the look-back
     * period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down while
     * still meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance
     * during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
     * performance requirements of your workload and there is an alternative instance type that provides better EBS IOPS
     * performance. This is identified by analyzing the <code>VolumeReadOps</code> and <code>VolumeWriteOps</code>
     * metric of EBS volumes attached to the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration can be
     * sized down while still meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration doesn't
     * meet the performance requirements of your workload and there is an alternative instance type that provides better
     * network bandwidth performance. This is identified by analyzing the <code>NetworkIn</code> and
     * <code>NetworkOut</code> metrics of the current instance during the look-back period. This finding reason happens
     * when the <code>NetworkIn</code> or <code>NetworkOut</code> performance of an instance is impacted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second) configuration
     * can be sized down while still meeting the performance requirements of your workload. This is identified by
     * analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance
     * during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second) configuration
     * doesn't meet the performance requirements of your workload and there is an alternative instance type that
     * provides better network PPS performance. This is identified by analyzing the <code>NetworkPacketsIn</code> and
     * <code>NetworkPacketsIn</code> metrics of the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down while
     * still meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance during the look-back
     * period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
     * performance requirements of your workload and there is an alternative instance type that provides better disk
     * IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and <code>DiskWriteOps</code>
     * metrics of the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be sized
     * down while still meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the look-back
     * period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration doesn't meet
     * the performance requirements of your workload and there is an alternative instance type that provides better disk
     * throughput performance. This is identified by analyzing the <code>DiskReadBytes</code> and
     * <code>DiskWriteBytes</code> metrics of the current instance during the look-back period.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * For more information about instance metrics, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List the
     * available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>. For
     * more information about EBS volume metrics, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch metrics
     * for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification of the instance.</p>
     *         <p>
     *         Finding reason codes for instances include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still
     *         meeting the performance requirements of your workload. This is identified by analyzing the
     *         <code>CPUUtilization</code> metric of the current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
     *         requirements of your workload and there is an alternative instance type that provides better CPU
     *         performance. This is identified by analyzing the <code>CPUUtilization</code> metric of the current
     *         instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while
     *         still meeting the performance requirements of your workload. This is identified by analyzing the memory
     *         utilization metric of the current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
     *         performance requirements of your workload and there is an alternative instance type that provides better
     *         memory performance. This is identified by analyzing the memory utilization metric of the current instance
     *         during the look-back period.
     *         </p>
     *         <note>
     *         <p>
     *         Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on
     *         them. For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory
     *         utilization with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux
     *         instances, Compute Optimizer analyses the <code>mem_used_percent</code> metric in the
     *         <code>CWAgent</code> namespace, or the legacy <code>MemoryUtilization</code> metric in the
     *         <code>System/Linux</code> namespace. On Windows instances, Compute Optimizer analyses the
     *         <code>Memory % Committed Bytes In Use</code> metric in the <code>CWAgent</code> namespace.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be
     *         sized down while still meeting the performance requirements of your workload. This is identified by
     *         analyzing the <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes
     *         attached to the current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't
     *         meet the performance requirements of your workload and there is an alternative instance type that
     *         provides better EBS throughput performance. This is identified by analyzing the
     *         <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the
     *         current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down
     *         while still meeting the performance requirements of your workload. This is identified by analyzing the
     *         <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the current
     *         instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
     *         performance requirements of your workload and there is an alternative instance type that provides better
     *         EBS IOPS performance. This is identified by analyzing the <code>VolumeReadOps</code> and
     *         <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance during the look-back
     *         period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration
     *         can be sized down while still meeting the performance requirements of your workload. This is identified
     *         by analyzing the <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance
     *         during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration
     *         doesn't meet the performance requirements of your workload and there is an alternative instance type that
     *         provides better network bandwidth performance. This is identified by analyzing the <code>NetworkIn</code>
     *         and <code>NetworkOut</code> metrics of the current instance during the look-back period. This finding
     *         reason happens when the <code>NetworkIn</code> or <code>NetworkOut</code> performance of an instance is
     *         impacted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
     *         configuration can be sized down while still meeting the performance requirements of your workload. This
     *         is identified by analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of
     *         the current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
     *         configuration doesn't meet the performance requirements of your workload and there is an alternative
     *         instance type that provides better network PPS performance. This is identified by analyzing the
     *         <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance during
     *         the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down
     *         while still meeting the performance requirements of your workload. This is identified by analyzing the
     *         <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance during the
     *         look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
     *         performance requirements of your workload and there is an alternative instance type that provides better
     *         disk IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and
     *         <code>DiskWriteOps</code> metrics of the current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be
     *         sized down while still meeting the performance requirements of your workload. This is identified by
     *         analyzing the <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance
     *         during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration
     *         doesn't meet the performance requirements of your workload and there is an alternative instance type that
     *         provides better disk throughput performance. This is identified by analyzing the
     *         <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the
     *         look-back period.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         For more information about instance metrics, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List the
     *         available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User
     *         Guide</i>. For more information about EBS volume metrics, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch
     *         metrics for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *         </p>
     */
    public final List<InstanceRecommendationFindingReasonCode> findingReasonCodes() {
        return InstanceRecommendationFindingReasonCodesCopier.copyStringToEnum(findingReasonCodes);
    }

    /**
     * For responses, this returns true if the service returned a value for the FindingReasonCodes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFindingReasonCodes() {
        return findingReasonCodes != null && !(findingReasonCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The reason for the finding classification of the instance.
     * </p>
     * <p>
     * Finding reason codes for instances include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still meeting
     * the performance requirements of your workload. This is identified by analyzing the <code>CPUUtilization</code>
     * metric of the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
     * requirements of your workload and there is an alternative instance type that provides better CPU performance.
     * This is identified by analyzing the <code>CPUUtilization</code> metric of the current instance during the
     * look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while still
     * meeting the performance requirements of your workload. This is identified by analyzing the memory utilization
     * metric of the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the performance
     * requirements of your workload and there is an alternative instance type that provides better memory performance.
     * This is identified by analyzing the memory utilization metric of the current instance during the look-back
     * period.
     * </p>
     * <note>
     * <p>
     * Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on them. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory utilization
     * with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux instances, Compute
     * Optimizer analyses the <code>mem_used_percent</code> metric in the <code>CWAgent</code> namespace, or the legacy
     * <code>MemoryUtilization</code> metric in the <code>System/Linux</code> namespace. On Windows instances, Compute
     * Optimizer analyses the <code>Memory % Committed Bytes In Use</code> metric in the <code>CWAgent</code> namespace.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be sized
     * down while still meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current
     * instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't meet
     * the performance requirements of your workload and there is an alternative instance type that provides better EBS
     * throughput performance. This is identified by analyzing the <code>VolumeReadBytes</code> and
     * <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current instance during the look-back
     * period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down while
     * still meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance
     * during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
     * performance requirements of your workload and there is an alternative instance type that provides better EBS IOPS
     * performance. This is identified by analyzing the <code>VolumeReadOps</code> and <code>VolumeWriteOps</code>
     * metric of EBS volumes attached to the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration can be
     * sized down while still meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration doesn't
     * meet the performance requirements of your workload and there is an alternative instance type that provides better
     * network bandwidth performance. This is identified by analyzing the <code>NetworkIn</code> and
     * <code>NetworkOut</code> metrics of the current instance during the look-back period. This finding reason happens
     * when the <code>NetworkIn</code> or <code>NetworkOut</code> performance of an instance is impacted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second) configuration
     * can be sized down while still meeting the performance requirements of your workload. This is identified by
     * analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance
     * during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second) configuration
     * doesn't meet the performance requirements of your workload and there is an alternative instance type that
     * provides better network PPS performance. This is identified by analyzing the <code>NetworkPacketsIn</code> and
     * <code>NetworkPacketsIn</code> metrics of the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down while
     * still meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance during the look-back
     * period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
     * performance requirements of your workload and there is an alternative instance type that provides better disk
     * IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and <code>DiskWriteOps</code>
     * metrics of the current instance during the look-back period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be sized
     * down while still meeting the performance requirements of your workload. This is identified by analyzing the
     * <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the look-back
     * period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration doesn't meet
     * the performance requirements of your workload and there is an alternative instance type that provides better disk
     * throughput performance. This is identified by analyzing the <code>DiskReadBytes</code> and
     * <code>DiskWriteBytes</code> metrics of the current instance during the look-back period.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * For more information about instance metrics, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List the
     * available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>. For
     * more information about EBS volume metrics, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch metrics
     * for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindingReasonCodes} method.
     * </p>
     * 
     * @return The reason for the finding classification of the instance.</p>
     *         <p>
     *         Finding reason codes for instances include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still
     *         meeting the performance requirements of your workload. This is identified by analyzing the
     *         <code>CPUUtilization</code> metric of the current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
     *         requirements of your workload and there is an alternative instance type that provides better CPU
     *         performance. This is identified by analyzing the <code>CPUUtilization</code> metric of the current
     *         instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while
     *         still meeting the performance requirements of your workload. This is identified by analyzing the memory
     *         utilization metric of the current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
     *         performance requirements of your workload and there is an alternative instance type that provides better
     *         memory performance. This is identified by analyzing the memory utilization metric of the current instance
     *         during the look-back period.
     *         </p>
     *         <note>
     *         <p>
     *         Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on
     *         them. For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory
     *         utilization with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux
     *         instances, Compute Optimizer analyses the <code>mem_used_percent</code> metric in the
     *         <code>CWAgent</code> namespace, or the legacy <code>MemoryUtilization</code> metric in the
     *         <code>System/Linux</code> namespace. On Windows instances, Compute Optimizer analyses the
     *         <code>Memory % Committed Bytes In Use</code> metric in the <code>CWAgent</code> namespace.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be
     *         sized down while still meeting the performance requirements of your workload. This is identified by
     *         analyzing the <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes
     *         attached to the current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't
     *         meet the performance requirements of your workload and there is an alternative instance type that
     *         provides better EBS throughput performance. This is identified by analyzing the
     *         <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the
     *         current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down
     *         while still meeting the performance requirements of your workload. This is identified by analyzing the
     *         <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the current
     *         instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
     *         performance requirements of your workload and there is an alternative instance type that provides better
     *         EBS IOPS performance. This is identified by analyzing the <code>VolumeReadOps</code> and
     *         <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance during the look-back
     *         period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration
     *         can be sized down while still meeting the performance requirements of your workload. This is identified
     *         by analyzing the <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance
     *         during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration
     *         doesn't meet the performance requirements of your workload and there is an alternative instance type that
     *         provides better network bandwidth performance. This is identified by analyzing the <code>NetworkIn</code>
     *         and <code>NetworkOut</code> metrics of the current instance during the look-back period. This finding
     *         reason happens when the <code>NetworkIn</code> or <code>NetworkOut</code> performance of an instance is
     *         impacted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
     *         configuration can be sized down while still meeting the performance requirements of your workload. This
     *         is identified by analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of
     *         the current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
     *         configuration doesn't meet the performance requirements of your workload and there is an alternative
     *         instance type that provides better network PPS performance. This is identified by analyzing the
     *         <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance during
     *         the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down
     *         while still meeting the performance requirements of your workload. This is identified by analyzing the
     *         <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance during the
     *         look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
     *         performance requirements of your workload and there is an alternative instance type that provides better
     *         disk IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and
     *         <code>DiskWriteOps</code> metrics of the current instance during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be
     *         sized down while still meeting the performance requirements of your workload. This is identified by
     *         analyzing the <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance
     *         during the look-back period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration
     *         doesn't meet the performance requirements of your workload and there is an alternative instance type that
     *         provides better disk throughput performance. This is identified by analyzing the
     *         <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the
     *         look-back period.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         For more information about instance metrics, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List the
     *         available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User
     *         Guide</i>. For more information about EBS volume metrics, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch
     *         metrics for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     *         </p>
     */
    public final List<String> findingReasonCodesAsStrings() {
        return findingReasonCodes;
    }

    /**
     * For responses, this returns true if the service returned a value for the UtilizationMetrics property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUtilizationMetrics() {
        return utilizationMetrics != null && !(utilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the utilization metrics of the instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUtilizationMetrics} method.
     * </p>
     * 
     * @return An array of objects that describe the utilization metrics of the instance.
     */
    public final List<UtilizationMetric> utilizationMetrics() {
        return utilizationMetrics;
    }

    /**
     * <p>
     * The number of days for which utilization metrics were analyzed for the instance.
     * </p>
     * 
     * @return The number of days for which utilization metrics were analyzed for the instance.
     */
    public final Double lookBackPeriodInDays() {
        return lookBackPeriodInDays;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecommendationOptions property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRecommendationOptions() {
        return recommendationOptions != null && !(recommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the recommendation options for the instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendationOptions} method.
     * </p>
     * 
     * @return An array of objects that describe the recommendation options for the instance.
     */
    public final List<InstanceRecommendationOption> recommendationOptions() {
        return recommendationOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecommendationSources property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRecommendationSources() {
        return recommendationSources != null && !(recommendationSources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the source resource of the recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendationSources} method.
     * </p>
     * 
     * @return An array of objects that describe the source resource of the recommendation.
     */
    public final List<RecommendationSource> recommendationSources() {
        return recommendationSources;
    }

    /**
     * <p>
     * The timestamp of when the instance recommendation was last generated.
     * </p>
     * 
     * @return The timestamp of when the instance recommendation was last generated.
     */
    public final Instant lastRefreshTimestamp() {
        return lastRefreshTimestamp;
    }

    /**
     * <p>
     * The risk of the current instance not meeting the performance needs of its workloads. The higher the risk, the
     * more likely the current instance cannot meet the performance requirements of its workload.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentPerformanceRisk} will return {@link CurrentPerformanceRisk#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #currentPerformanceRiskAsString}.
     * </p>
     * 
     * @return The risk of the current instance not meeting the performance needs of its workloads. The higher the risk,
     *         the more likely the current instance cannot meet the performance requirements of its workload.
     * @see CurrentPerformanceRisk
     */
    public final CurrentPerformanceRisk currentPerformanceRisk() {
        return CurrentPerformanceRisk.fromValue(currentPerformanceRisk);
    }

    /**
     * <p>
     * The risk of the current instance not meeting the performance needs of its workloads. The higher the risk, the
     * more likely the current instance cannot meet the performance requirements of its workload.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentPerformanceRisk} will return {@link CurrentPerformanceRisk#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #currentPerformanceRiskAsString}.
     * </p>
     * 
     * @return The risk of the current instance not meeting the performance needs of its workloads. The higher the risk,
     *         the more likely the current instance cannot meet the performance requirements of its workload.
     * @see CurrentPerformanceRisk
     */
    public final String currentPerformanceRiskAsString() {
        return currentPerformanceRisk;
    }

    /**
     * <p>
     * An object that describes the effective recommendation preferences for the instance.
     * </p>
     * 
     * @return An object that describes the effective recommendation preferences for the instance.
     */
    public final EffectiveRecommendationPreferences effectiveRecommendationPreferences() {
        return effectiveRecommendationPreferences;
    }

    /**
     * <p>
     * The applications that might be running on the instance as inferred by Compute Optimizer.
     * </p>
     * <p>
     * Compute Optimizer can infer if one of the following applications might be running on the instance:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memcached</code> - Infers that Memcached might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NGINX</code> - Infers that NGINX might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Redis</code> - Infers that Redis might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Kafka</code> - Infers that Kafka might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInferredWorkloadTypes} method.
     * </p>
     * 
     * @return The applications that might be running on the instance as inferred by Compute Optimizer.</p>
     *         <p>
     *         Compute Optimizer can infer if one of the following applications might be running on the instance:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memcached</code> - Infers that Memcached might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NGINX</code> - Infers that NGINX might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Redis</code> - Infers that Redis might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Kafka</code> - Infers that Kafka might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     *         </p>
     *         </li>
     */
    public final List<InferredWorkloadType> inferredWorkloadTypes() {
        return InferredWorkloadTypesCopier.copyStringToEnum(inferredWorkloadTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the InferredWorkloadTypes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInferredWorkloadTypes() {
        return inferredWorkloadTypes != null && !(inferredWorkloadTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The applications that might be running on the instance as inferred by Compute Optimizer.
     * </p>
     * <p>
     * Compute Optimizer can infer if one of the following applications might be running on the instance:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memcached</code> - Infers that Memcached might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NGINX</code> - Infers that NGINX might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Redis</code> - Infers that Redis might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Kafka</code> - Infers that Kafka might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInferredWorkloadTypes} method.
     * </p>
     * 
     * @return The applications that might be running on the instance as inferred by Compute Optimizer.</p>
     *         <p>
     *         Compute Optimizer can infer if one of the following applications might be running on the instance:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memcached</code> - Infers that Memcached might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NGINX</code> - Infers that NGINX might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Redis</code> - Infers that Redis might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Kafka</code> - Infers that Kafka might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     *         </p>
     *         </li>
     */
    public final List<String> inferredWorkloadTypesAsStrings() {
        return inferredWorkloadTypes;
    }

    /**
     * <p>
     * The state of the instance when the recommendation was generated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceState}
     * will return {@link InstanceState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceStateAsString}.
     * </p>
     * 
     * @return The state of the instance when the recommendation was generated.
     * @see InstanceState
     */
    public final InstanceState instanceState() {
        return InstanceState.fromValue(instanceState);
    }

    /**
     * <p>
     * The state of the instance when the recommendation was generated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceState}
     * will return {@link InstanceState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceStateAsString}.
     * </p>
     * 
     * @return The state of the instance when the recommendation was generated.
     * @see InstanceState
     */
    public final String instanceStateAsString() {
        return instanceState;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags assigned to your Amazon EC2 instance recommendations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags assigned to your Amazon EC2 instance recommendations.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceArn());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceName());
        hashCode = 31 * hashCode + Objects.hashCode(currentInstanceType());
        hashCode = 31 * hashCode + Objects.hashCode(findingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasFindingReasonCodes() ? findingReasonCodesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUtilizationMetrics() ? utilizationMetrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lookBackPeriodInDays());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendationOptions() ? recommendationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendationSources() ? recommendationSources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(currentPerformanceRiskAsString());
        hashCode = 31 * hashCode + Objects.hashCode(effectiveRecommendationPreferences());
        hashCode = 31 * hashCode + Objects.hashCode(hasInferredWorkloadTypes() ? inferredWorkloadTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(instanceStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceRecommendation)) {
            return false;
        }
        InstanceRecommendation other = (InstanceRecommendation) obj;
        return Objects.equals(instanceArn(), other.instanceArn()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(instanceName(), other.instanceName())
                && Objects.equals(currentInstanceType(), other.currentInstanceType())
                && Objects.equals(findingAsString(), other.findingAsString())
                && hasFindingReasonCodes() == other.hasFindingReasonCodes()
                && Objects.equals(findingReasonCodesAsStrings(), other.findingReasonCodesAsStrings())
                && hasUtilizationMetrics() == other.hasUtilizationMetrics()
                && Objects.equals(utilizationMetrics(), other.utilizationMetrics())
                && Objects.equals(lookBackPeriodInDays(), other.lookBackPeriodInDays())
                && hasRecommendationOptions() == other.hasRecommendationOptions()
                && Objects.equals(recommendationOptions(), other.recommendationOptions())
                && hasRecommendationSources() == other.hasRecommendationSources()
                && Objects.equals(recommendationSources(), other.recommendationSources())
                && Objects.equals(lastRefreshTimestamp(), other.lastRefreshTimestamp())
                && Objects.equals(currentPerformanceRiskAsString(), other.currentPerformanceRiskAsString())
                && Objects.equals(effectiveRecommendationPreferences(), other.effectiveRecommendationPreferences())
                && hasInferredWorkloadTypes() == other.hasInferredWorkloadTypes()
                && Objects.equals(inferredWorkloadTypesAsStrings(), other.inferredWorkloadTypesAsStrings())
                && Objects.equals(instanceStateAsString(), other.instanceStateAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceRecommendation").add("InstanceArn", instanceArn()).add("AccountId", accountId())
                .add("InstanceName", instanceName()).add("CurrentInstanceType", currentInstanceType())
                .add("Finding", findingAsString())
                .add("FindingReasonCodes", hasFindingReasonCodes() ? findingReasonCodesAsStrings() : null)
                .add("UtilizationMetrics", hasUtilizationMetrics() ? utilizationMetrics() : null)
                .add("LookBackPeriodInDays", lookBackPeriodInDays())
                .add("RecommendationOptions", hasRecommendationOptions() ? recommendationOptions() : null)
                .add("RecommendationSources", hasRecommendationSources() ? recommendationSources() : null)
                .add("LastRefreshTimestamp", lastRefreshTimestamp())
                .add("CurrentPerformanceRisk", currentPerformanceRiskAsString())
                .add("EffectiveRecommendationPreferences", effectiveRecommendationPreferences())
                .add("InferredWorkloadTypes", hasInferredWorkloadTypes() ? inferredWorkloadTypesAsStrings() : null)
                .add("InstanceState", instanceStateAsString()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "instanceArn":
            return Optional.ofNullable(clazz.cast(instanceArn()));
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "instanceName":
            return Optional.ofNullable(clazz.cast(instanceName()));
        case "currentInstanceType":
            return Optional.ofNullable(clazz.cast(currentInstanceType()));
        case "finding":
            return Optional.ofNullable(clazz.cast(findingAsString()));
        case "findingReasonCodes":
            return Optional.ofNullable(clazz.cast(findingReasonCodesAsStrings()));
        case "utilizationMetrics":
            return Optional.ofNullable(clazz.cast(utilizationMetrics()));
        case "lookBackPeriodInDays":
            return Optional.ofNullable(clazz.cast(lookBackPeriodInDays()));
        case "recommendationOptions":
            return Optional.ofNullable(clazz.cast(recommendationOptions()));
        case "recommendationSources":
            return Optional.ofNullable(clazz.cast(recommendationSources()));
        case "lastRefreshTimestamp":
            return Optional.ofNullable(clazz.cast(lastRefreshTimestamp()));
        case "currentPerformanceRisk":
            return Optional.ofNullable(clazz.cast(currentPerformanceRiskAsString()));
        case "effectiveRecommendationPreferences":
            return Optional.ofNullable(clazz.cast(effectiveRecommendationPreferences()));
        case "inferredWorkloadTypes":
            return Optional.ofNullable(clazz.cast(inferredWorkloadTypesAsStrings()));
        case "instanceState":
            return Optional.ofNullable(clazz.cast(instanceStateAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceRecommendation, T> g) {
        return obj -> g.apply((InstanceRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceRecommendation> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the current instance.
         * </p>
         * 
         * @param instanceArn
         *        The Amazon Resource Name (ARN) of the current instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceArn(String instanceArn);

        /**
         * <p>
         * The Amazon Web Services account ID of the instance.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The name of the current instance.
         * </p>
         * 
         * @param instanceName
         *        The name of the current instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceName(String instanceName);

        /**
         * <p>
         * The instance type of the current instance.
         * </p>
         * 
         * @param currentInstanceType
         *        The instance type of the current instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentInstanceType(String currentInstanceType);

        /**
         * <p>
         * The finding classification of the instance.
         * </p>
         * <p>
         * Findings for instances include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
         * specification of your instance, such as CPU, memory, or network, does not meet the performance requirements
         * of your workload. Under-provisioned instances may lead to poor application performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
         * specification of your instance, such as CPU, memory, or network, can be sized down while still meeting the
         * performance requirements of your workload, and no specification is under-provisioned. Over-provisioned
         * instances may lead to unnecessary infrastructure cost.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your instance,
         * such as CPU, memory, and network, meet the performance requirements of your workload and is not over
         * provisioned. For optimized resources, Compute Optimizer might recommend a new generation instance type.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification of the instance.</p>
         *        <p>
         *        Findings for instances include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
         *        specification of your instance, such as CPU, memory, or network, does not meet the performance
         *        requirements of your workload. Under-provisioned instances may lead to poor application performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
         *        specification of your instance, such as CPU, memory, or network, can be sized down while still meeting
         *        the performance requirements of your workload, and no specification is under-provisioned.
         *        Over-provisioned instances may lead to unnecessary infrastructure cost.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your
         *        instance, such as CPU, memory, and network, meet the performance requirements of your workload and is
         *        not over provisioned. For optimized resources, Compute Optimizer might recommend a new generation
         *        instance type.
         *        </p>
         *        </li>
         * @see Finding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Finding
         */
        Builder finding(String finding);

        /**
         * <p>
         * The finding classification of the instance.
         * </p>
         * <p>
         * Findings for instances include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
         * specification of your instance, such as CPU, memory, or network, does not meet the performance requirements
         * of your workload. Under-provisioned instances may lead to poor application performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
         * specification of your instance, such as CPU, memory, or network, can be sized down while still meeting the
         * performance requirements of your workload, and no specification is under-provisioned. Over-provisioned
         * instances may lead to unnecessary infrastructure cost.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your instance,
         * such as CPU, memory, and network, meet the performance requirements of your workload and is not over
         * provisioned. For optimized resources, Compute Optimizer might recommend a new generation instance type.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification of the instance.</p>
         *        <p>
         *        Findings for instances include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
         *        specification of your instance, such as CPU, memory, or network, does not meet the performance
         *        requirements of your workload. Under-provisioned instances may lead to poor application performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
         *        specification of your instance, such as CPU, memory, or network, can be sized down while still meeting
         *        the performance requirements of your workload, and no specification is under-provisioned.
         *        Over-provisioned instances may lead to unnecessary infrastructure cost.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your
         *        instance, such as CPU, memory, and network, meet the performance requirements of your workload and is
         *        not over provisioned. For optimized resources, Compute Optimizer might recommend a new generation
         *        instance type.
         *        </p>
         *        </li>
         * @see Finding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Finding
         */
        Builder finding(Finding finding);

        /**
         * <p>
         * The reason for the finding classification of the instance.
         * </p>
         * <p>
         * Finding reason codes for instances include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>CPUUtilization</code> metric of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
         * requirements of your workload and there is an alternative instance type that provides better CPU performance.
         * This is identified by analyzing the <code>CPUUtilization</code> metric of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while
         * still meeting the performance requirements of your workload. This is identified by analyzing the memory
         * utilization metric of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better
         * memory performance. This is identified by analyzing the memory utilization metric of the current instance
         * during the look-back period.
         * </p>
         * <note>
         * <p>
         * Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on them.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory
         * utilization with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux
         * instances, Compute Optimizer analyses the <code>mem_used_percent</code> metric in the <code>CWAgent</code>
         * namespace, or the legacy <code>MemoryUtilization</code> metric in the <code>System/Linux</code> namespace. On
         * Windows instances, Compute Optimizer analyses the <code>Memory % Committed Bytes In Use</code> metric in the
         * <code>CWAgent</code> namespace.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be sized
         * down while still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current
         * instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better EBS throughput performance. This is identified by analyzing the <code>VolumeReadBytes</code> and
         * <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down while
         * still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the current
         * instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better EBS
         * IOPS performance. This is identified by analyzing the <code>VolumeReadOps</code> and
         * <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration can be
         * sized down while still meeting the performance requirements of your workload. This is identified by analyzing
         * the <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration
         * doesn't meet the performance requirements of your workload and there is an alternative instance type that
         * provides better network bandwidth performance. This is identified by analyzing the <code>NetworkIn</code> and
         * <code>NetworkOut</code> metrics of the current instance during the look-back period. This finding reason
         * happens when the <code>NetworkIn</code> or <code>NetworkOut</code> performance of an instance is impacted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration can be sized down while still meeting the performance requirements of your workload. This is
         * identified by analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the
         * current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration doesn't meet the performance requirements of your workload and there is an alternative instance
         * type that provides better network PPS performance. This is identified by analyzing the
         * <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down
         * while still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better disk
         * IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and <code>DiskWriteOps</code>
         * metrics of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be
         * sized down while still meeting the performance requirements of your workload. This is identified by analyzing
         * the <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better disk throughput performance. This is identified by analyzing the <code>DiskReadBytes</code> and
         * <code>DiskWriteBytes</code> metrics of the current instance during the look-back period.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * For more information about instance metrics, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List the
         * available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * For more information about EBS volume metrics, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch
         * metrics for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * </note>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the instance.</p>
         *        <p>
         *        Finding reason codes for instances include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while
         *        still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>CPUUtilization</code> metric of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better CPU performance. This is identified by analyzing the <code>CPUUtilization</code> metric of the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down
         *        while still meeting the performance requirements of your workload. This is identified by analyzing the
         *        memory utilization metric of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better memory performance. This is identified by analyzing the memory utilization metric of the
         *        current instance during the look-back period.
         *        </p>
         *        <note>
         *        <p>
         *        Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on
         *        them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory
         *        utilization with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux
         *        instances, Compute Optimizer analyses the <code>mem_used_percent</code> metric in the
         *        <code>CWAgent</code> namespace, or the legacy <code>MemoryUtilization</code> metric in the
         *        <code>System/Linux</code> namespace. On Windows instances, Compute Optimizer analyses the
         *        <code>Memory % Committed Bytes In Use</code> metric in the <code>CWAgent</code> namespace.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes
         *        attached to the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better EBS throughput performance. This is identified by analyzing the
         *        <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down
         *        while still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better EBS IOPS performance. This is identified by analyzing the <code>VolumeReadOps</code> and
         *        <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance during the
         *        look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration
         *        can be sized down while still meeting the performance requirements of your workload. This is
         *        identified by analyzing the <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current
         *        instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network bandwidth performance. This is identified by analyzing the
         *        <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance during the
         *        look-back period. This finding reason happens when the <code>NetworkIn</code> or
         *        <code>NetworkOut</code> performance of an instance is impacted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration can be sized down while still meeting the performance requirements of your workload.
         *        This is identified by analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code>
         *        metrics of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network PPS performance. This is identified by analyzing the
         *        <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance during
         *        the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized
         *        down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance
         *        during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet
         *        the performance requirements of your workload and there is an alternative instance type that provides
         *        better disk IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and
         *        <code>DiskWriteOps</code> metrics of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current
         *        instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better disk throughput performance. This is identified by analyzing the
         *        <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the
         *        look-back period.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        For more information about instance metrics, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List
         *        the available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User
         *        Guide</i>. For more information about EBS volume metrics, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch
         *        metrics for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodesWithStrings(Collection<String> findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of the instance.
         * </p>
         * <p>
         * Finding reason codes for instances include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>CPUUtilization</code> metric of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
         * requirements of your workload and there is an alternative instance type that provides better CPU performance.
         * This is identified by analyzing the <code>CPUUtilization</code> metric of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while
         * still meeting the performance requirements of your workload. This is identified by analyzing the memory
         * utilization metric of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better
         * memory performance. This is identified by analyzing the memory utilization metric of the current instance
         * during the look-back period.
         * </p>
         * <note>
         * <p>
         * Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on them.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory
         * utilization with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux
         * instances, Compute Optimizer analyses the <code>mem_used_percent</code> metric in the <code>CWAgent</code>
         * namespace, or the legacy <code>MemoryUtilization</code> metric in the <code>System/Linux</code> namespace. On
         * Windows instances, Compute Optimizer analyses the <code>Memory % Committed Bytes In Use</code> metric in the
         * <code>CWAgent</code> namespace.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be sized
         * down while still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current
         * instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better EBS throughput performance. This is identified by analyzing the <code>VolumeReadBytes</code> and
         * <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down while
         * still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the current
         * instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better EBS
         * IOPS performance. This is identified by analyzing the <code>VolumeReadOps</code> and
         * <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration can be
         * sized down while still meeting the performance requirements of your workload. This is identified by analyzing
         * the <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration
         * doesn't meet the performance requirements of your workload and there is an alternative instance type that
         * provides better network bandwidth performance. This is identified by analyzing the <code>NetworkIn</code> and
         * <code>NetworkOut</code> metrics of the current instance during the look-back period. This finding reason
         * happens when the <code>NetworkIn</code> or <code>NetworkOut</code> performance of an instance is impacted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration can be sized down while still meeting the performance requirements of your workload. This is
         * identified by analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the
         * current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration doesn't meet the performance requirements of your workload and there is an alternative instance
         * type that provides better network PPS performance. This is identified by analyzing the
         * <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down
         * while still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better disk
         * IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and <code>DiskWriteOps</code>
         * metrics of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be
         * sized down while still meeting the performance requirements of your workload. This is identified by analyzing
         * the <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better disk throughput performance. This is identified by analyzing the <code>DiskReadBytes</code> and
         * <code>DiskWriteBytes</code> metrics of the current instance during the look-back period.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * For more information about instance metrics, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List the
         * available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * For more information about EBS volume metrics, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch
         * metrics for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * </note>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the instance.</p>
         *        <p>
         *        Finding reason codes for instances include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while
         *        still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>CPUUtilization</code> metric of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better CPU performance. This is identified by analyzing the <code>CPUUtilization</code> metric of the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down
         *        while still meeting the performance requirements of your workload. This is identified by analyzing the
         *        memory utilization metric of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better memory performance. This is identified by analyzing the memory utilization metric of the
         *        current instance during the look-back period.
         *        </p>
         *        <note>
         *        <p>
         *        Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on
         *        them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory
         *        utilization with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux
         *        instances, Compute Optimizer analyses the <code>mem_used_percent</code> metric in the
         *        <code>CWAgent</code> namespace, or the legacy <code>MemoryUtilization</code> metric in the
         *        <code>System/Linux</code> namespace. On Windows instances, Compute Optimizer analyses the
         *        <code>Memory % Committed Bytes In Use</code> metric in the <code>CWAgent</code> namespace.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes
         *        attached to the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better EBS throughput performance. This is identified by analyzing the
         *        <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down
         *        while still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better EBS IOPS performance. This is identified by analyzing the <code>VolumeReadOps</code> and
         *        <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance during the
         *        look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration
         *        can be sized down while still meeting the performance requirements of your workload. This is
         *        identified by analyzing the <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current
         *        instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network bandwidth performance. This is identified by analyzing the
         *        <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance during the
         *        look-back period. This finding reason happens when the <code>NetworkIn</code> or
         *        <code>NetworkOut</code> performance of an instance is impacted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration can be sized down while still meeting the performance requirements of your workload.
         *        This is identified by analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code>
         *        metrics of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network PPS performance. This is identified by analyzing the
         *        <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance during
         *        the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized
         *        down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance
         *        during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet
         *        the performance requirements of your workload and there is an alternative instance type that provides
         *        better disk IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and
         *        <code>DiskWriteOps</code> metrics of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current
         *        instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better disk throughput performance. This is identified by analyzing the
         *        <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the
         *        look-back period.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        For more information about instance metrics, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List
         *        the available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User
         *        Guide</i>. For more information about EBS volume metrics, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch
         *        metrics for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodesWithStrings(String... findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of the instance.
         * </p>
         * <p>
         * Finding reason codes for instances include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>CPUUtilization</code> metric of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
         * requirements of your workload and there is an alternative instance type that provides better CPU performance.
         * This is identified by analyzing the <code>CPUUtilization</code> metric of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while
         * still meeting the performance requirements of your workload. This is identified by analyzing the memory
         * utilization metric of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better
         * memory performance. This is identified by analyzing the memory utilization metric of the current instance
         * during the look-back period.
         * </p>
         * <note>
         * <p>
         * Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on them.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory
         * utilization with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux
         * instances, Compute Optimizer analyses the <code>mem_used_percent</code> metric in the <code>CWAgent</code>
         * namespace, or the legacy <code>MemoryUtilization</code> metric in the <code>System/Linux</code> namespace. On
         * Windows instances, Compute Optimizer analyses the <code>Memory % Committed Bytes In Use</code> metric in the
         * <code>CWAgent</code> namespace.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be sized
         * down while still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current
         * instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better EBS throughput performance. This is identified by analyzing the <code>VolumeReadBytes</code> and
         * <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down while
         * still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the current
         * instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better EBS
         * IOPS performance. This is identified by analyzing the <code>VolumeReadOps</code> and
         * <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration can be
         * sized down while still meeting the performance requirements of your workload. This is identified by analyzing
         * the <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration
         * doesn't meet the performance requirements of your workload and there is an alternative instance type that
         * provides better network bandwidth performance. This is identified by analyzing the <code>NetworkIn</code> and
         * <code>NetworkOut</code> metrics of the current instance during the look-back period. This finding reason
         * happens when the <code>NetworkIn</code> or <code>NetworkOut</code> performance of an instance is impacted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration can be sized down while still meeting the performance requirements of your workload. This is
         * identified by analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the
         * current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration doesn't meet the performance requirements of your workload and there is an alternative instance
         * type that provides better network PPS performance. This is identified by analyzing the
         * <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down
         * while still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better disk
         * IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and <code>DiskWriteOps</code>
         * metrics of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be
         * sized down while still meeting the performance requirements of your workload. This is identified by analyzing
         * the <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better disk throughput performance. This is identified by analyzing the <code>DiskReadBytes</code> and
         * <code>DiskWriteBytes</code> metrics of the current instance during the look-back period.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * For more information about instance metrics, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List the
         * available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * For more information about EBS volume metrics, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch
         * metrics for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * </note>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the instance.</p>
         *        <p>
         *        Finding reason codes for instances include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while
         *        still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>CPUUtilization</code> metric of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better CPU performance. This is identified by analyzing the <code>CPUUtilization</code> metric of the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down
         *        while still meeting the performance requirements of your workload. This is identified by analyzing the
         *        memory utilization metric of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better memory performance. This is identified by analyzing the memory utilization metric of the
         *        current instance during the look-back period.
         *        </p>
         *        <note>
         *        <p>
         *        Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on
         *        them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory
         *        utilization with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux
         *        instances, Compute Optimizer analyses the <code>mem_used_percent</code> metric in the
         *        <code>CWAgent</code> namespace, or the legacy <code>MemoryUtilization</code> metric in the
         *        <code>System/Linux</code> namespace. On Windows instances, Compute Optimizer analyses the
         *        <code>Memory % Committed Bytes In Use</code> metric in the <code>CWAgent</code> namespace.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes
         *        attached to the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better EBS throughput performance. This is identified by analyzing the
         *        <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down
         *        while still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better EBS IOPS performance. This is identified by analyzing the <code>VolumeReadOps</code> and
         *        <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance during the
         *        look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration
         *        can be sized down while still meeting the performance requirements of your workload. This is
         *        identified by analyzing the <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current
         *        instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network bandwidth performance. This is identified by analyzing the
         *        <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance during the
         *        look-back period. This finding reason happens when the <code>NetworkIn</code> or
         *        <code>NetworkOut</code> performance of an instance is impacted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration can be sized down while still meeting the performance requirements of your workload.
         *        This is identified by analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code>
         *        metrics of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network PPS performance. This is identified by analyzing the
         *        <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance during
         *        the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized
         *        down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance
         *        during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet
         *        the performance requirements of your workload and there is an alternative instance type that provides
         *        better disk IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and
         *        <code>DiskWriteOps</code> metrics of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current
         *        instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better disk throughput performance. This is identified by analyzing the
         *        <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the
         *        look-back period.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        For more information about instance metrics, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List
         *        the available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User
         *        Guide</i>. For more information about EBS volume metrics, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch
         *        metrics for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodes(Collection<InstanceRecommendationFindingReasonCode> findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of the instance.
         * </p>
         * <p>
         * Finding reason codes for instances include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while still
         * meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>CPUUtilization</code> metric of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the performance
         * requirements of your workload and there is an alternative instance type that provides better CPU performance.
         * This is identified by analyzing the <code>CPUUtilization</code> metric of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down while
         * still meeting the performance requirements of your workload. This is identified by analyzing the memory
         * utilization metric of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better
         * memory performance. This is identified by analyzing the memory utilization metric of the current instance
         * during the look-back period.
         * </p>
         * <note>
         * <p>
         * Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on them.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory
         * utilization with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux
         * instances, Compute Optimizer analyses the <code>mem_used_percent</code> metric in the <code>CWAgent</code>
         * namespace, or the legacy <code>MemoryUtilization</code> metric in the <code>System/Linux</code> namespace. On
         * Windows instances, Compute Optimizer analyses the <code>Memory % Committed Bytes In Use</code> metric in the
         * <code>CWAgent</code> namespace.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can be sized
         * down while still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current
         * instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better EBS throughput performance. This is identified by analyzing the <code>VolumeReadBytes</code> and
         * <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down while
         * still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the current
         * instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better EBS
         * IOPS performance. This is identified by analyzing the <code>VolumeReadOps</code> and
         * <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration can be
         * sized down while still meeting the performance requirements of your workload. This is identified by analyzing
         * the <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth configuration
         * doesn't meet the performance requirements of your workload and there is an alternative instance type that
         * provides better network bandwidth performance. This is identified by analyzing the <code>NetworkIn</code> and
         * <code>NetworkOut</code> metrics of the current instance during the look-back period. This finding reason
         * happens when the <code>NetworkIn</code> or <code>NetworkOut</code> performance of an instance is impacted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration can be sized down while still meeting the performance requirements of your workload. This is
         * identified by analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the
         * current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         * configuration doesn't meet the performance requirements of your workload and there is an alternative instance
         * type that provides better network PPS performance. This is identified by analyzing the
         * <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized down
         * while still meeting the performance requirements of your workload. This is identified by analyzing the
         * <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance during the look-back
         * period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet the
         * performance requirements of your workload and there is an alternative instance type that provides better disk
         * IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and <code>DiskWriteOps</code>
         * metrics of the current instance during the look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can be
         * sized down while still meeting the performance requirements of your workload. This is identified by analyzing
         * the <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the
         * look-back period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration doesn't
         * meet the performance requirements of your workload and there is an alternative instance type that provides
         * better disk throughput performance. This is identified by analyzing the <code>DiskReadBytes</code> and
         * <code>DiskWriteBytes</code> metrics of the current instance during the look-back period.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * For more information about instance metrics, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List the
         * available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * For more information about EBS volume metrics, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch
         * metrics for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * </note>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the instance.</p>
         *        <p>
         *        Finding reason codes for instances include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>CPUOverprovisioned</code> </b> — The instance’s CPU configuration can be sized down while
         *        still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>CPUUtilization</code> metric of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUUnderprovisioned</code> </b> — The instance’s CPU configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better CPU performance. This is identified by analyzing the <code>CPUUtilization</code> metric of the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The instance’s memory configuration can be sized down
         *        while still meeting the performance requirements of your workload. This is identified by analyzing the
         *        memory utilization metric of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The instance’s memory configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better memory performance. This is identified by analyzing the memory utilization metric of the
         *        current instance during the look-back period.
         *        </p>
         *        <note>
         *        <p>
         *        Memory utilization is analyzed only for resources that have the unified CloudWatch agent installed on
         *        them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling memory
         *        utilization with the Amazon CloudWatch Agent</a> in the <i>Compute Optimizer User Guide</i>. On Linux
         *        instances, Compute Optimizer analyses the <code>mem_used_percent</code> metric in the
         *        <code>CWAgent</code> namespace, or the legacy <code>MemoryUtilization</code> metric in the
         *        <code>System/Linux</code> namespace. On Windows instances, Compute Optimizer analyses the
         *        <code>Memory % Committed Bytes In Use</code> metric in the <code>CWAgent</code> namespace.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputOverprovisioned</code> </b> — The instance’s EBS throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes
         *        attached to the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSThroughputUnderprovisioned</code> </b> — The instance’s EBS throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better EBS throughput performance. This is identified by analyzing the
         *        <code>VolumeReadBytes</code> and <code>VolumeWriteBytes</code> metrics of EBS volumes attached to the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSOverprovisioned</code> </b> — The instance’s EBS IOPS configuration can be sized down
         *        while still meeting the performance requirements of your workload. This is identified by analyzing the
         *        <code>VolumeReadOps</code> and <code>VolumeWriteOps</code> metric of EBS volumes attached to the
         *        current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>EBSIOPSUnderprovisioned</code> </b> — The instance’s EBS IOPS configuration doesn't meet the
         *        performance requirements of your workload and there is an alternative instance type that provides
         *        better EBS IOPS performance. This is identified by analyzing the <code>VolumeReadOps</code> and
         *        <code>VolumeWriteOps</code> metric of EBS volumes attached to the current instance during the
         *        look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthOverprovisioned</code> </b> — The instance’s network bandwidth configuration
         *        can be sized down while still meeting the performance requirements of your workload. This is
         *        identified by analyzing the <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current
         *        instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkBandwidthUnderprovisioned</code> </b> — The instance’s network bandwidth
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network bandwidth performance. This is identified by analyzing the
         *        <code>NetworkIn</code> and <code>NetworkOut</code> metrics of the current instance during the
         *        look-back period. This finding reason happens when the <code>NetworkIn</code> or
         *        <code>NetworkOut</code> performance of an instance is impacted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSOverprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration can be sized down while still meeting the performance requirements of your workload.
         *        This is identified by analyzing the <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code>
         *        metrics of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkPPSUnderprovisioned</code> </b> — The instance’s network PPS (packets per second)
         *        configuration doesn't meet the performance requirements of your workload and there is an alternative
         *        instance type that provides better network PPS performance. This is identified by analyzing the
         *        <code>NetworkPacketsIn</code> and <code>NetworkPacketsIn</code> metrics of the current instance during
         *        the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSOverprovisioned</code> </b> — The instance’s disk IOPS configuration can be sized
         *        down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>DiskReadOps</code> and <code>DiskWriteOps</code> metrics of the current instance
         *        during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskIOPSUnderprovisioned</code> </b> — The instance’s disk IOPS configuration doesn't meet
         *        the performance requirements of your workload and there is an alternative instance type that provides
         *        better disk IOPS performance. This is identified by analyzing the <code>DiskReadOps</code> and
         *        <code>DiskWriteOps</code> metrics of the current instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputOverprovisioned</code> </b> — The instance’s disk throughput configuration can
         *        be sized down while still meeting the performance requirements of your workload. This is identified by
         *        analyzing the <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current
         *        instance during the look-back period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>DiskThroughputUnderprovisioned</code> </b> — The instance’s disk throughput configuration
         *        doesn't meet the performance requirements of your workload and there is an alternative instance type
         *        that provides better disk throughput performance. This is identified by analyzing the
         *        <code>DiskReadBytes</code> and <code>DiskWriteBytes</code> metrics of the current instance during the
         *        look-back period.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        For more information about instance metrics, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/viewing_metrics_with_cloudwatch.html">List
         *        the available CloudWatch metrics for your instances</a> in the <i>Amazon Elastic Compute Cloud User
         *        Guide</i>. For more information about EBS volume metrics, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using_cloudwatch_ebs.html">Amazon CloudWatch
         *        metrics for Amazon EBS</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodes(InstanceRecommendationFindingReasonCode... findingReasonCodes);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the instance.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(Collection<UtilizationMetric> utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the instance.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(UtilizationMetric... utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the instance.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder#build()} is called
         * immediately and its result is passed to {@link #utilizationMetrics(List<UtilizationMetric>)}.
         * 
         * @param utilizationMetrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationMetrics(java.util.Collection<UtilizationMetric>)
         */
        Builder utilizationMetrics(Consumer<UtilizationMetric.Builder>... utilizationMetrics);

        /**
         * <p>
         * The number of days for which utilization metrics were analyzed for the instance.
         * </p>
         * 
         * @param lookBackPeriodInDays
         *        The number of days for which utilization metrics were analyzed for the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lookBackPeriodInDays(Double lookBackPeriodInDays);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the instance.
         * </p>
         * 
         * @param recommendationOptions
         *        An array of objects that describe the recommendation options for the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationOptions(Collection<InstanceRecommendationOption> recommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the instance.
         * </p>
         * 
         * @param recommendationOptions
         *        An array of objects that describe the recommendation options for the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationOptions(InstanceRecommendationOption... recommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the instance.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.InstanceRecommendationOption.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.InstanceRecommendationOption#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.InstanceRecommendationOption.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #recommendationOptions(List<InstanceRecommendationOption>)}.
         * 
         * @param recommendationOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.InstanceRecommendationOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recommendationOptions(java.util.Collection<InstanceRecommendationOption>)
         */
        Builder recommendationOptions(Consumer<InstanceRecommendationOption.Builder>... recommendationOptions);

        /**
         * <p>
         * An array of objects that describe the source resource of the recommendation.
         * </p>
         * 
         * @param recommendationSources
         *        An array of objects that describe the source resource of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationSources(Collection<RecommendationSource> recommendationSources);

        /**
         * <p>
         * An array of objects that describe the source resource of the recommendation.
         * </p>
         * 
         * @param recommendationSources
         *        An array of objects that describe the source resource of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationSources(RecommendationSource... recommendationSources);

        /**
         * <p>
         * An array of objects that describe the source resource of the recommendation.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RecommendationSource.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RecommendationSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.RecommendationSource.Builder#build()} is called
         * immediately and its result is passed to {@link #recommendationSources(List<RecommendationSource>)}.
         * 
         * @param recommendationSources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.RecommendationSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recommendationSources(java.util.Collection<RecommendationSource>)
         */
        Builder recommendationSources(Consumer<RecommendationSource.Builder>... recommendationSources);

        /**
         * <p>
         * The timestamp of when the instance recommendation was last generated.
         * </p>
         * 
         * @param lastRefreshTimestamp
         *        The timestamp of when the instance recommendation was last generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTimestamp(Instant lastRefreshTimestamp);

        /**
         * <p>
         * The risk of the current instance not meeting the performance needs of its workloads. The higher the risk, the
         * more likely the current instance cannot meet the performance requirements of its workload.
         * </p>
         * 
         * @param currentPerformanceRisk
         *        The risk of the current instance not meeting the performance needs of its workloads. The higher the
         *        risk, the more likely the current instance cannot meet the performance requirements of its workload.
         * @see CurrentPerformanceRisk
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CurrentPerformanceRisk
         */
        Builder currentPerformanceRisk(String currentPerformanceRisk);

        /**
         * <p>
         * The risk of the current instance not meeting the performance needs of its workloads. The higher the risk, the
         * more likely the current instance cannot meet the performance requirements of its workload.
         * </p>
         * 
         * @param currentPerformanceRisk
         *        The risk of the current instance not meeting the performance needs of its workloads. The higher the
         *        risk, the more likely the current instance cannot meet the performance requirements of its workload.
         * @see CurrentPerformanceRisk
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CurrentPerformanceRisk
         */
        Builder currentPerformanceRisk(CurrentPerformanceRisk currentPerformanceRisk);

        /**
         * <p>
         * An object that describes the effective recommendation preferences for the instance.
         * </p>
         * 
         * @param effectiveRecommendationPreferences
         *        An object that describes the effective recommendation preferences for the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder effectiveRecommendationPreferences(EffectiveRecommendationPreferences effectiveRecommendationPreferences);

        /**
         * <p>
         * An object that describes the effective recommendation preferences for the instance.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link EffectiveRecommendationPreferences.Builder} avoiding the need to create one manually via
         * {@link EffectiveRecommendationPreferences#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EffectiveRecommendationPreferences.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #effectiveRecommendationPreferences(EffectiveRecommendationPreferences)}.
         * 
         * @param effectiveRecommendationPreferences
         *        a consumer that will call methods on {@link EffectiveRecommendationPreferences.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #effectiveRecommendationPreferences(EffectiveRecommendationPreferences)
         */
        default Builder effectiveRecommendationPreferences(
                Consumer<EffectiveRecommendationPreferences.Builder> effectiveRecommendationPreferences) {
            return effectiveRecommendationPreferences(EffectiveRecommendationPreferences.builder()
                    .applyMutation(effectiveRecommendationPreferences).build());
        }

        /**
         * <p>
         * The applications that might be running on the instance as inferred by Compute Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instance:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instance as inferred by Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instance:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypesWithStrings(Collection<String> inferredWorkloadTypes);

        /**
         * <p>
         * The applications that might be running on the instance as inferred by Compute Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instance:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instance as inferred by Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instance:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypesWithStrings(String... inferredWorkloadTypes);

        /**
         * <p>
         * The applications that might be running on the instance as inferred by Compute Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instance:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instance as inferred by Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instance:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypes(Collection<InferredWorkloadType> inferredWorkloadTypes);

        /**
         * <p>
         * The applications that might be running on the instance as inferred by Compute Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instance:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instance as inferred by Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instance:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypes(InferredWorkloadType... inferredWorkloadTypes);

        /**
         * <p>
         * The state of the instance when the recommendation was generated.
         * </p>
         * 
         * @param instanceState
         *        The state of the instance when the recommendation was generated.
         * @see InstanceState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceState
         */
        Builder instanceState(String instanceState);

        /**
         * <p>
         * The state of the instance when the recommendation was generated.
         * </p>
         * 
         * @param instanceState
         *        The state of the instance when the recommendation was generated.
         * @see InstanceState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceState
         */
        Builder instanceState(InstanceState instanceState);

        /**
         * <p>
         * A list of tags assigned to your Amazon EC2 instance recommendations.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to your Amazon EC2 instance recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags assigned to your Amazon EC2 instance recommendations.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to your Amazon EC2 instance recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags assigned to your Amazon EC2 instance recommendations.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.computeoptimizer.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String instanceArn;

        private String accountId;

        private String instanceName;

        private String currentInstanceType;

        private String finding;

        private List<String> findingReasonCodes = DefaultSdkAutoConstructList.getInstance();

        private List<UtilizationMetric> utilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private Double lookBackPeriodInDays;

        private List<InstanceRecommendationOption> recommendationOptions = DefaultSdkAutoConstructList.getInstance();

        private List<RecommendationSource> recommendationSources = DefaultSdkAutoConstructList.getInstance();

        private Instant lastRefreshTimestamp;

        private String currentPerformanceRisk;

        private EffectiveRecommendationPreferences effectiveRecommendationPreferences;

        private List<String> inferredWorkloadTypes = DefaultSdkAutoConstructList.getInstance();

        private String instanceState;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceRecommendation model) {
            instanceArn(model.instanceArn);
            accountId(model.accountId);
            instanceName(model.instanceName);
            currentInstanceType(model.currentInstanceType);
            finding(model.finding);
            findingReasonCodesWithStrings(model.findingReasonCodes);
            utilizationMetrics(model.utilizationMetrics);
            lookBackPeriodInDays(model.lookBackPeriodInDays);
            recommendationOptions(model.recommendationOptions);
            recommendationSources(model.recommendationSources);
            lastRefreshTimestamp(model.lastRefreshTimestamp);
            currentPerformanceRisk(model.currentPerformanceRisk);
            effectiveRecommendationPreferences(model.effectiveRecommendationPreferences);
            inferredWorkloadTypesWithStrings(model.inferredWorkloadTypes);
            instanceState(model.instanceState);
            tags(model.tags);
        }

        public final String getInstanceArn() {
            return instanceArn;
        }

        public final void setInstanceArn(String instanceArn) {
            this.instanceArn = instanceArn;
        }

        @Override
        public final Builder instanceArn(String instanceArn) {
            this.instanceArn = instanceArn;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getInstanceName() {
            return instanceName;
        }

        public final void setInstanceName(String instanceName) {
            this.instanceName = instanceName;
        }

        @Override
        public final Builder instanceName(String instanceName) {
            this.instanceName = instanceName;
            return this;
        }

        public final String getCurrentInstanceType() {
            return currentInstanceType;
        }

        public final void setCurrentInstanceType(String currentInstanceType) {
            this.currentInstanceType = currentInstanceType;
        }

        @Override
        public final Builder currentInstanceType(String currentInstanceType) {
            this.currentInstanceType = currentInstanceType;
            return this;
        }

        public final String getFinding() {
            return finding;
        }

        public final void setFinding(String finding) {
            this.finding = finding;
        }

        @Override
        public final Builder finding(String finding) {
            this.finding = finding;
            return this;
        }

        @Override
        public final Builder finding(Finding finding) {
            this.finding(finding == null ? null : finding.toString());
            return this;
        }

        public final Collection<String> getFindingReasonCodes() {
            if (findingReasonCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return findingReasonCodes;
        }

        public final void setFindingReasonCodes(Collection<String> findingReasonCodes) {
            this.findingReasonCodes = InstanceRecommendationFindingReasonCodesCopier.copy(findingReasonCodes);
        }

        @Override
        public final Builder findingReasonCodesWithStrings(Collection<String> findingReasonCodes) {
            this.findingReasonCodes = InstanceRecommendationFindingReasonCodesCopier.copy(findingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingReasonCodesWithStrings(String... findingReasonCodes) {
            findingReasonCodesWithStrings(Arrays.asList(findingReasonCodes));
            return this;
        }

        @Override
        public final Builder findingReasonCodes(Collection<InstanceRecommendationFindingReasonCode> findingReasonCodes) {
            this.findingReasonCodes = InstanceRecommendationFindingReasonCodesCopier.copyEnumToString(findingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingReasonCodes(InstanceRecommendationFindingReasonCode... findingReasonCodes) {
            findingReasonCodes(Arrays.asList(findingReasonCodes));
            return this;
        }

        public final List<UtilizationMetric.Builder> getUtilizationMetrics() {
            List<UtilizationMetric.Builder> result = UtilizationMetricsCopier.copyToBuilder(this.utilizationMetrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUtilizationMetrics(Collection<UtilizationMetric.BuilderImpl> utilizationMetrics) {
            this.utilizationMetrics = UtilizationMetricsCopier.copyFromBuilder(utilizationMetrics);
        }

        @Override
        public final Builder utilizationMetrics(Collection<UtilizationMetric> utilizationMetrics) {
            this.utilizationMetrics = UtilizationMetricsCopier.copy(utilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(UtilizationMetric... utilizationMetrics) {
            utilizationMetrics(Arrays.asList(utilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(Consumer<UtilizationMetric.Builder>... utilizationMetrics) {
            utilizationMetrics(Stream.of(utilizationMetrics).map(c -> UtilizationMetric.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Double getLookBackPeriodInDays() {
            return lookBackPeriodInDays;
        }

        public final void setLookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
        }

        @Override
        public final Builder lookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
            return this;
        }

        public final List<InstanceRecommendationOption.Builder> getRecommendationOptions() {
            List<InstanceRecommendationOption.Builder> result = RecommendationOptionsCopier
                    .copyToBuilder(this.recommendationOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRecommendationOptions(Collection<InstanceRecommendationOption.BuilderImpl> recommendationOptions) {
            this.recommendationOptions = RecommendationOptionsCopier.copyFromBuilder(recommendationOptions);
        }

        @Override
        public final Builder recommendationOptions(Collection<InstanceRecommendationOption> recommendationOptions) {
            this.recommendationOptions = RecommendationOptionsCopier.copy(recommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationOptions(InstanceRecommendationOption... recommendationOptions) {
            recommendationOptions(Arrays.asList(recommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationOptions(Consumer<InstanceRecommendationOption.Builder>... recommendationOptions) {
            recommendationOptions(Stream.of(recommendationOptions)
                    .map(c -> InstanceRecommendationOption.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<RecommendationSource.Builder> getRecommendationSources() {
            List<RecommendationSource.Builder> result = RecommendationSourcesCopier.copyToBuilder(this.recommendationSources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRecommendationSources(Collection<RecommendationSource.BuilderImpl> recommendationSources) {
            this.recommendationSources = RecommendationSourcesCopier.copyFromBuilder(recommendationSources);
        }

        @Override
        public final Builder recommendationSources(Collection<RecommendationSource> recommendationSources) {
            this.recommendationSources = RecommendationSourcesCopier.copy(recommendationSources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationSources(RecommendationSource... recommendationSources) {
            recommendationSources(Arrays.asList(recommendationSources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationSources(Consumer<RecommendationSource.Builder>... recommendationSources) {
            recommendationSources(Stream.of(recommendationSources)
                    .map(c -> RecommendationSource.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getLastRefreshTimestamp() {
            return lastRefreshTimestamp;
        }

        public final void setLastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
        }

        @Override
        public final Builder lastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
            return this;
        }

        public final String getCurrentPerformanceRisk() {
            return currentPerformanceRisk;
        }

        public final void setCurrentPerformanceRisk(String currentPerformanceRisk) {
            this.currentPerformanceRisk = currentPerformanceRisk;
        }

        @Override
        public final Builder currentPerformanceRisk(String currentPerformanceRisk) {
            this.currentPerformanceRisk = currentPerformanceRisk;
            return this;
        }

        @Override
        public final Builder currentPerformanceRisk(CurrentPerformanceRisk currentPerformanceRisk) {
            this.currentPerformanceRisk(currentPerformanceRisk == null ? null : currentPerformanceRisk.toString());
            return this;
        }

        public final EffectiveRecommendationPreferences.Builder getEffectiveRecommendationPreferences() {
            return effectiveRecommendationPreferences != null ? effectiveRecommendationPreferences.toBuilder() : null;
        }

        public final void setEffectiveRecommendationPreferences(
                EffectiveRecommendationPreferences.BuilderImpl effectiveRecommendationPreferences) {
            this.effectiveRecommendationPreferences = effectiveRecommendationPreferences != null ? effectiveRecommendationPreferences
                    .build() : null;
        }

        @Override
        public final Builder effectiveRecommendationPreferences(
                EffectiveRecommendationPreferences effectiveRecommendationPreferences) {
            this.effectiveRecommendationPreferences = effectiveRecommendationPreferences;
            return this;
        }

        public final Collection<String> getInferredWorkloadTypes() {
            if (inferredWorkloadTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return inferredWorkloadTypes;
        }

        public final void setInferredWorkloadTypes(Collection<String> inferredWorkloadTypes) {
            this.inferredWorkloadTypes = InferredWorkloadTypesCopier.copy(inferredWorkloadTypes);
        }

        @Override
        public final Builder inferredWorkloadTypesWithStrings(Collection<String> inferredWorkloadTypes) {
            this.inferredWorkloadTypes = InferredWorkloadTypesCopier.copy(inferredWorkloadTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inferredWorkloadTypesWithStrings(String... inferredWorkloadTypes) {
            inferredWorkloadTypesWithStrings(Arrays.asList(inferredWorkloadTypes));
            return this;
        }

        @Override
        public final Builder inferredWorkloadTypes(Collection<InferredWorkloadType> inferredWorkloadTypes) {
            this.inferredWorkloadTypes = InferredWorkloadTypesCopier.copyEnumToString(inferredWorkloadTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inferredWorkloadTypes(InferredWorkloadType... inferredWorkloadTypes) {
            inferredWorkloadTypes(Arrays.asList(inferredWorkloadTypes));
            return this;
        }

        public final String getInstanceState() {
            return instanceState;
        }

        public final void setInstanceState(String instanceState) {
            this.instanceState = instanceState;
        }

        @Override
        public final Builder instanceState(String instanceState) {
            this.instanceState = instanceState;
            return this;
        }

        @Override
        public final Builder instanceState(InstanceState instanceState) {
            this.instanceState(instanceState == null ? null : instanceState.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public InstanceRecommendation build() {
            return new InstanceRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
