/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The subnet and the security group that DataSync uses to access target EFS file system. The subnet must have at least
 * one mount target for that file system. The security group that you provide needs to be able to communicate with the
 * security group on the mount target in the subnet specified.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Ec2Config implements SdkPojo, Serializable, ToCopyableBuilder<Ec2Config.Builder, Ec2Config> {
    private static final SdkField<String> SUBNET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Ec2Config::subnetArn)).setter(setter(Builder::subnetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetArn").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Ec2Config::securityGroupArns))
            .setter(setter(Builder::securityGroupArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUBNET_ARN_FIELD,
            SECURITY_GROUP_ARNS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String subnetArn;

    private final List<String> securityGroupArns;

    private Ec2Config(BuilderImpl builder) {
        this.subnetArn = builder.subnetArn;
        this.securityGroupArns = builder.securityGroupArns;
    }

    /**
     * <p>
     * The ARN of the subnet and the security group that DataSync uses to access the target EFS file system.
     * </p>
     * 
     * @return The ARN of the subnet and the security group that DataSync uses to access the target EFS file system.
     */
    public String subnetArn() {
        return subnetArn;
    }

    /**
     * Returns true if the SecurityGroupArns property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSecurityGroupArns() {
        return securityGroupArns != null && !(securityGroupArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Names (ARNs) of the security groups that are configured for the Amazon EC2 resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSecurityGroupArns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Amazon Resource Names (ARNs) of the security groups that are configured for the Amazon EC2 resource.
     */
    public List<String> securityGroupArns() {
        return securityGroupArns;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(subnetArn());
        hashCode = 31 * hashCode + Objects.hashCode(securityGroupArns());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Ec2Config)) {
            return false;
        }
        Ec2Config other = (Ec2Config) obj;
        return Objects.equals(subnetArn(), other.subnetArn()) && Objects.equals(securityGroupArns(), other.securityGroupArns());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Ec2Config").add("SubnetArn", subnetArn()).add("SecurityGroupArns", securityGroupArns()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SubnetArn":
            return Optional.ofNullable(clazz.cast(subnetArn()));
        case "SecurityGroupArns":
            return Optional.ofNullable(clazz.cast(securityGroupArns()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Ec2Config, T> g) {
        return obj -> g.apply((Ec2Config) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Ec2Config> {
        /**
         * <p>
         * The ARN of the subnet and the security group that DataSync uses to access the target EFS file system.
         * </p>
         * 
         * @param subnetArn
         *        The ARN of the subnet and the security group that DataSync uses to access the target EFS file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetArn(String subnetArn);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the security groups that are configured for the Amazon EC2 resource.
         * </p>
         * 
         * @param securityGroupArns
         *        The Amazon Resource Names (ARNs) of the security groups that are configured for the Amazon EC2
         *        resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupArns(Collection<String> securityGroupArns);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the security groups that are configured for the Amazon EC2 resource.
         * </p>
         * 
         * @param securityGroupArns
         *        The Amazon Resource Names (ARNs) of the security groups that are configured for the Amazon EC2
         *        resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupArns(String... securityGroupArns);
    }

    static final class BuilderImpl implements Builder {
        private String subnetArn;

        private List<String> securityGroupArns = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Ec2Config model) {
            subnetArn(model.subnetArn);
            securityGroupArns(model.securityGroupArns);
        }

        public final String getSubnetArn() {
            return subnetArn;
        }

        @Override
        public final Builder subnetArn(String subnetArn) {
            this.subnetArn = subnetArn;
            return this;
        }

        public final void setSubnetArn(String subnetArn) {
            this.subnetArn = subnetArn;
        }

        public final Collection<String> getSecurityGroupArns() {
            return securityGroupArns;
        }

        @Override
        public final Builder securityGroupArns(Collection<String> securityGroupArns) {
            this.securityGroupArns = Ec2SecurityGroupArnListCopier.copy(securityGroupArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupArns(String... securityGroupArns) {
            securityGroupArns(Arrays.asList(securityGroupArns));
            return this;
        }

        public final void setSecurityGroupArns(Collection<String> securityGroupArns) {
            this.securityGroupArns = Ec2SecurityGroupArnListCopier.copy(securityGroupArns);
        }

        @Override
        public Ec2Config build() {
            return new Ec2Config(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
