/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticsearch;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.elasticsearch.model.AddTagsRequest;
import software.amazon.awssdk.services.elasticsearch.model.AddTagsResponse;
import software.amazon.awssdk.services.elasticsearch.model.BaseException;
import software.amazon.awssdk.services.elasticsearch.model.CancelElasticsearchServiceSoftwareUpdateRequest;
import software.amazon.awssdk.services.elasticsearch.model.CancelElasticsearchServiceSoftwareUpdateResponse;
import software.amazon.awssdk.services.elasticsearch.model.CreateElasticsearchDomainRequest;
import software.amazon.awssdk.services.elasticsearch.model.CreateElasticsearchDomainResponse;
import software.amazon.awssdk.services.elasticsearch.model.DeleteElasticsearchDomainRequest;
import software.amazon.awssdk.services.elasticsearch.model.DeleteElasticsearchDomainResponse;
import software.amazon.awssdk.services.elasticsearch.model.DeleteElasticsearchServiceRoleRequest;
import software.amazon.awssdk.services.elasticsearch.model.DeleteElasticsearchServiceRoleResponse;
import software.amazon.awssdk.services.elasticsearch.model.DescribeElasticsearchDomainConfigRequest;
import software.amazon.awssdk.services.elasticsearch.model.DescribeElasticsearchDomainConfigResponse;
import software.amazon.awssdk.services.elasticsearch.model.DescribeElasticsearchDomainRequest;
import software.amazon.awssdk.services.elasticsearch.model.DescribeElasticsearchDomainResponse;
import software.amazon.awssdk.services.elasticsearch.model.DescribeElasticsearchDomainsRequest;
import software.amazon.awssdk.services.elasticsearch.model.DescribeElasticsearchDomainsResponse;
import software.amazon.awssdk.services.elasticsearch.model.DescribeElasticsearchInstanceTypeLimitsRequest;
import software.amazon.awssdk.services.elasticsearch.model.DescribeElasticsearchInstanceTypeLimitsResponse;
import software.amazon.awssdk.services.elasticsearch.model.DescribeReservedElasticsearchInstanceOfferingsRequest;
import software.amazon.awssdk.services.elasticsearch.model.DescribeReservedElasticsearchInstanceOfferingsResponse;
import software.amazon.awssdk.services.elasticsearch.model.DescribeReservedElasticsearchInstancesRequest;
import software.amazon.awssdk.services.elasticsearch.model.DescribeReservedElasticsearchInstancesResponse;
import software.amazon.awssdk.services.elasticsearch.model.DisabledOperationException;
import software.amazon.awssdk.services.elasticsearch.model.ElasticsearchException;
import software.amazon.awssdk.services.elasticsearch.model.ElasticsearchRequest;
import software.amazon.awssdk.services.elasticsearch.model.GetCompatibleElasticsearchVersionsRequest;
import software.amazon.awssdk.services.elasticsearch.model.GetCompatibleElasticsearchVersionsResponse;
import software.amazon.awssdk.services.elasticsearch.model.GetUpgradeHistoryRequest;
import software.amazon.awssdk.services.elasticsearch.model.GetUpgradeHistoryResponse;
import software.amazon.awssdk.services.elasticsearch.model.GetUpgradeStatusRequest;
import software.amazon.awssdk.services.elasticsearch.model.GetUpgradeStatusResponse;
import software.amazon.awssdk.services.elasticsearch.model.InternalException;
import software.amazon.awssdk.services.elasticsearch.model.InvalidTypeException;
import software.amazon.awssdk.services.elasticsearch.model.LimitExceededException;
import software.amazon.awssdk.services.elasticsearch.model.ListDomainNamesRequest;
import software.amazon.awssdk.services.elasticsearch.model.ListDomainNamesResponse;
import software.amazon.awssdk.services.elasticsearch.model.ListElasticsearchInstanceTypesRequest;
import software.amazon.awssdk.services.elasticsearch.model.ListElasticsearchInstanceTypesResponse;
import software.amazon.awssdk.services.elasticsearch.model.ListElasticsearchVersionsRequest;
import software.amazon.awssdk.services.elasticsearch.model.ListElasticsearchVersionsResponse;
import software.amazon.awssdk.services.elasticsearch.model.ListTagsRequest;
import software.amazon.awssdk.services.elasticsearch.model.ListTagsResponse;
import software.amazon.awssdk.services.elasticsearch.model.PurchaseReservedElasticsearchInstanceOfferingRequest;
import software.amazon.awssdk.services.elasticsearch.model.PurchaseReservedElasticsearchInstanceOfferingResponse;
import software.amazon.awssdk.services.elasticsearch.model.RemoveTagsRequest;
import software.amazon.awssdk.services.elasticsearch.model.RemoveTagsResponse;
import software.amazon.awssdk.services.elasticsearch.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.elasticsearch.model.ResourceNotFoundException;
import software.amazon.awssdk.services.elasticsearch.model.StartElasticsearchServiceSoftwareUpdateRequest;
import software.amazon.awssdk.services.elasticsearch.model.StartElasticsearchServiceSoftwareUpdateResponse;
import software.amazon.awssdk.services.elasticsearch.model.UpdateElasticsearchDomainConfigRequest;
import software.amazon.awssdk.services.elasticsearch.model.UpdateElasticsearchDomainConfigResponse;
import software.amazon.awssdk.services.elasticsearch.model.UpgradeElasticsearchDomainRequest;
import software.amazon.awssdk.services.elasticsearch.model.UpgradeElasticsearchDomainResponse;
import software.amazon.awssdk.services.elasticsearch.model.ValidationException;
import software.amazon.awssdk.services.elasticsearch.paginators.DescribeReservedElasticsearchInstanceOfferingsIterable;
import software.amazon.awssdk.services.elasticsearch.paginators.DescribeReservedElasticsearchInstancesIterable;
import software.amazon.awssdk.services.elasticsearch.paginators.GetUpgradeHistoryIterable;
import software.amazon.awssdk.services.elasticsearch.paginators.ListElasticsearchInstanceTypesIterable;
import software.amazon.awssdk.services.elasticsearch.paginators.ListElasticsearchVersionsIterable;
import software.amazon.awssdk.services.elasticsearch.transform.AddTagsRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.CancelElasticsearchServiceSoftwareUpdateRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.CreateElasticsearchDomainRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.DeleteElasticsearchDomainRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.DeleteElasticsearchServiceRoleRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.DescribeElasticsearchDomainConfigRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.DescribeElasticsearchDomainRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.DescribeElasticsearchDomainsRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.DescribeElasticsearchInstanceTypeLimitsRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.DescribeReservedElasticsearchInstanceOfferingsRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.DescribeReservedElasticsearchInstancesRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.GetCompatibleElasticsearchVersionsRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.GetUpgradeHistoryRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.GetUpgradeStatusRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.ListDomainNamesRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.ListElasticsearchInstanceTypesRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.ListElasticsearchVersionsRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.PurchaseReservedElasticsearchInstanceOfferingRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.RemoveTagsRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.StartElasticsearchServiceSoftwareUpdateRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.UpdateElasticsearchDomainConfigRequestMarshaller;
import software.amazon.awssdk.services.elasticsearch.transform.UpgradeElasticsearchDomainRequestMarshaller;

/**
 * Internal implementation of {@link ElasticsearchClient}.
 *
 * @see ElasticsearchClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultElasticsearchClient implements ElasticsearchClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultElasticsearchClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Attaches tags to an existing Elasticsearch domain. Tags are a set of case-sensitive key value pairs. An
     * Elasticsearch domain may have up to 10 tags. See <a href=
     * "http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-managedomains.html#es-managedomains-awsresorcetagging"
     * target="_blank"> Tagging Amazon Elasticsearch Service Domains for more information.</a>
     * </p>
     *
     * @param addTagsRequest
     *        Container for the parameters to the <code><a>AddTags</a></code> operation. Specify the tags that you want
     *        to attach to the Elasticsearch domain.
     * @return Result of the AddTags operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws LimitExceededException
     *         An exception for trying to create more than allowed resources or sub-resources. Gives http status code of
     *         409.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.AddTags
     */
    @Override
    public AddTagsResponse addTags(AddTagsRequest addTagsRequest) throws BaseException, LimitExceededException,
            ValidationException, InternalException, AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AddTagsRequest, AddTagsResponse>().withOperationName("AddTags")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(addTagsRequest)
                .withMarshaller(new AddTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Cancels a scheduled service software update for an Amazon ES domain. You can only perform this operation before
     * the <code>AutomatedUpdateDate</code> and when the <code>UpdateStatus</code> is in the <code>PENDING_UPDATE</code>
     * state.
     * </p>
     *
     * @param cancelElasticsearchServiceSoftwareUpdateRequest
     *        Container for the parameters to the <code><a>CancelElasticsearchServiceSoftwareUpdate</a></code>
     *        operation. Specifies the name of the Elasticsearch domain that you wish to cancel a service software
     *        update on.
     * @return Result of the CancelElasticsearchServiceSoftwareUpdate operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.CancelElasticsearchServiceSoftwareUpdate
     */
    @Override
    public CancelElasticsearchServiceSoftwareUpdateResponse cancelElasticsearchServiceSoftwareUpdate(
            CancelElasticsearchServiceSoftwareUpdateRequest cancelElasticsearchServiceSoftwareUpdateRequest)
            throws BaseException, InternalException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelElasticsearchServiceSoftwareUpdateResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, CancelElasticsearchServiceSoftwareUpdateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CancelElasticsearchServiceSoftwareUpdateRequest, CancelElasticsearchServiceSoftwareUpdateResponse>()
                        .withOperationName("CancelElasticsearchServiceSoftwareUpdate").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler)
                        .withInput(cancelElasticsearchServiceSoftwareUpdateRequest)
                        .withMarshaller(new CancelElasticsearchServiceSoftwareUpdateRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a new Elasticsearch domain. For more information, see <a href=
     * "http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomains"
     * target="_blank">Creating Elasticsearch Domains</a> in the <i>Amazon Elasticsearch Service Developer Guide</i>.
     * </p>
     *
     * @param createElasticsearchDomainRequest
     * @return Result of the CreateElasticsearchDomain operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws DisabledOperationException
     *         An error occured because the client wanted to access a not supported operation. Gives http status code of
     *         409.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws InvalidTypeException
     *         An exception for trying to create or access sub-resource that is either invalid or not supported. Gives
     *         http status code of 409.
     * @throws LimitExceededException
     *         An exception for trying to create more than allowed resources or sub-resources. Gives http status code of
     *         409.
     * @throws ResourceAlreadyExistsException
     *         An exception for creating a resource that already exists. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.CreateElasticsearchDomain
     */
    @Override
    public CreateElasticsearchDomainResponse createElasticsearchDomain(
            CreateElasticsearchDomainRequest createElasticsearchDomainRequest) throws BaseException, DisabledOperationException,
            InternalException, InvalidTypeException, LimitExceededException, ResourceAlreadyExistsException, ValidationException,
            AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateElasticsearchDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateElasticsearchDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CreateElasticsearchDomainRequest, CreateElasticsearchDomainResponse>()
                        .withOperationName("CreateElasticsearchDomain").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(createElasticsearchDomainRequest)
                        .withMarshaller(new CreateElasticsearchDomainRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Permanently deletes the specified Elasticsearch domain and all of its data. Once a domain is deleted, it cannot
     * be recovered.
     * </p>
     *
     * @param deleteElasticsearchDomainRequest
     *        Container for the parameters to the <code><a>DeleteElasticsearchDomain</a></code> operation. Specifies the
     *        name of the Elasticsearch domain that you want to delete.
     * @return Result of the DeleteElasticsearchDomain operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.DeleteElasticsearchDomain
     */
    @Override
    public DeleteElasticsearchDomainResponse deleteElasticsearchDomain(
            DeleteElasticsearchDomainRequest deleteElasticsearchDomainRequest) throws BaseException, InternalException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteElasticsearchDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteElasticsearchDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeleteElasticsearchDomainRequest, DeleteElasticsearchDomainResponse>()
                        .withOperationName("DeleteElasticsearchDomain").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deleteElasticsearchDomainRequest)
                        .withMarshaller(new DeleteElasticsearchDomainRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the service-linked role that Elasticsearch Service uses to manage and maintain VPC domains. Role deletion
     * will fail if any existing VPC domains use the role. You must delete any such Elasticsearch domains before
     * deleting the role. See <a
     * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html#es-enabling-slr"
     * target="_blank">Deleting Elasticsearch Service Role</a> in <i>VPC Endpoints for Amazon Elasticsearch Service
     * Domains</i>.
     * </p>
     *
     * @param deleteElasticsearchServiceRoleRequest
     * @return Result of the DeleteElasticsearchServiceRole operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.DeleteElasticsearchServiceRole
     */
    @Override
    public DeleteElasticsearchServiceRoleResponse deleteElasticsearchServiceRole(
            DeleteElasticsearchServiceRoleRequest deleteElasticsearchServiceRoleRequest) throws BaseException, InternalException,
            ValidationException, AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteElasticsearchServiceRoleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteElasticsearchServiceRoleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeleteElasticsearchServiceRoleRequest, DeleteElasticsearchServiceRoleResponse>()
                        .withOperationName("DeleteElasticsearchServiceRole").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deleteElasticsearchServiceRoleRequest)
                        .withMarshaller(new DeleteElasticsearchServiceRoleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns domain configuration information about the specified Elasticsearch domain, including the domain ID,
     * domain endpoint, and domain ARN.
     * </p>
     *
     * @param describeElasticsearchDomainRequest
     *        Container for the parameters to the <code><a>DescribeElasticsearchDomain</a></code> operation.
     * @return Result of the DescribeElasticsearchDomain operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.DescribeElasticsearchDomain
     */
    @Override
    public DescribeElasticsearchDomainResponse describeElasticsearchDomain(
            DescribeElasticsearchDomainRequest describeElasticsearchDomainRequest) throws BaseException, InternalException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeElasticsearchDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeElasticsearchDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeElasticsearchDomainRequest, DescribeElasticsearchDomainResponse>()
                        .withOperationName("DescribeElasticsearchDomain").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeElasticsearchDomainRequest)
                        .withMarshaller(new DescribeElasticsearchDomainRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides cluster configuration information about the specified Elasticsearch domain, such as the state, creation
     * date, update version, and update date for cluster options.
     * </p>
     *
     * @param describeElasticsearchDomainConfigRequest
     *        Container for the parameters to the <code>DescribeElasticsearchDomainConfig</code> operation. Specifies
     *        the domain name for which you want configuration information.
     * @return Result of the DescribeElasticsearchDomainConfig operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.DescribeElasticsearchDomainConfig
     */
    @Override
    public DescribeElasticsearchDomainConfigResponse describeElasticsearchDomainConfig(
            DescribeElasticsearchDomainConfigRequest describeElasticsearchDomainConfigRequest) throws BaseException,
            InternalException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeElasticsearchDomainConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeElasticsearchDomainConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeElasticsearchDomainConfigRequest, DescribeElasticsearchDomainConfigResponse>()
                        .withOperationName("DescribeElasticsearchDomainConfig").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeElasticsearchDomainConfigRequest)
                        .withMarshaller(new DescribeElasticsearchDomainConfigRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns domain configuration information about the specified Elasticsearch domains, including the domain ID,
     * domain endpoint, and domain ARN.
     * </p>
     *
     * @param describeElasticsearchDomainsRequest
     *        Container for the parameters to the <code><a>DescribeElasticsearchDomains</a></code> operation. By
     *        default, the API returns the status of all Elasticsearch domains.
     * @return Result of the DescribeElasticsearchDomains operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.DescribeElasticsearchDomains
     */
    @Override
    public DescribeElasticsearchDomainsResponse describeElasticsearchDomains(
            DescribeElasticsearchDomainsRequest describeElasticsearchDomainsRequest) throws BaseException, InternalException,
            ValidationException, AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeElasticsearchDomainsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeElasticsearchDomainsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeElasticsearchDomainsRequest, DescribeElasticsearchDomainsResponse>()
                        .withOperationName("DescribeElasticsearchDomains").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeElasticsearchDomainsRequest)
                        .withMarshaller(new DescribeElasticsearchDomainsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describe Elasticsearch Limits for a given InstanceType and ElasticsearchVersion. When modifying existing Domain,
     * specify the <code> <a>DomainName</a> </code> to know what Limits are supported for modifying.
     * </p>
     *
     * @param describeElasticsearchInstanceTypeLimitsRequest
     *        Container for the parameters to <code> <a>DescribeElasticsearchInstanceTypeLimits</a> </code> operation.
     * @return Result of the DescribeElasticsearchInstanceTypeLimits operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws InvalidTypeException
     *         An exception for trying to create or access sub-resource that is either invalid or not supported. Gives
     *         http status code of 409.
     * @throws LimitExceededException
     *         An exception for trying to create more than allowed resources or sub-resources. Gives http status code of
     *         409.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.DescribeElasticsearchInstanceTypeLimits
     */
    @Override
    public DescribeElasticsearchInstanceTypeLimitsResponse describeElasticsearchInstanceTypeLimits(
            DescribeElasticsearchInstanceTypeLimitsRequest describeElasticsearchInstanceTypeLimitsRequest) throws BaseException,
            InternalException, InvalidTypeException, LimitExceededException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeElasticsearchInstanceTypeLimitsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeElasticsearchInstanceTypeLimitsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeElasticsearchInstanceTypeLimitsRequest, DescribeElasticsearchInstanceTypeLimitsResponse>()
                        .withOperationName("DescribeElasticsearchInstanceTypeLimits").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeElasticsearchInstanceTypeLimitsRequest)
                        .withMarshaller(new DescribeElasticsearchInstanceTypeLimitsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists available reserved Elasticsearch instance offerings.
     * </p>
     *
     * @param describeReservedElasticsearchInstanceOfferingsRequest
     *        Container for parameters to <code>DescribeReservedElasticsearchInstanceOfferings</code>
     * @return Result of the DescribeReservedElasticsearchInstanceOfferings operation returned by the service.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws DisabledOperationException
     *         An error occured because the client wanted to access a not supported operation. Gives http status code of
     *         409.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.DescribeReservedElasticsearchInstanceOfferings
     */
    @Override
    public DescribeReservedElasticsearchInstanceOfferingsResponse describeReservedElasticsearchInstanceOfferings(
            DescribeReservedElasticsearchInstanceOfferingsRequest describeReservedElasticsearchInstanceOfferingsRequest)
            throws ResourceNotFoundException, ValidationException, DisabledOperationException, InternalException,
            AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeReservedElasticsearchInstanceOfferingsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeReservedElasticsearchInstanceOfferingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeReservedElasticsearchInstanceOfferingsRequest, DescribeReservedElasticsearchInstanceOfferingsResponse>()
                        .withOperationName("DescribeReservedElasticsearchInstanceOfferings").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler)
                        .withInput(describeReservedElasticsearchInstanceOfferingsRequest)
                        .withMarshaller(new DescribeReservedElasticsearchInstanceOfferingsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists available reserved Elasticsearch instance offerings.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeReservedElasticsearchInstanceOfferings(software.amazon.awssdk.services.elasticsearch.model.DescribeReservedElasticsearchInstanceOfferingsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticsearch.paginators.DescribeReservedElasticsearchInstanceOfferingsIterable responses = client.describeReservedElasticsearchInstanceOfferingsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.elasticsearch.paginators.DescribeReservedElasticsearchInstanceOfferingsIterable responses = client
     *             .describeReservedElasticsearchInstanceOfferingsPaginator(request);
     *     for (software.amazon.awssdk.services.elasticsearch.model.DescribeReservedElasticsearchInstanceOfferingsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticsearch.paginators.DescribeReservedElasticsearchInstanceOfferingsIterable responses = client.describeReservedElasticsearchInstanceOfferingsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeReservedElasticsearchInstanceOfferings(software.amazon.awssdk.services.elasticsearch.model.DescribeReservedElasticsearchInstanceOfferingsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeReservedElasticsearchInstanceOfferingsRequest
     *        Container for parameters to <code>DescribeReservedElasticsearchInstanceOfferings</code>
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws DisabledOperationException
     *         An error occured because the client wanted to access a not supported operation. Gives http status code of
     *         409.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.DescribeReservedElasticsearchInstanceOfferings
     */
    @Override
    public DescribeReservedElasticsearchInstanceOfferingsIterable describeReservedElasticsearchInstanceOfferingsPaginator(
            DescribeReservedElasticsearchInstanceOfferingsRequest describeReservedElasticsearchInstanceOfferingsRequest)
            throws ResourceNotFoundException, ValidationException, DisabledOperationException, InternalException,
            AwsServiceException, SdkClientException, ElasticsearchException {
        return new DescribeReservedElasticsearchInstanceOfferingsIterable(this,
                applyPaginatorUserAgent(describeReservedElasticsearchInstanceOfferingsRequest));
    }

    /**
     * <p>
     * Returns information about reserved Elasticsearch instances for this account.
     * </p>
     *
     * @param describeReservedElasticsearchInstancesRequest
     *        Container for parameters to <code>DescribeReservedElasticsearchInstances</code>
     * @return Result of the DescribeReservedElasticsearchInstances operation returned by the service.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws DisabledOperationException
     *         An error occured because the client wanted to access a not supported operation. Gives http status code of
     *         409.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.DescribeReservedElasticsearchInstances
     */
    @Override
    public DescribeReservedElasticsearchInstancesResponse describeReservedElasticsearchInstances(
            DescribeReservedElasticsearchInstancesRequest describeReservedElasticsearchInstancesRequest)
            throws ResourceNotFoundException, InternalException, ValidationException, DisabledOperationException,
            AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeReservedElasticsearchInstancesResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeReservedElasticsearchInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeReservedElasticsearchInstancesRequest, DescribeReservedElasticsearchInstancesResponse>()
                        .withOperationName("DescribeReservedElasticsearchInstances").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeReservedElasticsearchInstancesRequest)
                        .withMarshaller(new DescribeReservedElasticsearchInstancesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about reserved Elasticsearch instances for this account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeReservedElasticsearchInstances(software.amazon.awssdk.services.elasticsearch.model.DescribeReservedElasticsearchInstancesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticsearch.paginators.DescribeReservedElasticsearchInstancesIterable responses = client.describeReservedElasticsearchInstancesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.elasticsearch.paginators.DescribeReservedElasticsearchInstancesIterable responses = client
     *             .describeReservedElasticsearchInstancesPaginator(request);
     *     for (software.amazon.awssdk.services.elasticsearch.model.DescribeReservedElasticsearchInstancesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticsearch.paginators.DescribeReservedElasticsearchInstancesIterable responses = client.describeReservedElasticsearchInstancesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeReservedElasticsearchInstances(software.amazon.awssdk.services.elasticsearch.model.DescribeReservedElasticsearchInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeReservedElasticsearchInstancesRequest
     *        Container for parameters to <code>DescribeReservedElasticsearchInstances</code>
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws DisabledOperationException
     *         An error occured because the client wanted to access a not supported operation. Gives http status code of
     *         409.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.DescribeReservedElasticsearchInstances
     */
    @Override
    public DescribeReservedElasticsearchInstancesIterable describeReservedElasticsearchInstancesPaginator(
            DescribeReservedElasticsearchInstancesRequest describeReservedElasticsearchInstancesRequest)
            throws ResourceNotFoundException, InternalException, ValidationException, DisabledOperationException,
            AwsServiceException, SdkClientException, ElasticsearchException {
        return new DescribeReservedElasticsearchInstancesIterable(this,
                applyPaginatorUserAgent(describeReservedElasticsearchInstancesRequest));
    }

    /**
     * <p>
     * Returns a list of upgrade compatible Elastisearch versions. You can optionally pass a
     * <code> <a>DomainName</a> </code> to get all upgrade compatible Elasticsearch versions for that specific domain.
     * </p>
     *
     * @param getCompatibleElasticsearchVersionsRequest
     *        Container for request parameters to <code> <a>GetCompatibleElasticsearchVersions</a> </code> operation.
     * @return Result of the GetCompatibleElasticsearchVersions operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws DisabledOperationException
     *         An error occured because the client wanted to access a not supported operation. Gives http status code of
     *         409.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.GetCompatibleElasticsearchVersions
     */
    @Override
    public GetCompatibleElasticsearchVersionsResponse getCompatibleElasticsearchVersions(
            GetCompatibleElasticsearchVersionsRequest getCompatibleElasticsearchVersionsRequest) throws BaseException,
            ResourceNotFoundException, DisabledOperationException, ValidationException, InternalException, AwsServiceException,
            SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCompatibleElasticsearchVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCompatibleElasticsearchVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetCompatibleElasticsearchVersionsRequest, GetCompatibleElasticsearchVersionsResponse>()
                        .withOperationName("GetCompatibleElasticsearchVersions").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getCompatibleElasticsearchVersionsRequest)
                        .withMarshaller(new GetCompatibleElasticsearchVersionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the complete history of the last 10 upgrades that were performed on the domain.
     * </p>
     *
     * @param getUpgradeHistoryRequest
     *        Container for request parameters to <code> <a>GetUpgradeHistory</a> </code> operation.
     * @return Result of the GetUpgradeHistory operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws DisabledOperationException
     *         An error occured because the client wanted to access a not supported operation. Gives http status code of
     *         409.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.GetUpgradeHistory
     */
    @Override
    public GetUpgradeHistoryResponse getUpgradeHistory(GetUpgradeHistoryRequest getUpgradeHistoryRequest) throws BaseException,
            ResourceNotFoundException, DisabledOperationException, ValidationException, InternalException, AwsServiceException,
            SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetUpgradeHistoryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetUpgradeHistoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetUpgradeHistoryRequest, GetUpgradeHistoryResponse>()
                .withOperationName("GetUpgradeHistory").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getUpgradeHistoryRequest)
                .withMarshaller(new GetUpgradeHistoryRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the complete history of the last 10 upgrades that were performed on the domain.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getUpgradeHistory(software.amazon.awssdk.services.elasticsearch.model.GetUpgradeHistoryRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticsearch.paginators.GetUpgradeHistoryIterable responses = client.getUpgradeHistoryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.elasticsearch.paginators.GetUpgradeHistoryIterable responses = client
     *             .getUpgradeHistoryPaginator(request);
     *     for (software.amazon.awssdk.services.elasticsearch.model.GetUpgradeHistoryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticsearch.paginators.GetUpgradeHistoryIterable responses = client.getUpgradeHistoryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getUpgradeHistory(software.amazon.awssdk.services.elasticsearch.model.GetUpgradeHistoryRequest)}
     * operation.</b>
     * </p>
     *
     * @param getUpgradeHistoryRequest
     *        Container for request parameters to <code> <a>GetUpgradeHistory</a> </code> operation.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws DisabledOperationException
     *         An error occured because the client wanted to access a not supported operation. Gives http status code of
     *         409.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.GetUpgradeHistory
     */
    @Override
    public GetUpgradeHistoryIterable getUpgradeHistoryPaginator(GetUpgradeHistoryRequest getUpgradeHistoryRequest)
            throws BaseException, ResourceNotFoundException, DisabledOperationException, ValidationException, InternalException,
            AwsServiceException, SdkClientException, ElasticsearchException {
        return new GetUpgradeHistoryIterable(this, applyPaginatorUserAgent(getUpgradeHistoryRequest));
    }

    /**
     * <p>
     * Retrieves the latest status of the last upgrade or upgrade eligibility check that was performed on the domain.
     * </p>
     *
     * @param getUpgradeStatusRequest
     *        Container for request parameters to <code> <a>GetUpgradeStatus</a> </code> operation.
     * @return Result of the GetUpgradeStatus operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws DisabledOperationException
     *         An error occured because the client wanted to access a not supported operation. Gives http status code of
     *         409.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.GetUpgradeStatus
     */
    @Override
    public GetUpgradeStatusResponse getUpgradeStatus(GetUpgradeStatusRequest getUpgradeStatusRequest) throws BaseException,
            ResourceNotFoundException, DisabledOperationException, ValidationException, InternalException, AwsServiceException,
            SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetUpgradeStatusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetUpgradeStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetUpgradeStatusRequest, GetUpgradeStatusResponse>()
                .withOperationName("GetUpgradeStatus").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getUpgradeStatusRequest)
                .withMarshaller(new GetUpgradeStatusRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the name of all Elasticsearch domains owned by the current user's account.
     * </p>
     *
     * @param listDomainNamesRequest
     * @return Result of the ListDomainNames operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.ListDomainNames
     */
    @Override
    public ListDomainNamesResponse listDomainNames(ListDomainNamesRequest listDomainNamesRequest) throws BaseException,
            ValidationException, AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDomainNamesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDomainNamesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListDomainNamesRequest, ListDomainNamesResponse>()
                .withOperationName("ListDomainNames").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listDomainNamesRequest)
                .withMarshaller(new ListDomainNamesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List all Elasticsearch instance types that are supported for given ElasticsearchVersion
     * </p>
     *
     * @param listElasticsearchInstanceTypesRequest
     *        Container for the parameters to the <code> <a>ListElasticsearchInstanceTypes</a> </code> operation.
     * @return Result of the ListElasticsearchInstanceTypes operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.ListElasticsearchInstanceTypes
     */
    @Override
    public ListElasticsearchInstanceTypesResponse listElasticsearchInstanceTypes(
            ListElasticsearchInstanceTypesRequest listElasticsearchInstanceTypesRequest) throws BaseException, InternalException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListElasticsearchInstanceTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListElasticsearchInstanceTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListElasticsearchInstanceTypesRequest, ListElasticsearchInstanceTypesResponse>()
                        .withOperationName("ListElasticsearchInstanceTypes").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listElasticsearchInstanceTypesRequest)
                        .withMarshaller(new ListElasticsearchInstanceTypesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List all Elasticsearch instance types that are supported for given ElasticsearchVersion
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listElasticsearchInstanceTypes(software.amazon.awssdk.services.elasticsearch.model.ListElasticsearchInstanceTypesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticsearch.paginators.ListElasticsearchInstanceTypesIterable responses = client.listElasticsearchInstanceTypesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.elasticsearch.paginators.ListElasticsearchInstanceTypesIterable responses = client
     *             .listElasticsearchInstanceTypesPaginator(request);
     *     for (software.amazon.awssdk.services.elasticsearch.model.ListElasticsearchInstanceTypesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticsearch.paginators.ListElasticsearchInstanceTypesIterable responses = client.listElasticsearchInstanceTypesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listElasticsearchInstanceTypes(software.amazon.awssdk.services.elasticsearch.model.ListElasticsearchInstanceTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listElasticsearchInstanceTypesRequest
     *        Container for the parameters to the <code> <a>ListElasticsearchInstanceTypes</a> </code> operation.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.ListElasticsearchInstanceTypes
     */
    @Override
    public ListElasticsearchInstanceTypesIterable listElasticsearchInstanceTypesPaginator(
            ListElasticsearchInstanceTypesRequest listElasticsearchInstanceTypesRequest) throws BaseException, InternalException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, ElasticsearchException {
        return new ListElasticsearchInstanceTypesIterable(this, applyPaginatorUserAgent(listElasticsearchInstanceTypesRequest));
    }

    /**
     * <p>
     * List all supported Elasticsearch versions
     * </p>
     *
     * @param listElasticsearchVersionsRequest
     *        Container for the parameters to the <code> <a>ListElasticsearchVersions</a> </code> operation.
     *        <p>
     *        Use <code> <a>MaxResults</a> </code> to control the maximum number of results to retrieve in a single
     *        call.
     *        </p>
     *        <p>
     *        Use <code> <a>NextToken</a> </code> in response to retrieve more results. If the received response does
     *        not contain a NextToken, then there are no more results to retrieve.
     *        </p>
     * @return Result of the ListElasticsearchVersions operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.ListElasticsearchVersions
     */
    @Override
    public ListElasticsearchVersionsResponse listElasticsearchVersions(
            ListElasticsearchVersionsRequest listElasticsearchVersionsRequest) throws BaseException, InternalException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListElasticsearchVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListElasticsearchVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListElasticsearchVersionsRequest, ListElasticsearchVersionsResponse>()
                        .withOperationName("ListElasticsearchVersions").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listElasticsearchVersionsRequest)
                        .withMarshaller(new ListElasticsearchVersionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List all supported Elasticsearch versions
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listElasticsearchVersions(software.amazon.awssdk.services.elasticsearch.model.ListElasticsearchVersionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticsearch.paginators.ListElasticsearchVersionsIterable responses = client.listElasticsearchVersionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.elasticsearch.paginators.ListElasticsearchVersionsIterable responses = client
     *             .listElasticsearchVersionsPaginator(request);
     *     for (software.amazon.awssdk.services.elasticsearch.model.ListElasticsearchVersionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticsearch.paginators.ListElasticsearchVersionsIterable responses = client.listElasticsearchVersionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listElasticsearchVersions(software.amazon.awssdk.services.elasticsearch.model.ListElasticsearchVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listElasticsearchVersionsRequest
     *        Container for the parameters to the <code> <a>ListElasticsearchVersions</a> </code> operation.
     *        <p>
     *        Use <code> <a>MaxResults</a> </code> to control the maximum number of results to retrieve in a single
     *        call.
     *        </p>
     *        <p>
     *        Use <code> <a>NextToken</a> </code> in response to retrieve more results. If the received response does
     *        not contain a NextToken, then there are no more results to retrieve.
     *        </p>
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.ListElasticsearchVersions
     */
    @Override
    public ListElasticsearchVersionsIterable listElasticsearchVersionsPaginator(
            ListElasticsearchVersionsRequest listElasticsearchVersionsRequest) throws BaseException, InternalException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, ElasticsearchException {
        return new ListElasticsearchVersionsIterable(this, applyPaginatorUserAgent(listElasticsearchVersionsRequest));
    }

    /**
     * <p>
     * Returns all tags for the given Elasticsearch domain.
     * </p>
     *
     * @param listTagsRequest
     *        Container for the parameters to the <code><a>ListTags</a></code> operation. Specify the <code>ARN</code>
     *        for the Elasticsearch domain to which the tags are attached that you want to view are attached.
     * @return Result of the ListTags operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.ListTags
     */
    @Override
    public ListTagsResponse listTags(ListTagsRequest listTagsRequest) throws BaseException, ResourceNotFoundException,
            ValidationException, InternalException, AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>().withOperationName("ListTags")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listTagsRequest)
                .withMarshaller(new ListTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Allows you to purchase reserved Elasticsearch instances.
     * </p>
     *
     * @param purchaseReservedElasticsearchInstanceOfferingRequest
     *        Container for parameters to <code>PurchaseReservedElasticsearchInstanceOffering</code>
     * @return Result of the PurchaseReservedElasticsearchInstanceOffering operation returned by the service.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ResourceAlreadyExistsException
     *         An exception for creating a resource that already exists. Gives http status code of 400.
     * @throws LimitExceededException
     *         An exception for trying to create more than allowed resources or sub-resources. Gives http status code of
     *         409.
     * @throws DisabledOperationException
     *         An error occured because the client wanted to access a not supported operation. Gives http status code of
     *         409.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.PurchaseReservedElasticsearchInstanceOffering
     */
    @Override
    public PurchaseReservedElasticsearchInstanceOfferingResponse purchaseReservedElasticsearchInstanceOffering(
            PurchaseReservedElasticsearchInstanceOfferingRequest purchaseReservedElasticsearchInstanceOfferingRequest)
            throws ResourceNotFoundException, ResourceAlreadyExistsException, LimitExceededException, DisabledOperationException,
            ValidationException, InternalException, AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PurchaseReservedElasticsearchInstanceOfferingResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, PurchaseReservedElasticsearchInstanceOfferingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<PurchaseReservedElasticsearchInstanceOfferingRequest, PurchaseReservedElasticsearchInstanceOfferingResponse>()
                        .withOperationName("PurchaseReservedElasticsearchInstanceOffering").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler)
                        .withInput(purchaseReservedElasticsearchInstanceOfferingRequest)
                        .withMarshaller(new PurchaseReservedElasticsearchInstanceOfferingRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the specified set of tags from the specified Elasticsearch domain.
     * </p>
     *
     * @param removeTagsRequest
     *        Container for the parameters to the <code><a>RemoveTags</a></code> operation. Specify the <code>ARN</code>
     *        for the Elasticsearch domain from which you want to remove the specified <code>TagKey</code>.
     * @return Result of the RemoveTags operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.RemoveTags
     */
    @Override
    public RemoveTagsResponse removeTags(RemoveTagsRequest removeTagsRequest) throws BaseException, ValidationException,
            InternalException, AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RemoveTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RemoveTagsRequest, RemoveTagsResponse>()
                .withOperationName("RemoveTags").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(removeTagsRequest)
                .withMarshaller(new RemoveTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Schedules a service software update for an Amazon ES domain.
     * </p>
     *
     * @param startElasticsearchServiceSoftwareUpdateRequest
     *        Container for the parameters to the <code><a>StartElasticsearchServiceSoftwareUpdate</a></code> operation.
     *        Specifies the name of the Elasticsearch domain that you wish to schedule a service software update on.
     * @return Result of the StartElasticsearchServiceSoftwareUpdate operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.StartElasticsearchServiceSoftwareUpdate
     */
    @Override
    public StartElasticsearchServiceSoftwareUpdateResponse startElasticsearchServiceSoftwareUpdate(
            StartElasticsearchServiceSoftwareUpdateRequest startElasticsearchServiceSoftwareUpdateRequest) throws BaseException,
            InternalException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartElasticsearchServiceSoftwareUpdateResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, StartElasticsearchServiceSoftwareUpdateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<StartElasticsearchServiceSoftwareUpdateRequest, StartElasticsearchServiceSoftwareUpdateResponse>()
                        .withOperationName("StartElasticsearchServiceSoftwareUpdate").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(startElasticsearchServiceSoftwareUpdateRequest)
                        .withMarshaller(new StartElasticsearchServiceSoftwareUpdateRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Modifies the cluster configuration of the specified Elasticsearch domain, setting as setting the instance type
     * and the number of instances.
     * </p>
     *
     * @param updateElasticsearchDomainConfigRequest
     *        Container for the parameters to the <code><a>UpdateElasticsearchDomain</a></code> operation. Specifies the
     *        type and number of instances in the domain cluster.
     * @return Result of the UpdateElasticsearchDomainConfig operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws InvalidTypeException
     *         An exception for trying to create or access sub-resource that is either invalid or not supported. Gives
     *         http status code of 409.
     * @throws LimitExceededException
     *         An exception for trying to create more than allowed resources or sub-resources. Gives http status code of
     *         409.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.UpdateElasticsearchDomainConfig
     */
    @Override
    public UpdateElasticsearchDomainConfigResponse updateElasticsearchDomainConfig(
            UpdateElasticsearchDomainConfigRequest updateElasticsearchDomainConfigRequest) throws BaseException,
            InternalException, InvalidTypeException, LimitExceededException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateElasticsearchDomainConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateElasticsearchDomainConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateElasticsearchDomainConfigRequest, UpdateElasticsearchDomainConfigResponse>()
                        .withOperationName("UpdateElasticsearchDomainConfig").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateElasticsearchDomainConfigRequest)
                        .withMarshaller(new UpdateElasticsearchDomainConfigRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Allows you to either upgrade your domain or perform an Upgrade eligibility check to a compatible Elasticsearch
     * version.
     * </p>
     *
     * @param upgradeElasticsearchDomainRequest
     *        Container for request parameters to <code> <a>UpgradeElasticsearchDomain</a> </code> operation.
     * @return Result of the UpgradeElasticsearchDomain operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws ResourceNotFoundException
     *         An exception for accessing or deleting a resource that does not exist. Gives http status code of 400.
     * @throws ResourceAlreadyExistsException
     *         An exception for creating a resource that already exists. Gives http status code of 400.
     * @throws DisabledOperationException
     *         An error occured because the client wanted to access a not supported operation. Gives http status code of
     *         409.
     * @throws ValidationException
     *         An exception for missing / invalid input fields. Gives http status code of 400.
     * @throws InternalException
     *         The request processing has failed because of an unknown error, exception or failure (the failure is
     *         internal to the service) . Gives http status code of 500.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticsearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticsearchClient.UpgradeElasticsearchDomain
     */
    @Override
    public UpgradeElasticsearchDomainResponse upgradeElasticsearchDomain(
            UpgradeElasticsearchDomainRequest upgradeElasticsearchDomainRequest) throws BaseException, ResourceNotFoundException,
            ResourceAlreadyExistsException, DisabledOperationException, ValidationException, InternalException,
            AwsServiceException, SdkClientException, ElasticsearchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpgradeElasticsearchDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpgradeElasticsearchDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpgradeElasticsearchDomainRequest, UpgradeElasticsearchDomainResponse>()
                        .withOperationName("UpgradeElasticsearchDomain").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(upgradeElasticsearchDomainRequest)
                        .withMarshaller(new UpgradeElasticsearchDomainRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ElasticsearchException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DisabledOperationException")
                                .exceptionBuilderSupplier(DisabledOperationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTypeException")
                                .exceptionBuilderSupplier(InvalidTypeException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BaseException").exceptionBuilderSupplier(BaseException::builder)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ElasticsearchRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
