/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>This data type is used with the GameLift FleetIQ and game server groups.</b>
 * </p>
 * <p>
 * Properties that describe a game server group resource. A game server group manages certain properties related to a
 * corresponding Amazon EC2 Auto Scaling group.
 * </p>
 * <p>
 * A game server group is created by a successful call to <code>CreateGameServerGroup</code> and deleted by calling
 * <code>DeleteGameServerGroup</code>. Game server group activity can be temporarily suspended and resumed by calling
 * <code>SuspendGameServerGroup</code> and <code>ResumeGameServerGroup</code>, respectively.
 * </p>
 * <p>
 * <b>Related actions</b>
 * </p>
 * <p>
 * <a>CreateGameServerGroup</a> | <a>ListGameServerGroups</a> | <a>DescribeGameServerGroup</a> |
 * <a>UpdateGameServerGroup</a> | <a>DeleteGameServerGroup</a> | <a>ResumeGameServerGroup</a> |
 * <a>SuspendGameServerGroup</a> | <a>DescribeGameServerInstances</a> | <a
 * href="https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/reference-awssdk-fleetiq.html">All APIs by task</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GameServerGroup implements SdkPojo, Serializable, ToCopyableBuilder<GameServerGroup.Builder, GameServerGroup> {
    private static final SdkField<String> GAME_SERVER_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameServerGroupName").getter(getter(GameServerGroup::gameServerGroupName))
            .setter(setter(Builder::gameServerGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerGroupName").build())
            .build();

    private static final SdkField<String> GAME_SERVER_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameServerGroupArn").getter(getter(GameServerGroup::gameServerGroupArn))
            .setter(setter(Builder::gameServerGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerGroupArn").build())
            .build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(GameServerGroup::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<List<InstanceDefinition>> INSTANCE_DEFINITIONS_FIELD = SdkField
            .<List<InstanceDefinition>> builder(MarshallingType.LIST)
            .memberName("InstanceDefinitions")
            .getter(getter(GameServerGroup::instanceDefinitions))
            .setter(setter(Builder::instanceDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InstanceDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> BALANCING_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BalancingStrategy").getter(getter(GameServerGroup::balancingStrategyAsString))
            .setter(setter(Builder::balancingStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BalancingStrategy").build()).build();

    private static final SdkField<String> GAME_SERVER_PROTECTION_POLICY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("GameServerProtectionPolicy")
            .getter(getter(GameServerGroup::gameServerProtectionPolicyAsString))
            .setter(setter(Builder::gameServerProtectionPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerProtectionPolicy").build())
            .build();

    private static final SdkField<String> AUTO_SCALING_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AutoScalingGroupArn").getter(getter(GameServerGroup::autoScalingGroupArn))
            .setter(setter(Builder::autoScalingGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoScalingGroupArn").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(GameServerGroup::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusReason").getter(getter(GameServerGroup::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final SdkField<List<String>> SUSPENDED_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SuspendedActions")
            .getter(getter(GameServerGroup::suspendedActionsAsStrings))
            .setter(setter(Builder::suspendedActionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SuspendedActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(GameServerGroup::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTime").getter(getter(GameServerGroup::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GAME_SERVER_GROUP_NAME_FIELD,
            GAME_SERVER_GROUP_ARN_FIELD, ROLE_ARN_FIELD, INSTANCE_DEFINITIONS_FIELD, BALANCING_STRATEGY_FIELD,
            GAME_SERVER_PROTECTION_POLICY_FIELD, AUTO_SCALING_GROUP_ARN_FIELD, STATUS_FIELD, STATUS_REASON_FIELD,
            SUSPENDED_ACTIONS_FIELD, CREATION_TIME_FIELD, LAST_UPDATED_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String gameServerGroupName;

    private final String gameServerGroupArn;

    private final String roleArn;

    private final List<InstanceDefinition> instanceDefinitions;

    private final String balancingStrategy;

    private final String gameServerProtectionPolicy;

    private final String autoScalingGroupArn;

    private final String status;

    private final String statusReason;

    private final List<String> suspendedActions;

    private final Instant creationTime;

    private final Instant lastUpdatedTime;

    private GameServerGroup(BuilderImpl builder) {
        this.gameServerGroupName = builder.gameServerGroupName;
        this.gameServerGroupArn = builder.gameServerGroupArn;
        this.roleArn = builder.roleArn;
        this.instanceDefinitions = builder.instanceDefinitions;
        this.balancingStrategy = builder.balancingStrategy;
        this.gameServerProtectionPolicy = builder.gameServerProtectionPolicy;
        this.autoScalingGroupArn = builder.autoScalingGroupArn;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.suspendedActions = builder.suspendedActions;
        this.creationTime = builder.creationTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
    }

    /**
     * <p>
     * A developer-defined identifier for the game server group. The name is unique for each Region in each Amazon Web
     * Services account.
     * </p>
     * 
     * @return A developer-defined identifier for the game server group. The name is unique for each Region in each
     *         Amazon Web Services account.
     */
    public final String gameServerGroupName() {
        return gameServerGroupName;
    }

    /**
     * <p>
     * A generated unique ID for the game server group.
     * </p>
     * 
     * @return A generated unique ID for the game server group.
     */
    public final String gameServerGroupArn() {
        return gameServerGroupArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * for an IAM role that allows Amazon GameLift to access your Amazon EC2 Auto Scaling groups.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) for an IAM role that
     *         allows Amazon GameLift to access your Amazon EC2 Auto Scaling groups.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceDefinitions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceDefinitions() {
        return instanceDefinitions != null && !(instanceDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of Amazon EC2 instance types that GameLift FleetIQ can use when balancing and automatically scaling
     * instances in the corresponding Auto Scaling group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceDefinitions} method.
     * </p>
     * 
     * @return The set of Amazon EC2 instance types that GameLift FleetIQ can use when balancing and automatically
     *         scaling instances in the corresponding Auto Scaling group.
     */
    public final List<InstanceDefinition> instanceDefinitions() {
        return instanceDefinitions;
    }

    /**
     * <p>
     * Indicates how GameLift FleetIQ balances the use of Spot Instances and On-Demand Instances in the game server
     * group. Method options include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SPOT_ONLY</code> - Only Spot Instances are used in the game server group. If Spot Instances are unavailable
     * or not viable for game hosting, the game server group provides no hosting capacity until Spot Instances can again
     * be used. Until then, no new instances are started, and the existing nonviable Spot Instances are terminated
     * (after current gameplay ends) and are not replaced.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SPOT_PREFERRED</code> - (default value) Spot Instances are used whenever available in the game server
     * group. If Spot Instances are unavailable, the game server group continues to provide hosting capacity by falling
     * back to On-Demand Instances. Existing nonviable Spot Instances are terminated (after current gameplay ends) and
     * are replaced with new On-Demand Instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ON_DEMAND_ONLY</code> - Only On-Demand Instances are used in the game server group. No Spot Instances are
     * used, even when available, while this balancing strategy is in force.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #balancingStrategy}
     * will return {@link BalancingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #balancingStrategyAsString}.
     * </p>
     * 
     * @return Indicates how GameLift FleetIQ balances the use of Spot Instances and On-Demand Instances in the game
     *         server group. Method options include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SPOT_ONLY</code> - Only Spot Instances are used in the game server group. If Spot Instances are
     *         unavailable or not viable for game hosting, the game server group provides no hosting capacity until Spot
     *         Instances can again be used. Until then, no new instances are started, and the existing nonviable Spot
     *         Instances are terminated (after current gameplay ends) and are not replaced.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SPOT_PREFERRED</code> - (default value) Spot Instances are used whenever available in the game
     *         server group. If Spot Instances are unavailable, the game server group continues to provide hosting
     *         capacity by falling back to On-Demand Instances. Existing nonviable Spot Instances are terminated (after
     *         current gameplay ends) and are replaced with new On-Demand Instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ON_DEMAND_ONLY</code> - Only On-Demand Instances are used in the game server group. No Spot
     *         Instances are used, even when available, while this balancing strategy is in force.
     *         </p>
     *         </li>
     * @see BalancingStrategy
     */
    public final BalancingStrategy balancingStrategy() {
        return BalancingStrategy.fromValue(balancingStrategy);
    }

    /**
     * <p>
     * Indicates how GameLift FleetIQ balances the use of Spot Instances and On-Demand Instances in the game server
     * group. Method options include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SPOT_ONLY</code> - Only Spot Instances are used in the game server group. If Spot Instances are unavailable
     * or not viable for game hosting, the game server group provides no hosting capacity until Spot Instances can again
     * be used. Until then, no new instances are started, and the existing nonviable Spot Instances are terminated
     * (after current gameplay ends) and are not replaced.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SPOT_PREFERRED</code> - (default value) Spot Instances are used whenever available in the game server
     * group. If Spot Instances are unavailable, the game server group continues to provide hosting capacity by falling
     * back to On-Demand Instances. Existing nonviable Spot Instances are terminated (after current gameplay ends) and
     * are replaced with new On-Demand Instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ON_DEMAND_ONLY</code> - Only On-Demand Instances are used in the game server group. No Spot Instances are
     * used, even when available, while this balancing strategy is in force.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #balancingStrategy}
     * will return {@link BalancingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #balancingStrategyAsString}.
     * </p>
     * 
     * @return Indicates how GameLift FleetIQ balances the use of Spot Instances and On-Demand Instances in the game
     *         server group. Method options include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SPOT_ONLY</code> - Only Spot Instances are used in the game server group. If Spot Instances are
     *         unavailable or not viable for game hosting, the game server group provides no hosting capacity until Spot
     *         Instances can again be used. Until then, no new instances are started, and the existing nonviable Spot
     *         Instances are terminated (after current gameplay ends) and are not replaced.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SPOT_PREFERRED</code> - (default value) Spot Instances are used whenever available in the game
     *         server group. If Spot Instances are unavailable, the game server group continues to provide hosting
     *         capacity by falling back to On-Demand Instances. Existing nonviable Spot Instances are terminated (after
     *         current gameplay ends) and are replaced with new On-Demand Instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ON_DEMAND_ONLY</code> - Only On-Demand Instances are used in the game server group. No Spot
     *         Instances are used, even when available, while this balancing strategy is in force.
     *         </p>
     *         </li>
     * @see BalancingStrategy
     */
    public final String balancingStrategyAsString() {
        return balancingStrategy;
    }

    /**
     * <p>
     * A flag that indicates whether instances in the game server group are protected from early termination.
     * Unprotected instances that have active game servers running might be terminated during a scale-down event,
     * causing players to be dropped from the game. Protected instances cannot be terminated while there are active game
     * servers running except in the event of a forced game server group deletion (see ). An exception to this is with
     * Spot Instances, which can be terminated by Amazon Web Services regardless of protection status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #gameServerProtectionPolicy} will return {@link GameServerProtectionPolicy#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #gameServerProtectionPolicyAsString}.
     * </p>
     * 
     * @return A flag that indicates whether instances in the game server group are protected from early termination.
     *         Unprotected instances that have active game servers running might be terminated during a scale-down
     *         event, causing players to be dropped from the game. Protected instances cannot be terminated while there
     *         are active game servers running except in the event of a forced game server group deletion (see ). An
     *         exception to this is with Spot Instances, which can be terminated by Amazon Web Services regardless of
     *         protection status.
     * @see GameServerProtectionPolicy
     */
    public final GameServerProtectionPolicy gameServerProtectionPolicy() {
        return GameServerProtectionPolicy.fromValue(gameServerProtectionPolicy);
    }

    /**
     * <p>
     * A flag that indicates whether instances in the game server group are protected from early termination.
     * Unprotected instances that have active game servers running might be terminated during a scale-down event,
     * causing players to be dropped from the game. Protected instances cannot be terminated while there are active game
     * servers running except in the event of a forced game server group deletion (see ). An exception to this is with
     * Spot Instances, which can be terminated by Amazon Web Services regardless of protection status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #gameServerProtectionPolicy} will return {@link GameServerProtectionPolicy#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #gameServerProtectionPolicyAsString}.
     * </p>
     * 
     * @return A flag that indicates whether instances in the game server group are protected from early termination.
     *         Unprotected instances that have active game servers running might be terminated during a scale-down
     *         event, causing players to be dropped from the game. Protected instances cannot be terminated while there
     *         are active game servers running except in the event of a forced game server group deletion (see ). An
     *         exception to this is with Spot Instances, which can be terminated by Amazon Web Services regardless of
     *         protection status.
     * @see GameServerProtectionPolicy
     */
    public final String gameServerProtectionPolicyAsString() {
        return gameServerProtectionPolicy;
    }

    /**
     * <p>
     * A generated unique ID for the Amazon EC2 Auto Scaling group that is associated with this game server group.
     * </p>
     * 
     * @return A generated unique ID for the Amazon EC2 Auto Scaling group that is associated with this game server
     *         group.
     */
    public final String autoScalingGroupArn() {
        return autoScalingGroupArn;
    }

    /**
     * <p>
     * The current status of the game server group. Possible statuses include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NEW</code> - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVATING</code> - GameLift FleetIQ is setting up a game server group, which includes creating an Auto
     * Scaling group in your Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The game server group has been successfully created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE_SCHEDULED</code> - A request to delete the game server group has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request and is
     * processing it. GameLift FleetIQ must first complete and release hosts before it deletes the Auto Scaling group
     * and the game server group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETED</code> - The game server group has been successfully deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ERROR</code> - The asynchronous processes of activating or deleting a game server group has failed,
     * resulting in an error state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GameServerGroupStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the game server group. Possible statuses include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NEW</code> - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVATING</code> - GameLift FleetIQ is setting up a game server group, which includes creating an
     *         Auto Scaling group in your Amazon Web Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The game server group has been successfully created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE_SCHEDULED</code> - A request to delete the game server group has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code>
     *         request and is processing it. GameLift FleetIQ must first complete and release hosts before it deletes
     *         the Auto Scaling group and the game server group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETED</code> - The game server group has been successfully deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ERROR</code> - The asynchronous processes of activating or deleting a game server group has failed,
     *         resulting in an error state.
     *         </p>
     *         </li>
     * @see GameServerGroupStatus
     */
    public final GameServerGroupStatus status() {
        return GameServerGroupStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the game server group. Possible statuses include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NEW</code> - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVATING</code> - GameLift FleetIQ is setting up a game server group, which includes creating an Auto
     * Scaling group in your Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The game server group has been successfully created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE_SCHEDULED</code> - A request to delete the game server group has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request and is
     * processing it. GameLift FleetIQ must first complete and release hosts before it deletes the Auto Scaling group
     * and the game server group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETED</code> - The game server group has been successfully deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ERROR</code> - The asynchronous processes of activating or deleting a game server group has failed,
     * resulting in an error state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GameServerGroupStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the game server group. Possible statuses include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NEW</code> - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVATING</code> - GameLift FleetIQ is setting up a game server group, which includes creating an
     *         Auto Scaling group in your Amazon Web Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The game server group has been successfully created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE_SCHEDULED</code> - A request to delete the game server group has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code>
     *         request and is processing it. GameLift FleetIQ must first complete and release hosts before it deletes
     *         the Auto Scaling group and the game server group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETED</code> - The game server group has been successfully deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ERROR</code> - The asynchronous processes of activating or deleting a game server group has failed,
     *         resulting in an error state.
     *         </p>
     *         </li>
     * @see GameServerGroupStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Additional information about the current game server group status. This information might provide additional
     * insight on groups that are in <code>ERROR</code> status.
     * </p>
     * 
     * @return Additional information about the current game server group status. This information might provide
     *         additional insight on groups that are in <code>ERROR</code> status.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * A list of activities that are currently suspended for this game server group. If this property is empty, all
     * activities are occurring.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSuspendedActions} method.
     * </p>
     * 
     * @return A list of activities that are currently suspended for this game server group. If this property is empty,
     *         all activities are occurring.
     */
    public final List<GameServerGroupAction> suspendedActions() {
        return GameServerGroupActionsCopier.copyStringToEnum(suspendedActions);
    }

    /**
     * For responses, this returns true if the service returned a value for the SuspendedActions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSuspendedActions() {
        return suspendedActions != null && !(suspendedActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of activities that are currently suspended for this game server group. If this property is empty, all
     * activities are occurring.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSuspendedActions} method.
     * </p>
     * 
     * @return A list of activities that are currently suspended for this game server group. If this property is empty,
     *         all activities are occurring.
     */
    public final List<String> suspendedActionsAsStrings() {
        return suspendedActions;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * A timestamp that indicates when this game server group was last updated.
     * </p>
     * 
     * @return A timestamp that indicates when this game server group was last updated.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(gameServerGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceDefinitions() ? instanceDefinitions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(balancingStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerProtectionPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(hasSuspendedActions() ? suspendedActionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GameServerGroup)) {
            return false;
        }
        GameServerGroup other = (GameServerGroup) obj;
        return Objects.equals(gameServerGroupName(), other.gameServerGroupName())
                && Objects.equals(gameServerGroupArn(), other.gameServerGroupArn()) && Objects.equals(roleArn(), other.roleArn())
                && hasInstanceDefinitions() == other.hasInstanceDefinitions()
                && Objects.equals(instanceDefinitions(), other.instanceDefinitions())
                && Objects.equals(balancingStrategyAsString(), other.balancingStrategyAsString())
                && Objects.equals(gameServerProtectionPolicyAsString(), other.gameServerProtectionPolicyAsString())
                && Objects.equals(autoScalingGroupArn(), other.autoScalingGroupArn())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason()) && hasSuspendedActions() == other.hasSuspendedActions()
                && Objects.equals(suspendedActionsAsStrings(), other.suspendedActionsAsStrings())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GameServerGroup").add("GameServerGroupName", gameServerGroupName())
                .add("GameServerGroupArn", gameServerGroupArn()).add("RoleArn", roleArn())
                .add("InstanceDefinitions", hasInstanceDefinitions() ? instanceDefinitions() : null)
                .add("BalancingStrategy", balancingStrategyAsString())
                .add("GameServerProtectionPolicy", gameServerProtectionPolicyAsString())
                .add("AutoScalingGroupArn", autoScalingGroupArn()).add("Status", statusAsString())
                .add("StatusReason", statusReason())
                .add("SuspendedActions", hasSuspendedActions() ? suspendedActionsAsStrings() : null)
                .add("CreationTime", creationTime()).add("LastUpdatedTime", lastUpdatedTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "GameServerGroupName":
            return Optional.ofNullable(clazz.cast(gameServerGroupName()));
        case "GameServerGroupArn":
            return Optional.ofNullable(clazz.cast(gameServerGroupArn()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "InstanceDefinitions":
            return Optional.ofNullable(clazz.cast(instanceDefinitions()));
        case "BalancingStrategy":
            return Optional.ofNullable(clazz.cast(balancingStrategyAsString()));
        case "GameServerProtectionPolicy":
            return Optional.ofNullable(clazz.cast(gameServerProtectionPolicyAsString()));
        case "AutoScalingGroupArn":
            return Optional.ofNullable(clazz.cast(autoScalingGroupArn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "SuspendedActions":
            return Optional.ofNullable(clazz.cast(suspendedActionsAsStrings()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GameServerGroup, T> g) {
        return obj -> g.apply((GameServerGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GameServerGroup> {
        /**
         * <p>
         * A developer-defined identifier for the game server group. The name is unique for each Region in each Amazon
         * Web Services account.
         * </p>
         * 
         * @param gameServerGroupName
         *        A developer-defined identifier for the game server group. The name is unique for each Region in each
         *        Amazon Web Services account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerGroupName(String gameServerGroupName);

        /**
         * <p>
         * A generated unique ID for the game server group.
         * </p>
         * 
         * @param gameServerGroupArn
         *        A generated unique ID for the game server group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerGroupArn(String gameServerGroupArn);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) for an IAM role that
         * allows Amazon GameLift to access your Amazon EC2 Auto Scaling groups.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) for an IAM role
         *        that allows Amazon GameLift to access your Amazon EC2 Auto Scaling groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The set of Amazon EC2 instance types that GameLift FleetIQ can use when balancing and automatically scaling
         * instances in the corresponding Auto Scaling group.
         * </p>
         * 
         * @param instanceDefinitions
         *        The set of Amazon EC2 instance types that GameLift FleetIQ can use when balancing and automatically
         *        scaling instances in the corresponding Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceDefinitions(Collection<InstanceDefinition> instanceDefinitions);

        /**
         * <p>
         * The set of Amazon EC2 instance types that GameLift FleetIQ can use when balancing and automatically scaling
         * instances in the corresponding Auto Scaling group.
         * </p>
         * 
         * @param instanceDefinitions
         *        The set of Amazon EC2 instance types that GameLift FleetIQ can use when balancing and automatically
         *        scaling instances in the corresponding Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceDefinitions(InstanceDefinition... instanceDefinitions);

        /**
         * <p>
         * The set of Amazon EC2 instance types that GameLift FleetIQ can use when balancing and automatically scaling
         * instances in the corresponding Auto Scaling group.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.InstanceDefinition.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.gamelift.model.InstanceDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.InstanceDefinition.Builder#build()} is called
         * immediately and its result is passed to {@link #instanceDefinitions(List<InstanceDefinition>)}.
         * 
         * @param instanceDefinitions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.InstanceDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceDefinitions(java.util.Collection<InstanceDefinition>)
         */
        Builder instanceDefinitions(Consumer<InstanceDefinition.Builder>... instanceDefinitions);

        /**
         * <p>
         * Indicates how GameLift FleetIQ balances the use of Spot Instances and On-Demand Instances in the game server
         * group. Method options include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SPOT_ONLY</code> - Only Spot Instances are used in the game server group. If Spot Instances are
         * unavailable or not viable for game hosting, the game server group provides no hosting capacity until Spot
         * Instances can again be used. Until then, no new instances are started, and the existing nonviable Spot
         * Instances are terminated (after current gameplay ends) and are not replaced.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SPOT_PREFERRED</code> - (default value) Spot Instances are used whenever available in the game server
         * group. If Spot Instances are unavailable, the game server group continues to provide hosting capacity by
         * falling back to On-Demand Instances. Existing nonviable Spot Instances are terminated (after current gameplay
         * ends) and are replaced with new On-Demand Instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ON_DEMAND_ONLY</code> - Only On-Demand Instances are used in the game server group. No Spot Instances
         * are used, even when available, while this balancing strategy is in force.
         * </p>
         * </li>
         * </ul>
         * 
         * @param balancingStrategy
         *        Indicates how GameLift FleetIQ balances the use of Spot Instances and On-Demand Instances in the game
         *        server group. Method options include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SPOT_ONLY</code> - Only Spot Instances are used in the game server group. If Spot Instances are
         *        unavailable or not viable for game hosting, the game server group provides no hosting capacity until
         *        Spot Instances can again be used. Until then, no new instances are started, and the existing nonviable
         *        Spot Instances are terminated (after current gameplay ends) and are not replaced.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SPOT_PREFERRED</code> - (default value) Spot Instances are used whenever available in the game
         *        server group. If Spot Instances are unavailable, the game server group continues to provide hosting
         *        capacity by falling back to On-Demand Instances. Existing nonviable Spot Instances are terminated
         *        (after current gameplay ends) and are replaced with new On-Demand Instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ON_DEMAND_ONLY</code> - Only On-Demand Instances are used in the game server group. No Spot
         *        Instances are used, even when available, while this balancing strategy is in force.
         *        </p>
         *        </li>
         * @see BalancingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BalancingStrategy
         */
        Builder balancingStrategy(String balancingStrategy);

        /**
         * <p>
         * Indicates how GameLift FleetIQ balances the use of Spot Instances and On-Demand Instances in the game server
         * group. Method options include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SPOT_ONLY</code> - Only Spot Instances are used in the game server group. If Spot Instances are
         * unavailable or not viable for game hosting, the game server group provides no hosting capacity until Spot
         * Instances can again be used. Until then, no new instances are started, and the existing nonviable Spot
         * Instances are terminated (after current gameplay ends) and are not replaced.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SPOT_PREFERRED</code> - (default value) Spot Instances are used whenever available in the game server
         * group. If Spot Instances are unavailable, the game server group continues to provide hosting capacity by
         * falling back to On-Demand Instances. Existing nonviable Spot Instances are terminated (after current gameplay
         * ends) and are replaced with new On-Demand Instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ON_DEMAND_ONLY</code> - Only On-Demand Instances are used in the game server group. No Spot Instances
         * are used, even when available, while this balancing strategy is in force.
         * </p>
         * </li>
         * </ul>
         * 
         * @param balancingStrategy
         *        Indicates how GameLift FleetIQ balances the use of Spot Instances and On-Demand Instances in the game
         *        server group. Method options include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SPOT_ONLY</code> - Only Spot Instances are used in the game server group. If Spot Instances are
         *        unavailable or not viable for game hosting, the game server group provides no hosting capacity until
         *        Spot Instances can again be used. Until then, no new instances are started, and the existing nonviable
         *        Spot Instances are terminated (after current gameplay ends) and are not replaced.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SPOT_PREFERRED</code> - (default value) Spot Instances are used whenever available in the game
         *        server group. If Spot Instances are unavailable, the game server group continues to provide hosting
         *        capacity by falling back to On-Demand Instances. Existing nonviable Spot Instances are terminated
         *        (after current gameplay ends) and are replaced with new On-Demand Instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ON_DEMAND_ONLY</code> - Only On-Demand Instances are used in the game server group. No Spot
         *        Instances are used, even when available, while this balancing strategy is in force.
         *        </p>
         *        </li>
         * @see BalancingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BalancingStrategy
         */
        Builder balancingStrategy(BalancingStrategy balancingStrategy);

        /**
         * <p>
         * A flag that indicates whether instances in the game server group are protected from early termination.
         * Unprotected instances that have active game servers running might be terminated during a scale-down event,
         * causing players to be dropped from the game. Protected instances cannot be terminated while there are active
         * game servers running except in the event of a forced game server group deletion (see ). An exception to this
         * is with Spot Instances, which can be terminated by Amazon Web Services regardless of protection status.
         * </p>
         * 
         * @param gameServerProtectionPolicy
         *        A flag that indicates whether instances in the game server group are protected from early termination.
         *        Unprotected instances that have active game servers running might be terminated during a scale-down
         *        event, causing players to be dropped from the game. Protected instances cannot be terminated while
         *        there are active game servers running except in the event of a forced game server group deletion (see
         *        ). An exception to this is with Spot Instances, which can be terminated by Amazon Web Services
         *        regardless of protection status.
         * @see GameServerProtectionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerProtectionPolicy
         */
        Builder gameServerProtectionPolicy(String gameServerProtectionPolicy);

        /**
         * <p>
         * A flag that indicates whether instances in the game server group are protected from early termination.
         * Unprotected instances that have active game servers running might be terminated during a scale-down event,
         * causing players to be dropped from the game. Protected instances cannot be terminated while there are active
         * game servers running except in the event of a forced game server group deletion (see ). An exception to this
         * is with Spot Instances, which can be terminated by Amazon Web Services regardless of protection status.
         * </p>
         * 
         * @param gameServerProtectionPolicy
         *        A flag that indicates whether instances in the game server group are protected from early termination.
         *        Unprotected instances that have active game servers running might be terminated during a scale-down
         *        event, causing players to be dropped from the game. Protected instances cannot be terminated while
         *        there are active game servers running except in the event of a forced game server group deletion (see
         *        ). An exception to this is with Spot Instances, which can be terminated by Amazon Web Services
         *        regardless of protection status.
         * @see GameServerProtectionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerProtectionPolicy
         */
        Builder gameServerProtectionPolicy(GameServerProtectionPolicy gameServerProtectionPolicy);

        /**
         * <p>
         * A generated unique ID for the Amazon EC2 Auto Scaling group that is associated with this game server group.
         * </p>
         * 
         * @param autoScalingGroupArn
         *        A generated unique ID for the Amazon EC2 Auto Scaling group that is associated with this game server
         *        group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroupArn(String autoScalingGroupArn);

        /**
         * <p>
         * The current status of the game server group. Possible statuses include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NEW</code> - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVATING</code> - GameLift FleetIQ is setting up a game server group, which includes creating an Auto
         * Scaling group in your Amazon Web Services account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The game server group has been successfully created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE_SCHEDULED</code> - A request to delete the game server group has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request
         * and is processing it. GameLift FleetIQ must first complete and release hosts before it deletes the Auto
         * Scaling group and the game server group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETED</code> - The game server group has been successfully deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ERROR</code> - The asynchronous processes of activating or deleting a game server group has failed,
         * resulting in an error state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the game server group. Possible statuses include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NEW</code> - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVATING</code> - GameLift FleetIQ is setting up a game server group, which includes creating
         *        an Auto Scaling group in your Amazon Web Services account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The game server group has been successfully created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE_SCHEDULED</code> - A request to delete the game server group has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code>
         *        request and is processing it. GameLift FleetIQ must first complete and release hosts before it deletes
         *        the Auto Scaling group and the game server group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETED</code> - The game server group has been successfully deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ERROR</code> - The asynchronous processes of activating or deleting a game server group has
         *        failed, resulting in an error state.
         *        </p>
         *        </li>
         * @see GameServerGroupStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerGroupStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the game server group. Possible statuses include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NEW</code> - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVATING</code> - GameLift FleetIQ is setting up a game server group, which includes creating an Auto
         * Scaling group in your Amazon Web Services account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The game server group has been successfully created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE_SCHEDULED</code> - A request to delete the game server group has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request
         * and is processing it. GameLift FleetIQ must first complete and release hosts before it deletes the Auto
         * Scaling group and the game server group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETED</code> - The game server group has been successfully deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ERROR</code> - The asynchronous processes of activating or deleting a game server group has failed,
         * resulting in an error state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the game server group. Possible statuses include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NEW</code> - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVATING</code> - GameLift FleetIQ is setting up a game server group, which includes creating
         *        an Auto Scaling group in your Amazon Web Services account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The game server group has been successfully created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE_SCHEDULED</code> - A request to delete the game server group has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code>
         *        request and is processing it. GameLift FleetIQ must first complete and release hosts before it deletes
         *        the Auto Scaling group and the game server group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETED</code> - The game server group has been successfully deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ERROR</code> - The asynchronous processes of activating or deleting a game server group has
         *        failed, resulting in an error state.
         *        </p>
         *        </li>
         * @see GameServerGroupStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerGroupStatus
         */
        Builder status(GameServerGroupStatus status);

        /**
         * <p>
         * Additional information about the current game server group status. This information might provide additional
         * insight on groups that are in <code>ERROR</code> status.
         * </p>
         * 
         * @param statusReason
         *        Additional information about the current game server group status. This information might provide
         *        additional insight on groups that are in <code>ERROR</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * A list of activities that are currently suspended for this game server group. If this property is empty, all
         * activities are occurring.
         * </p>
         * 
         * @param suspendedActions
         *        A list of activities that are currently suspended for this game server group. If this property is
         *        empty, all activities are occurring.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedActionsWithStrings(Collection<String> suspendedActions);

        /**
         * <p>
         * A list of activities that are currently suspended for this game server group. If this property is empty, all
         * activities are occurring.
         * </p>
         * 
         * @param suspendedActions
         *        A list of activities that are currently suspended for this game server group. If this property is
         *        empty, all activities are occurring.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedActionsWithStrings(String... suspendedActions);

        /**
         * <p>
         * A list of activities that are currently suspended for this game server group. If this property is empty, all
         * activities are occurring.
         * </p>
         * 
         * @param suspendedActions
         *        A list of activities that are currently suspended for this game server group. If this property is
         *        empty, all activities are occurring.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedActions(Collection<GameServerGroupAction> suspendedActions);

        /**
         * <p>
         * A list of activities that are currently suspended for this game server group. If this property is empty, all
         * activities are occurring.
         * </p>
         * 
         * @param suspendedActions
         *        A list of activities that are currently suspended for this game server group. If this property is
         *        empty, all activities are occurring.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedActions(GameServerGroupAction... suspendedActions);

        /**
         * <p>
         * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param creationTime
         *        A time stamp indicating when this data object was created. Format is a number expressed in Unix time
         *        as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A timestamp that indicates when this game server group was last updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        A timestamp that indicates when this game server group was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);
    }

    static final class BuilderImpl implements Builder {
        private String gameServerGroupName;

        private String gameServerGroupArn;

        private String roleArn;

        private List<InstanceDefinition> instanceDefinitions = DefaultSdkAutoConstructList.getInstance();

        private String balancingStrategy;

        private String gameServerProtectionPolicy;

        private String autoScalingGroupArn;

        private String status;

        private String statusReason;

        private List<String> suspendedActions = DefaultSdkAutoConstructList.getInstance();

        private Instant creationTime;

        private Instant lastUpdatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(GameServerGroup model) {
            gameServerGroupName(model.gameServerGroupName);
            gameServerGroupArn(model.gameServerGroupArn);
            roleArn(model.roleArn);
            instanceDefinitions(model.instanceDefinitions);
            balancingStrategy(model.balancingStrategy);
            gameServerProtectionPolicy(model.gameServerProtectionPolicy);
            autoScalingGroupArn(model.autoScalingGroupArn);
            status(model.status);
            statusReason(model.statusReason);
            suspendedActionsWithStrings(model.suspendedActions);
            creationTime(model.creationTime);
            lastUpdatedTime(model.lastUpdatedTime);
        }

        public final String getGameServerGroupName() {
            return gameServerGroupName;
        }

        public final void setGameServerGroupName(String gameServerGroupName) {
            this.gameServerGroupName = gameServerGroupName;
        }

        @Override
        public final Builder gameServerGroupName(String gameServerGroupName) {
            this.gameServerGroupName = gameServerGroupName;
            return this;
        }

        public final String getGameServerGroupArn() {
            return gameServerGroupArn;
        }

        public final void setGameServerGroupArn(String gameServerGroupArn) {
            this.gameServerGroupArn = gameServerGroupArn;
        }

        @Override
        public final Builder gameServerGroupArn(String gameServerGroupArn) {
            this.gameServerGroupArn = gameServerGroupArn;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final List<InstanceDefinition.Builder> getInstanceDefinitions() {
            List<InstanceDefinition.Builder> result = InstanceDefinitionsCopier.copyToBuilder(this.instanceDefinitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstanceDefinitions(Collection<InstanceDefinition.BuilderImpl> instanceDefinitions) {
            this.instanceDefinitions = InstanceDefinitionsCopier.copyFromBuilder(instanceDefinitions);
        }

        @Override
        public final Builder instanceDefinitions(Collection<InstanceDefinition> instanceDefinitions) {
            this.instanceDefinitions = InstanceDefinitionsCopier.copy(instanceDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceDefinitions(InstanceDefinition... instanceDefinitions) {
            instanceDefinitions(Arrays.asList(instanceDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceDefinitions(Consumer<InstanceDefinition.Builder>... instanceDefinitions) {
            instanceDefinitions(Stream.of(instanceDefinitions).map(c -> InstanceDefinition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getBalancingStrategy() {
            return balancingStrategy;
        }

        public final void setBalancingStrategy(String balancingStrategy) {
            this.balancingStrategy = balancingStrategy;
        }

        @Override
        public final Builder balancingStrategy(String balancingStrategy) {
            this.balancingStrategy = balancingStrategy;
            return this;
        }

        @Override
        public final Builder balancingStrategy(BalancingStrategy balancingStrategy) {
            this.balancingStrategy(balancingStrategy == null ? null : balancingStrategy.toString());
            return this;
        }

        public final String getGameServerProtectionPolicy() {
            return gameServerProtectionPolicy;
        }

        public final void setGameServerProtectionPolicy(String gameServerProtectionPolicy) {
            this.gameServerProtectionPolicy = gameServerProtectionPolicy;
        }

        @Override
        public final Builder gameServerProtectionPolicy(String gameServerProtectionPolicy) {
            this.gameServerProtectionPolicy = gameServerProtectionPolicy;
            return this;
        }

        @Override
        public final Builder gameServerProtectionPolicy(GameServerProtectionPolicy gameServerProtectionPolicy) {
            this.gameServerProtectionPolicy(gameServerProtectionPolicy == null ? null : gameServerProtectionPolicy.toString());
            return this;
        }

        public final String getAutoScalingGroupArn() {
            return autoScalingGroupArn;
        }

        public final void setAutoScalingGroupArn(String autoScalingGroupArn) {
            this.autoScalingGroupArn = autoScalingGroupArn;
        }

        @Override
        public final Builder autoScalingGroupArn(String autoScalingGroupArn) {
            this.autoScalingGroupArn = autoScalingGroupArn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(GameServerGroupStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final Collection<String> getSuspendedActions() {
            if (suspendedActions instanceof SdkAutoConstructList) {
                return null;
            }
            return suspendedActions;
        }

        public final void setSuspendedActions(Collection<String> suspendedActions) {
            this.suspendedActions = GameServerGroupActionsCopier.copy(suspendedActions);
        }

        @Override
        public final Builder suspendedActionsWithStrings(Collection<String> suspendedActions) {
            this.suspendedActions = GameServerGroupActionsCopier.copy(suspendedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder suspendedActionsWithStrings(String... suspendedActions) {
            suspendedActionsWithStrings(Arrays.asList(suspendedActions));
            return this;
        }

        @Override
        public final Builder suspendedActions(Collection<GameServerGroupAction> suspendedActions) {
            this.suspendedActions = GameServerGroupActionsCopier.copyEnumToString(suspendedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder suspendedActions(GameServerGroupAction... suspendedActions) {
            suspendedActions(Arrays.asList(suspendedActions));
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        @Override
        public GameServerGroup build() {
            return new GameServerGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
