/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.health;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.health.model.ConcurrentModificationException;
import software.amazon.awssdk.services.health.model.DescribeAffectedAccountsForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DescribeAffectedAccountsForOrganizationResponse;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesForOrganizationResponse;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest;
import software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse;
import software.amazon.awssdk.services.health.model.DescribeEntityAggregatesRequest;
import software.amazon.awssdk.services.health.model.DescribeEntityAggregatesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest;
import software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsForOrganizationResponse;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsRequest;
import software.amazon.awssdk.services.health.model.DescribeEventDetailsResponse;
import software.amazon.awssdk.services.health.model.DescribeEventTypesRequest;
import software.amazon.awssdk.services.health.model.DescribeEventTypesResponse;
import software.amazon.awssdk.services.health.model.DescribeEventsForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DescribeEventsForOrganizationResponse;
import software.amazon.awssdk.services.health.model.DescribeEventsRequest;
import software.amazon.awssdk.services.health.model.DescribeEventsResponse;
import software.amazon.awssdk.services.health.model.DescribeHealthServiceStatusForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DescribeHealthServiceStatusForOrganizationResponse;
import software.amazon.awssdk.services.health.model.DisableHealthServiceAccessForOrganizationRequest;
import software.amazon.awssdk.services.health.model.DisableHealthServiceAccessForOrganizationResponse;
import software.amazon.awssdk.services.health.model.EnableHealthServiceAccessForOrganizationRequest;
import software.amazon.awssdk.services.health.model.EnableHealthServiceAccessForOrganizationResponse;
import software.amazon.awssdk.services.health.model.HealthException;
import software.amazon.awssdk.services.health.model.HealthRequest;
import software.amazon.awssdk.services.health.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.health.model.UnsupportedLocaleException;
import software.amazon.awssdk.services.health.paginators.DescribeAffectedAccountsForOrganizationPublisher;
import software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesForOrganizationPublisher;
import software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesPublisher;
import software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesPublisher;
import software.amazon.awssdk.services.health.paginators.DescribeEventTypesPublisher;
import software.amazon.awssdk.services.health.paginators.DescribeEventsForOrganizationPublisher;
import software.amazon.awssdk.services.health.paginators.DescribeEventsPublisher;
import software.amazon.awssdk.services.health.transform.DescribeAffectedAccountsForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeAffectedEntitiesForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeAffectedEntitiesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEntityAggregatesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventAggregatesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventDetailsForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventDetailsRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventTypesRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventsForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeEventsRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DescribeHealthServiceStatusForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.DisableHealthServiceAccessForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.health.transform.EnableHealthServiceAccessForOrganizationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link HealthAsyncClient}.
 *
 * @see HealthAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultHealthAsyncClient implements HealthAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultHealthAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultHealthAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Returns a list of accounts in the organization from AWS Organizations that are affected by the provided event.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable AWS Health to work with AWS Organizations. To do this,
     * call the <a>EnableHealthServiceAccessForOrganization</a> operation from your organization's master account.
     * </p>
     *
     * @param describeAffectedAccountsForOrganizationRequest
     * @return A Java Future containing the result of the DescribeAffectedAccountsForOrganization operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeAffectedAccountsForOrganization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedAccountsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAffectedAccountsForOrganizationResponse> describeAffectedAccountsForOrganization(
            DescribeAffectedAccountsForOrganizationRequest describeAffectedAccountsForOrganizationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAffectedAccountsForOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeAffectedAccountsForOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAffectedAccountsForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAffectedAccountsForOrganizationRequest, DescribeAffectedAccountsForOrganizationResponse>()
                            .withOperationName("DescribeAffectedAccountsForOrganization")
                            .withMarshaller(new DescribeAffectedAccountsForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAffectedAccountsForOrganizationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of accounts in the organization from AWS Organizations that are affected by the provided event.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable AWS Health to work with AWS Organizations. To do this,
     * call the <a>EnableHealthServiceAccessForOrganization</a> operation from your organization's master account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAffectedAccountsForOrganization(software.amazon.awssdk.services.health.model.DescribeAffectedAccountsForOrganizationRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedAccountsForOrganizationPublisher publisher = client.describeAffectedAccountsForOrganizationPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedAccountsForOrganizationPublisher publisher = client.describeAffectedAccountsForOrganizationPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeAffectedAccountsForOrganizationResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeAffectedAccountsForOrganizationResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAffectedAccountsForOrganization(software.amazon.awssdk.services.health.model.DescribeAffectedAccountsForOrganizationRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAffectedAccountsForOrganizationRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeAffectedAccountsForOrganization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedAccountsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeAffectedAccountsForOrganizationPublisher describeAffectedAccountsForOrganizationPaginator(
            DescribeAffectedAccountsForOrganizationRequest describeAffectedAccountsForOrganizationRequest) {
        return new DescribeAffectedAccountsForOrganizationPublisher(this,
                applyPaginatorUserAgent(describeAffectedAccountsForOrganizationRequest));
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by the specified events, based on the specified filter
     * criteria. Entities can refer to individual customer resources, groups of customer resources, or any other
     * construct, depending on the AWS service. Events that have impact beyond that of the affected entities, or where
     * the extent of impact is unknown, include at least one entity indicating this.
     * </p>
     * <p>
     * At least one event ARN is required. Results are sorted by the <code>lastUpdatedTime</code> of the entity,
     * starting with the most recent.
     * </p>
     *
     * @param describeAffectedEntitiesRequest
     * @return A Java Future containing the result of the DescribeAffectedEntities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeAffectedEntities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAffectedEntitiesResponse> describeAffectedEntities(
            DescribeAffectedEntitiesRequest describeAffectedEntitiesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAffectedEntitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAffectedEntitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAffectedEntitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAffectedEntitiesRequest, DescribeAffectedEntitiesResponse>()
                            .withOperationName("DescribeAffectedEntities")
                            .withMarshaller(new DescribeAffectedEntitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAffectedEntitiesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by one or more events for one or more accounts in your
     * organization in AWS Organizations, based on the filter criteria. Entities can refer to individual customer
     * resources, groups of customer resources, or any other construct, depending on the AWS service.
     * </p>
     * <p>
     * At least one event ARN and account ID are required. Results are sorted by the <code>lastUpdatedTime</code> of the
     * entity, starting with the most recent.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable AWS Health to work with AWS Organizations. To do this,
     * call the <a>EnableHealthServiceAccessForOrganization</a> operation from your organization's master account.
     * </p>
     *
     * @param describeAffectedEntitiesForOrganizationRequest
     * @return A Java Future containing the result of the DescribeAffectedEntitiesForOrganization operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeAffectedEntitiesForOrganization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntitiesForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAffectedEntitiesForOrganizationResponse> describeAffectedEntitiesForOrganization(
            DescribeAffectedEntitiesForOrganizationRequest describeAffectedEntitiesForOrganizationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAffectedEntitiesForOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeAffectedEntitiesForOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAffectedEntitiesForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAffectedEntitiesForOrganizationRequest, DescribeAffectedEntitiesForOrganizationResponse>()
                            .withOperationName("DescribeAffectedEntitiesForOrganization")
                            .withMarshaller(new DescribeAffectedEntitiesForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAffectedEntitiesForOrganizationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by one or more events for one or more accounts in your
     * organization in AWS Organizations, based on the filter criteria. Entities can refer to individual customer
     * resources, groups of customer resources, or any other construct, depending on the AWS service.
     * </p>
     * <p>
     * At least one event ARN and account ID are required. Results are sorted by the <code>lastUpdatedTime</code> of the
     * entity, starting with the most recent.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable AWS Health to work with AWS Organizations. To do this,
     * call the <a>EnableHealthServiceAccessForOrganization</a> operation from your organization's master account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAffectedEntitiesForOrganization(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesForOrganizationRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesForOrganizationPublisher publisher = client.describeAffectedEntitiesForOrganizationPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesForOrganizationPublisher publisher = client.describeAffectedEntitiesForOrganizationPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesForOrganizationResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesForOrganizationResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAffectedEntitiesForOrganization(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesForOrganizationRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAffectedEntitiesForOrganizationRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeAffectedEntitiesForOrganization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntitiesForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeAffectedEntitiesForOrganizationPublisher describeAffectedEntitiesForOrganizationPaginator(
            DescribeAffectedEntitiesForOrganizationRequest describeAffectedEntitiesForOrganizationRequest) {
        return new DescribeAffectedEntitiesForOrganizationPublisher(this,
                applyPaginatorUserAgent(describeAffectedEntitiesForOrganizationRequest));
    }

    /**
     * <p>
     * Returns a list of entities that have been affected by the specified events, based on the specified filter
     * criteria. Entities can refer to individual customer resources, groups of customer resources, or any other
     * construct, depending on the AWS service. Events that have impact beyond that of the affected entities, or where
     * the extent of impact is unknown, include at least one entity indicating this.
     * </p>
     * <p>
     * At least one event ARN is required. Results are sorted by the <code>lastUpdatedTime</code> of the entity,
     * starting with the most recent.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAffectedEntities(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesPublisher publisher = client.describeAffectedEntitiesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeAffectedEntitiesPublisher publisher = client.describeAffectedEntitiesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAffectedEntities(software.amazon.awssdk.services.health.model.DescribeAffectedEntitiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAffectedEntitiesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeAffectedEntities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeAffectedEntities"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeAffectedEntitiesPublisher describeAffectedEntitiesPaginator(
            DescribeAffectedEntitiesRequest describeAffectedEntitiesRequest) {
        return new DescribeAffectedEntitiesPublisher(this, applyPaginatorUserAgent(describeAffectedEntitiesRequest));
    }

    /**
     * <p>
     * Returns the number of entities that are affected by each of the specified events. If no events are specified, the
     * counts of all affected entities are returned.
     * </p>
     *
     * @param describeEntityAggregatesRequest
     * @return A Java Future containing the result of the DescribeEntityAggregates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEntityAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEntityAggregates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEntityAggregatesResponse> describeEntityAggregates(
            DescribeEntityAggregatesRequest describeEntityAggregatesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEntityAggregatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEntityAggregatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEntityAggregatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEntityAggregatesRequest, DescribeEntityAggregatesResponse>()
                            .withOperationName("DescribeEntityAggregates")
                            .withMarshaller(new DescribeEntityAggregatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEntityAggregatesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the number of events of each event type (issue, scheduled change, and account notification). If no filter
     * is specified, the counts of all events in each category are returned.
     * </p>
     *
     * @param describeEventAggregatesRequest
     * @return A Java Future containing the result of the DescribeEventAggregates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventAggregates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventAggregatesResponse> describeEventAggregates(
            DescribeEventAggregatesRequest describeEventAggregatesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventAggregatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventAggregatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventAggregatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventAggregatesRequest, DescribeEventAggregatesResponse>()
                            .withOperationName("DescribeEventAggregates")
                            .withMarshaller(new DescribeEventAggregatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEventAggregatesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the number of events of each event type (issue, scheduled change, and account notification). If no filter
     * is specified, the counts of all events in each category are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEventAggregates(software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesPublisher publisher = client.describeEventAggregatesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventAggregatesPublisher publisher = client.describeEventAggregatesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventAggregatesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventAggregates(software.amazon.awssdk.services.health.model.DescribeEventAggregatesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEventAggregatesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventAggregates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventAggregates" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeEventAggregatesPublisher describeEventAggregatesPaginator(
            DescribeEventAggregatesRequest describeEventAggregatesRequest) {
        return new DescribeEventAggregatesPublisher(this, applyPaginatorUserAgent(describeEventAggregatesRequest));
    }

    /**
     * <p>
     * Returns detailed information about one or more specified events. Information includes standard event data
     * (region, service, and so on, as returned by <a>DescribeEvents</a>), a detailed event description, and possible
     * additional metadata that depends upon the nature of the event. Affected entities are not included; to retrieve
     * those, use the <a>DescribeAffectedEntities</a> operation.
     * </p>
     * <p>
     * If a specified event cannot be retrieved, an error message is returned for that event.
     * </p>
     *
     * @param describeEventDetailsRequest
     * @return A Java Future containing the result of the DescribeEventDetails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventDetails" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventDetailsResponse> describeEventDetails(
            DescribeEventDetailsRequest describeEventDetailsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventDetailsRequest, DescribeEventDetailsResponse>()
                            .withOperationName("DescribeEventDetails")
                            .withMarshaller(new DescribeEventDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEventDetailsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed information about one or more specified events for one or more accounts in your organization.
     * Information includes standard event data (Region, service, and so on, as returned by
     * <a>DescribeEventsForOrganization</a>, a detailed event description, and possible additional metadata that depends
     * upon the nature of the event. Affected entities are not included; to retrieve those, use the
     * <a>DescribeAffectedEntitiesForOrganization</a> operation.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable AWS Health to work with AWS Organizations. To do this,
     * call the <a>EnableHealthServiceAccessForOrganization</a> operation from your organization's master account.
     * </p>
     *
     * @param describeEventDetailsForOrganizationRequest
     * @return A Java Future containing the result of the DescribeEventDetailsForOrganization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventDetailsForOrganization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventDetailsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventDetailsForOrganizationResponse> describeEventDetailsForOrganization(
            DescribeEventDetailsForOrganizationRequest describeEventDetailsForOrganizationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventDetailsForOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeEventDetailsForOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventDetailsForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventDetailsForOrganizationRequest, DescribeEventDetailsForOrganizationResponse>()
                            .withOperationName("DescribeEventDetailsForOrganization")
                            .withMarshaller(new DescribeEventDetailsForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEventDetailsForOrganizationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. If no filter criteria are specified, all event
     * types are returned, in no particular order.
     * </p>
     *
     * @param describeEventTypesRequest
     * @return A Java Future containing the result of the DescribeEventTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventTypesResponse> describeEventTypes(DescribeEventTypesRequest describeEventTypesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventTypesRequest, DescribeEventTypesResponse>()
                            .withOperationName("DescribeEventTypes")
                            .withMarshaller(new DescribeEventTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEventTypesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the event types that meet the specified filter criteria. If no filter criteria are specified, all event
     * types are returned, in no particular order.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesPublisher publisher = client.describeEventTypesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventTypesPublisher publisher = client.describeEventTypesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventTypesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventTypesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventTypes(software.amazon.awssdk.services.health.model.DescribeEventTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEventTypesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventTypes" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeEventTypesPublisher describeEventTypesPaginator(DescribeEventTypesRequest describeEventTypesRequest) {
        return new DescribeEventTypesPublisher(this, applyPaginatorUserAgent(describeEventTypesRequest));
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a>DescribeEventDetails</a> and
     * <a>DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent.
     * </p>
     *
     * @param describeEventsRequest
     * @return A Java Future containing the result of the DescribeEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventsResponse> describeEvents(DescribeEventsRequest describeEventsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventsRequest, DescribeEventsResponse>()
                            .withOperationName("DescribeEvents")
                            .withMarshaller(new DescribeEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEventsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about events across your organization in AWS Organizations, meeting the specified filter
     * criteria. Events are returned in a summary form and do not include the accounts impacted, detailed description,
     * any additional metadata that depends on the event type, or any affected resources. To retrieve that information,
     * use the <a>DescribeAffectedAccountsForOrganization</a>, <a>DescribeEventDetailsForOrganization</a>, and
     * <a>DescribeAffectedEntitiesForOrganization</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events across your organization are returned. Results are sorted by
     * <code>lastModifiedTime</code>, starting with the most recent.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable Health to work with AWS Organizations. To do this, call
     * the <a>EnableHealthServiceAccessForOrganization</a> operation from your organization's master account.
     * </p>
     *
     * @param describeEventsForOrganizationRequest
     * @return A Java Future containing the result of the DescribeEventsForOrganization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventsForOrganization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventsForOrganizationResponse> describeEventsForOrganization(
            DescribeEventsForOrganizationRequest describeEventsForOrganizationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventsForOrganizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventsForOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventsForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventsForOrganizationRequest, DescribeEventsForOrganizationResponse>()
                            .withOperationName("DescribeEventsForOrganization")
                            .withMarshaller(new DescribeEventsForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEventsForOrganizationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about events across your organization in AWS Organizations, meeting the specified filter
     * criteria. Events are returned in a summary form and do not include the accounts impacted, detailed description,
     * any additional metadata that depends on the event type, or any affected resources. To retrieve that information,
     * use the <a>DescribeAffectedAccountsForOrganization</a>, <a>DescribeEventDetailsForOrganization</a>, and
     * <a>DescribeAffectedEntitiesForOrganization</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events across your organization are returned. Results are sorted by
     * <code>lastModifiedTime</code>, starting with the most recent.
     * </p>
     * <p>
     * Before you can call this operation, you must first enable Health to work with AWS Organizations. To do this, call
     * the <a>EnableHealthServiceAccessForOrganization</a> operation from your organization's master account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeEventsForOrganization(software.amazon.awssdk.services.health.model.DescribeEventsForOrganizationRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsForOrganizationPublisher publisher = client.describeEventsForOrganizationPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsForOrganizationPublisher publisher = client.describeEventsForOrganizationPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventsForOrganizationResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventsForOrganizationResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEventsForOrganization(software.amazon.awssdk.services.health.model.DescribeEventsForOrganizationRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeEventsForOrganizationRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEventsForOrganization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEventsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeEventsForOrganizationPublisher describeEventsForOrganizationPaginator(
            DescribeEventsForOrganizationRequest describeEventsForOrganizationRequest) {
        return new DescribeEventsForOrganizationPublisher(this, applyPaginatorUserAgent(describeEventsForOrganizationRequest));
    }

    /**
     * <p>
     * Returns information about events that meet the specified filter criteria. Events are returned in a summary form
     * and do not include the detailed description, any additional metadata that depends on the event type, or any
     * affected resources. To retrieve that information, use the <a>DescribeEventDetails</a> and
     * <a>DescribeAffectedEntities</a> operations.
     * </p>
     * <p>
     * If no filter criteria are specified, all events are returned. Results are sorted by <code>lastModifiedTime</code>
     * , starting with the most recent.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsPublisher publisher = client.describeEventsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.health.paginators.DescribeEventsPublisher publisher = client.describeEventsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.health.model.DescribeEventsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.health.model.DescribeEventsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeEvents(software.amazon.awssdk.services.health.model.DescribeEventsRequest)} operation.</b>
     * </p>
     *
     * @param describeEventsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPaginationTokenException The specified pagination token (<code>nextToken</code>) is not valid.
     *         </li>
     *         <li>UnsupportedLocaleException The specified locale is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeEventsPublisher describeEventsPaginator(DescribeEventsRequest describeEventsRequest) {
        return new DescribeEventsPublisher(this, applyPaginatorUserAgent(describeEventsRequest));
    }

    /**
     * <p>
     * This operation provides status information on enabling or disabling AWS Health to work with your organization. To
     * call this operation, you must sign in as an IAM user, assume an IAM role, or sign in as the root user (not
     * recommended) in the organization's master account.
     * </p>
     *
     * @param describeHealthServiceStatusForOrganizationRequest
     * @return A Java Future containing the result of the DescribeHealthServiceStatusForOrganization operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DescribeHealthServiceStatusForOrganization
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DescribeHealthServiceStatusForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeHealthServiceStatusForOrganizationResponse> describeHealthServiceStatusForOrganization(
            DescribeHealthServiceStatusForOrganizationRequest describeHealthServiceStatusForOrganizationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeHealthServiceStatusForOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeHealthServiceStatusForOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeHealthServiceStatusForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeHealthServiceStatusForOrganizationRequest, DescribeHealthServiceStatusForOrganizationResponse>()
                            .withOperationName("DescribeHealthServiceStatusForOrganization")
                            .withMarshaller(new DescribeHealthServiceStatusForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeHealthServiceStatusForOrganizationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Calling this operation disables Health from working with AWS Organizations. This does not remove the Service
     * Linked Role (SLR) from the the master account in your organization. Use the IAM console, API, or AWS CLI to
     * remove the SLR if desired. To call this operation, you must sign in as an IAM user, assume an IAM role, or sign
     * in as the root user (not recommended) in the organization's master account.
     * </p>
     *
     * @param disableHealthServiceAccessForOrganizationRequest
     * @return A Java Future containing the result of the DisableHealthServiceAccessForOrganization operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException <a>EnableHealthServiceAccessForOrganization</a> is already in
     *         progress. Wait for the action to complete before trying again. To get the current status, use the
     *         <a>DescribeHealthServiceStatusForOrganization</a> operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.DisableHealthServiceAccessForOrganization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/DisableHealthServiceAccessForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableHealthServiceAccessForOrganizationResponse> disableHealthServiceAccessForOrganization(
            DisableHealthServiceAccessForOrganizationRequest disableHealthServiceAccessForOrganizationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableHealthServiceAccessForOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisableHealthServiceAccessForOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableHealthServiceAccessForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableHealthServiceAccessForOrganizationRequest, DisableHealthServiceAccessForOrganizationResponse>()
                            .withOperationName("DisableHealthServiceAccessForOrganization")
                            .withMarshaller(new DisableHealthServiceAccessForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disableHealthServiceAccessForOrganizationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Calling this operation enables AWS Health to work with AWS Organizations. This applies a Service Linked Role
     * (SLR) to the master account in the organization. To learn more about the steps in this process, visit enabling
     * service access for AWS Health in AWS Organizations. To call this operation, you must sign in as an IAM user,
     * assume an IAM role, or sign in as the root user (not recommended) in the organization's master account.
     * </p>
     *
     * @param enableHealthServiceAccessForOrganizationRequest
     * @return A Java Future containing the result of the EnableHealthServiceAccessForOrganization operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException <a>EnableHealthServiceAccessForOrganization</a> is already in
     *         progress. Wait for the action to complete before trying again. To get the current status, use the
     *         <a>DescribeHealthServiceStatusForOrganization</a> operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>HealthException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample HealthAsyncClient.EnableHealthServiceAccessForOrganization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/health-2016-08-04/EnableHealthServiceAccessForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableHealthServiceAccessForOrganizationResponse> enableHealthServiceAccessForOrganization(
            EnableHealthServiceAccessForOrganizationRequest enableHealthServiceAccessForOrganizationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableHealthServiceAccessForOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, EnableHealthServiceAccessForOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableHealthServiceAccessForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableHealthServiceAccessForOrganizationRequest, EnableHealthServiceAccessForOrganizationResponse>()
                            .withOperationName("EnableHealthServiceAccessForOrganization")
                            .withMarshaller(new EnableHealthServiceAccessForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(enableHealthServiceAccessForOrganizationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(HealthException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedLocale")
                                .exceptionBuilderSupplier(UnsupportedLocaleException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationToken")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
    }

    private <T extends HealthRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
