/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotevents;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotevents.model.CreateDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.CreateDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.CreateInputRequest;
import software.amazon.awssdk.services.iotevents.model.CreateInputResponse;
import software.amazon.awssdk.services.iotevents.model.DeleteDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.DeleteDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.DeleteInputRequest;
import software.amazon.awssdk.services.iotevents.model.DeleteInputResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeInputRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeInputResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeLoggingOptionsRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeLoggingOptionsResponse;
import software.amazon.awssdk.services.iotevents.model.InternalFailureException;
import software.amazon.awssdk.services.iotevents.model.InvalidRequestException;
import software.amazon.awssdk.services.iotevents.model.IotEventsException;
import software.amazon.awssdk.services.iotevents.model.LimitExceededException;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelVersionsRequest;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelVersionsResponse;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelsRequest;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelsResponse;
import software.amazon.awssdk.services.iotevents.model.ListInputsRequest;
import software.amazon.awssdk.services.iotevents.model.ListInputsResponse;
import software.amazon.awssdk.services.iotevents.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.iotevents.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.iotevents.model.PutLoggingOptionsRequest;
import software.amazon.awssdk.services.iotevents.model.PutLoggingOptionsResponse;
import software.amazon.awssdk.services.iotevents.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.iotevents.model.ResourceInUseException;
import software.amazon.awssdk.services.iotevents.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotevents.model.ServiceUnavailableException;
import software.amazon.awssdk.services.iotevents.model.TagResourceRequest;
import software.amazon.awssdk.services.iotevents.model.TagResourceResponse;
import software.amazon.awssdk.services.iotevents.model.ThrottlingException;
import software.amazon.awssdk.services.iotevents.model.UnsupportedOperationException;
import software.amazon.awssdk.services.iotevents.model.UntagResourceRequest;
import software.amazon.awssdk.services.iotevents.model.UntagResourceResponse;
import software.amazon.awssdk.services.iotevents.model.UpdateDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.UpdateDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.UpdateInputRequest;
import software.amazon.awssdk.services.iotevents.model.UpdateInputResponse;
import software.amazon.awssdk.services.iotevents.transform.CreateDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.CreateInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DeleteDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DeleteInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeLoggingOptionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListDetectorModelVersionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListDetectorModelsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListInputsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.PutLoggingOptionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UpdateDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UpdateInputRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IotEventsAsyncClient}.
 *
 * @see IotEventsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotEventsAsyncClient implements IotEventsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIotEventsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotEventsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a detector model.
     * </p>
     *
     * @param createDetectorModelRequest
     * @return A Java Future containing the result of the CreateDetectorModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>LimitExceededException A limit was exceeded.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.CreateDetectorModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/CreateDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDetectorModelResponse> createDetectorModel(
            CreateDetectorModelRequest createDetectorModelRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDetectorModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDetectorModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDetectorModelRequest, CreateDetectorModelResponse>()
                            .withOperationName("CreateDetectorModel")
                            .withMarshaller(new CreateDetectorModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDetectorModelRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an input.
     * </p>
     *
     * @param createInputRequest
     * @return A Java Future containing the result of the CreateInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ResourceAlreadyExistsException The resource already exists.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.CreateInput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/CreateInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateInputResponse> createInput(CreateInputRequest createInputRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateInputRequest, CreateInputResponse>()
                            .withOperationName("CreateInput").withMarshaller(new CreateInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createInputRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a detector model. Any active instances of the detector model are also deleted.
     * </p>
     *
     * @param deleteDetectorModelRequest
     * @return A Java Future containing the result of the DeleteDetectorModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DeleteDetectorModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DeleteDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDetectorModelResponse> deleteDetectorModel(
            DeleteDetectorModelRequest deleteDetectorModelRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDetectorModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDetectorModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDetectorModelRequest, DeleteDetectorModelResponse>()
                            .withOperationName("DeleteDetectorModel")
                            .withMarshaller(new DeleteDetectorModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDetectorModelRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an input.
     * </p>
     *
     * @param deleteInputRequest
     * @return A Java Future containing the result of the DeleteInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DeleteInput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DeleteInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteInputResponse> deleteInput(DeleteInputRequest deleteInputRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInputRequest, DeleteInputResponse>()
                            .withOperationName("DeleteInput").withMarshaller(new DeleteInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteInputRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a detector model. If the <code>"version"</code> parameter is not specified, information about the
     * latest version is returned.
     * </p>
     *
     * @param describeDetectorModelRequest
     * @return A Java Future containing the result of the DescribeDetectorModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DescribeDetectorModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeDetectorModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDetectorModelResponse> describeDetectorModel(
            DescribeDetectorModelRequest describeDetectorModelRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDetectorModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDetectorModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDetectorModelRequest, DescribeDetectorModelResponse>()
                            .withOperationName("DescribeDetectorModel")
                            .withMarshaller(new DescribeDetectorModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDetectorModelRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an input.
     * </p>
     *
     * @param describeInputRequest
     * @return A Java Future containing the result of the DescribeInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DescribeInput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeInputResponse> describeInput(DescribeInputRequest describeInputRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInputRequest, DescribeInputResponse>()
                            .withOperationName("DescribeInput")
                            .withMarshaller(new DescribeInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeInputRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current settings of the AWS IoT Events logging options.
     * </p>
     *
     * @param describeLoggingOptionsRequest
     * @return A Java Future containing the result of the DescribeLoggingOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.DescribeLoggingOptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeLoggingOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLoggingOptionsResponse> describeLoggingOptions(
            DescribeLoggingOptionsRequest describeLoggingOptionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLoggingOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLoggingOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLoggingOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLoggingOptionsRequest, DescribeLoggingOptionsResponse>()
                            .withOperationName("DescribeLoggingOptions")
                            .withMarshaller(new DescribeLoggingOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeLoggingOptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the versions of a detector model. Only the metadata associated with each detector model version is
     * returned.
     * </p>
     *
     * @param listDetectorModelVersionsRequest
     * @return A Java Future containing the result of the ListDetectorModelVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.ListDetectorModelVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListDetectorModelVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDetectorModelVersionsResponse> listDetectorModelVersions(
            ListDetectorModelVersionsRequest listDetectorModelVersionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDetectorModelVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDetectorModelVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDetectorModelVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDetectorModelVersionsRequest, ListDetectorModelVersionsResponse>()
                            .withOperationName("ListDetectorModelVersions")
                            .withMarshaller(new ListDetectorModelVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDetectorModelVersionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the detector models you have created. Only the metadata associated with each detector model is returned.
     * </p>
     *
     * @param listDetectorModelsRequest
     * @return A Java Future containing the result of the ListDetectorModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.ListDetectorModels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListDetectorModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDetectorModelsResponse> listDetectorModels(ListDetectorModelsRequest listDetectorModelsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDetectorModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDetectorModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDetectorModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDetectorModelsRequest, ListDetectorModelsResponse>()
                            .withOperationName("ListDetectorModels")
                            .withMarshaller(new ListDetectorModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDetectorModelsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the inputs you have created.
     * </p>
     *
     * @param listInputsRequest
     * @return A Java Future containing the result of the ListInputs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.ListInputs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListInputs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListInputsResponse> listInputs(ListInputsRequest listInputsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInputsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListInputsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInputsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInputsRequest, ListInputsResponse>().withOperationName("ListInputs")
                            .withMarshaller(new ListInputsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listInputsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags (metadata) you have assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets or updates the AWS IoT Events logging options.
     * </p>
     * <p>
     * If you update the value of any <code>"loggingOptions"</code> field, it takes up to one minute for the change to
     * take effect. Also, if you change the policy attached to the role you specified in the <code>"roleArn"</code>
     * field (for example, to correct an invalid policy) it takes up to five minutes for that change to take effect.
     * </p>
     *
     * @param putLoggingOptionsRequest
     * @return A Java Future containing the result of the PutLoggingOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.PutLoggingOptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/PutLoggingOptions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutLoggingOptionsResponse> putLoggingOptions(PutLoggingOptionsRequest putLoggingOptionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutLoggingOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutLoggingOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutLoggingOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutLoggingOptionsRequest, PutLoggingOptionsResponse>()
                            .withOperationName("PutLoggingOptions")
                            .withMarshaller(new PutLoggingOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putLoggingOptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds to or modifies the tags of the given resource. Tags are metadata that can be used to manage a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>LimitExceededException A limit was exceeded.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the given tags (metadata) from the resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a detector model. Detectors (instances) spawned by the previous version are deleted and then re-created
     * as new inputs arrive.
     * </p>
     *
     * @param updateDetectorModelRequest
     * @return A Java Future containing the result of the UpdateDetectorModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.UpdateDetectorModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UpdateDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDetectorModelResponse> updateDetectorModel(
            UpdateDetectorModelRequest updateDetectorModelRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDetectorModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDetectorModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDetectorModelRequest, UpdateDetectorModelResponse>()
                            .withOperationName("UpdateDetectorModel")
                            .withMarshaller(new UpdateDetectorModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDetectorModelRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an input.
     * </p>
     *
     * @param updateInputRequest
     * @return A Java Future containing the result of the UpdateInput operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request was invalid.</li>
     *         <li>ThrottlingException The request could not be completed due to throttling.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>InternalFailureException An internal failure occurred.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ResourceInUseException The resource is in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotEventsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotEventsAsyncClient.UpdateInput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UpdateInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateInputResponse> updateInput(UpdateInputRequest updateInputRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateInputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateInputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateInputRequest, UpdateInputResponse>()
                            .withOperationName("UpdateInput").withMarshaller(new UpdateInputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateInputRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotEventsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).httpStatusCode(501).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(410).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
