/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotevents;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotevents.model.CreateDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.CreateDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.CreateInputRequest;
import software.amazon.awssdk.services.iotevents.model.CreateInputResponse;
import software.amazon.awssdk.services.iotevents.model.DeleteDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.DeleteDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.DeleteInputRequest;
import software.amazon.awssdk.services.iotevents.model.DeleteInputResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeInputRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeInputResponse;
import software.amazon.awssdk.services.iotevents.model.DescribeLoggingOptionsRequest;
import software.amazon.awssdk.services.iotevents.model.DescribeLoggingOptionsResponse;
import software.amazon.awssdk.services.iotevents.model.InternalFailureException;
import software.amazon.awssdk.services.iotevents.model.InvalidRequestException;
import software.amazon.awssdk.services.iotevents.model.IotEventsException;
import software.amazon.awssdk.services.iotevents.model.LimitExceededException;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelVersionsRequest;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelVersionsResponse;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelsRequest;
import software.amazon.awssdk.services.iotevents.model.ListDetectorModelsResponse;
import software.amazon.awssdk.services.iotevents.model.ListInputsRequest;
import software.amazon.awssdk.services.iotevents.model.ListInputsResponse;
import software.amazon.awssdk.services.iotevents.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.iotevents.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.iotevents.model.PutLoggingOptionsRequest;
import software.amazon.awssdk.services.iotevents.model.PutLoggingOptionsResponse;
import software.amazon.awssdk.services.iotevents.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.iotevents.model.ResourceInUseException;
import software.amazon.awssdk.services.iotevents.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotevents.model.ServiceUnavailableException;
import software.amazon.awssdk.services.iotevents.model.TagResourceRequest;
import software.amazon.awssdk.services.iotevents.model.TagResourceResponse;
import software.amazon.awssdk.services.iotevents.model.ThrottlingException;
import software.amazon.awssdk.services.iotevents.model.UnsupportedOperationException;
import software.amazon.awssdk.services.iotevents.model.UntagResourceRequest;
import software.amazon.awssdk.services.iotevents.model.UntagResourceResponse;
import software.amazon.awssdk.services.iotevents.model.UpdateDetectorModelRequest;
import software.amazon.awssdk.services.iotevents.model.UpdateDetectorModelResponse;
import software.amazon.awssdk.services.iotevents.model.UpdateInputRequest;
import software.amazon.awssdk.services.iotevents.model.UpdateInputResponse;
import software.amazon.awssdk.services.iotevents.transform.CreateDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.CreateInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DeleteDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DeleteInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeInputRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.DescribeLoggingOptionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListDetectorModelVersionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListDetectorModelsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListInputsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.PutLoggingOptionsRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UpdateDetectorModelRequestMarshaller;
import software.amazon.awssdk.services.iotevents.transform.UpdateInputRequestMarshaller;

/**
 * Internal implementation of {@link IotEventsClient}.
 *
 * @see IotEventsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotEventsClient implements IotEventsClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotEventsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a detector model.
     * </p>
     *
     * @param createDetectorModelRequest
     * @return Result of the CreateDetectorModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ResourceAlreadyExistsException
     *         The resource already exists.
     * @throws LimitExceededException
     *         A limit was exceeded.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.CreateDetectorModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/CreateDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateDetectorModelResponse createDetectorModel(CreateDetectorModelRequest createDetectorModelRequest)
            throws InvalidRequestException, ResourceInUseException, ResourceAlreadyExistsException, LimitExceededException,
            ThrottlingException, InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException,
            IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDetectorModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateDetectorModelRequest, CreateDetectorModelResponse>()
                .withOperationName("CreateDetectorModel").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createDetectorModelRequest)
                .withMarshaller(new CreateDetectorModelRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates an input.
     * </p>
     *
     * @param createInputRequest
     * @return Result of the CreateInput operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ResourceAlreadyExistsException
     *         The resource already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.CreateInput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/CreateInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateInputResponse createInput(CreateInputRequest createInputRequest) throws InvalidRequestException,
            ThrottlingException, InternalFailureException, ServiceUnavailableException, ResourceAlreadyExistsException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateInputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateInputRequest, CreateInputResponse>()
                .withOperationName("CreateInput").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createInputRequest)
                .withMarshaller(new CreateInputRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a detector model. Any active instances of the detector model are also deleted.
     * </p>
     *
     * @param deleteDetectorModelRequest
     * @return Result of the DeleteDetectorModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DeleteDetectorModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DeleteDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteDetectorModelResponse deleteDetectorModel(DeleteDetectorModelRequest deleteDetectorModelRequest)
            throws InvalidRequestException, ResourceInUseException, ResourceNotFoundException, ThrottlingException,
            InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDetectorModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteDetectorModelRequest, DeleteDetectorModelResponse>()
                .withOperationName("DeleteDetectorModel").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteDetectorModelRequest)
                .withMarshaller(new DeleteDetectorModelRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes an input.
     * </p>
     *
     * @param deleteInputRequest
     * @return Result of the DeleteInput operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DeleteInput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DeleteInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteInputResponse deleteInput(DeleteInputRequest deleteInputRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            ResourceInUseException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteInputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteInputRequest, DeleteInputResponse>()
                .withOperationName("DeleteInput").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteInputRequest)
                .withMarshaller(new DeleteInputRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes a detector model. If the <code>"version"</code> parameter is not specified, information about the
     * latest version is returned.
     * </p>
     *
     * @param describeDetectorModelRequest
     * @return Result of the DescribeDetectorModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DescribeDetectorModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeDetectorModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDetectorModelResponse describeDetectorModel(DescribeDetectorModelRequest describeDetectorModelRequest)
            throws InvalidRequestException, ResourceNotFoundException, ThrottlingException, InternalFailureException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDetectorModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeDetectorModelRequest, DescribeDetectorModelResponse>()
                .withOperationName("DescribeDetectorModel").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeDetectorModelRequest)
                .withMarshaller(new DescribeDetectorModelRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes an input.
     * </p>
     *
     * @param describeInputRequest
     * @return Result of the DescribeInput operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DescribeInput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeInputResponse describeInput(DescribeInputRequest describeInputRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeInputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeInputRequest, DescribeInputResponse>()
                .withOperationName("DescribeInput").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeInputRequest)
                .withMarshaller(new DescribeInputRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the current settings of the AWS IoT Events logging options.
     * </p>
     *
     * @param describeLoggingOptionsRequest
     * @return Result of the DescribeLoggingOptions operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.DescribeLoggingOptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/DescribeLoggingOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLoggingOptionsResponse describeLoggingOptions(DescribeLoggingOptionsRequest describeLoggingOptionsRequest)
            throws InvalidRequestException, ThrottlingException, InternalFailureException, ResourceNotFoundException,
            ServiceUnavailableException, UnsupportedOperationException, AwsServiceException, SdkClientException,
            IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeLoggingOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeLoggingOptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeLoggingOptionsRequest, DescribeLoggingOptionsResponse>()
                .withOperationName("DescribeLoggingOptions").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeLoggingOptionsRequest)
                .withMarshaller(new DescribeLoggingOptionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all the versions of a detector model. Only the metadata associated with each detector model version is
     * returned.
     * </p>
     *
     * @param listDetectorModelVersionsRequest
     * @return Result of the ListDetectorModelVersions operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.ListDetectorModelVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListDetectorModelVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDetectorModelVersionsResponse listDetectorModelVersions(
            ListDetectorModelVersionsRequest listDetectorModelVersionsRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDetectorModelVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDetectorModelVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListDetectorModelVersionsRequest, ListDetectorModelVersionsResponse>()
                        .withOperationName("ListDetectorModelVersions").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listDetectorModelVersionsRequest)
                        .withMarshaller(new ListDetectorModelVersionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the detector models you have created. Only the metadata associated with each detector model is returned.
     * </p>
     *
     * @param listDetectorModelsRequest
     * @return Result of the ListDetectorModels operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.ListDetectorModels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListDetectorModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDetectorModelsResponse listDetectorModels(ListDetectorModelsRequest listDetectorModelsRequest)
            throws InvalidRequestException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDetectorModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDetectorModelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListDetectorModelsRequest, ListDetectorModelsResponse>()
                .withOperationName("ListDetectorModels").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listDetectorModelsRequest)
                .withMarshaller(new ListDetectorModelsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the inputs you have created.
     * </p>
     *
     * @param listInputsRequest
     * @return Result of the ListInputs operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.ListInputs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListInputs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListInputsResponse listInputs(ListInputsRequest listInputsRequest) throws InvalidRequestException,
            ThrottlingException, InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException,
            IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInputsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInputsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListInputsRequest, ListInputsResponse>()
                .withOperationName("ListInputs").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listInputsRequest)
                .withMarshaller(new ListInputsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the tags (metadata) you have assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidRequestException, ResourceNotFoundException, ResourceInUseException, ThrottlingException,
            InternalFailureException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sets or updates the AWS IoT Events logging options.
     * </p>
     * <p>
     * If you update the value of any <code>"loggingOptions"</code> field, it takes up to one minute for the change to
     * take effect. Also, if you change the policy attached to the role you specified in the <code>"roleArn"</code>
     * field (for example, to correct an invalid policy) it takes up to five minutes for that change to take effect.
     * </p>
     *
     * @param putLoggingOptionsRequest
     * @return Result of the PutLoggingOptions operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.PutLoggingOptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/PutLoggingOptions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutLoggingOptionsResponse putLoggingOptions(PutLoggingOptionsRequest putLoggingOptionsRequest)
            throws InvalidRequestException, ThrottlingException, InternalFailureException, ServiceUnavailableException,
            UnsupportedOperationException, ResourceInUseException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutLoggingOptionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutLoggingOptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutLoggingOptionsRequest, PutLoggingOptionsResponse>()
                .withOperationName("PutLoggingOptions").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putLoggingOptionsRequest)
                .withMarshaller(new PutLoggingOptionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds to or modifies the tags of the given resource. Tags are metadata that can be used to manage a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws LimitExceededException
     *         A limit was exceeded.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidRequestException,
            ResourceNotFoundException, ResourceInUseException, ThrottlingException, LimitExceededException,
            InternalFailureException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the given tags (metadata) from the resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidRequestException,
            ResourceNotFoundException, ResourceInUseException, ThrottlingException, InternalFailureException,
            AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates a detector model. Detectors (instances) spawned by the previous version are deleted and then re-created
     * as new inputs arrive.
     * </p>
     *
     * @param updateDetectorModelRequest
     * @return Result of the UpdateDetectorModel operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.UpdateDetectorModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UpdateDetectorModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateDetectorModelResponse updateDetectorModel(UpdateDetectorModelRequest updateDetectorModelRequest)
            throws InvalidRequestException, ResourceInUseException, ResourceNotFoundException, ThrottlingException,
            InternalFailureException, ServiceUnavailableException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDetectorModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDetectorModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateDetectorModelRequest, UpdateDetectorModelResponse>()
                .withOperationName("UpdateDetectorModel").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateDetectorModelRequest)
                .withMarshaller(new UpdateDetectorModelRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates an input.
     * </p>
     *
     * @param updateInputRequest
     * @return Result of the UpdateInput operation returned by the service.
     * @throws InvalidRequestException
     *         The request was invalid.
     * @throws ThrottlingException
     *         The request could not be completed due to throttling.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws InternalFailureException
     *         An internal failure occurred.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ResourceInUseException
     *         The resource is in use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotEventsClient.UpdateInput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iotevents-2018-07-27/UpdateInput" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateInputResponse updateInput(UpdateInputRequest updateInputRequest) throws InvalidRequestException,
            ThrottlingException, ResourceNotFoundException, InternalFailureException, ServiceUnavailableException,
            ResourceInUseException, AwsServiceException, SdkClientException, IotEventsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateInputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateInputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateInputRequest, UpdateInputResponse>()
                .withOperationName("UpdateInput").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateInputRequest)
                .withMarshaller(new UpdateInputRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotEventsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).httpStatusCode(501).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(410).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
