/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackagevod;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediapackagevod.model.CreateAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreateAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.CreatePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeleteAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeleteAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DeletePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribeAssetRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribeAssetResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingConfigurationRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingConfigurationResponse;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingGroupRequest;
import software.amazon.awssdk.services.mediapackagevod.model.DescribePackagingGroupResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ForbiddenException;
import software.amazon.awssdk.services.mediapackagevod.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediapackagevod.model.ListAssetsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListAssetsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsRequest;
import software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsResponse;
import software.amazon.awssdk.services.mediapackagevod.model.MediaPackageVodException;
import software.amazon.awssdk.services.mediapackagevod.model.MediaPackageVodRequest;
import software.amazon.awssdk.services.mediapackagevod.model.NotFoundException;
import software.amazon.awssdk.services.mediapackagevod.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mediapackagevod.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediapackagevod.model.UnprocessableEntityException;
import software.amazon.awssdk.services.mediapackagevod.paginators.ListAssetsIterable;
import software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingConfigurationsIterable;
import software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingGroupsIterable;
import software.amazon.awssdk.services.mediapackagevod.transform.CreateAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.CreatePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.CreatePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeleteAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeletePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DeletePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribeAssetRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribePackagingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.DescribePackagingGroupRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListAssetsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListPackagingConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.mediapackagevod.transform.ListPackagingGroupsRequestMarshaller;

/**
 * Internal implementation of {@link MediaPackageVodClient}.
 *
 * @see MediaPackageVodClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaPackageVodClient implements MediaPackageVodClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaPackageVodClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Creates a new MediaPackage VOD Asset resource.
     *
     * @param createAssetRequest
     *        A new MediaPackage VOD Asset configuration.
     * @return Result of the CreateAsset operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.CreateAsset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreateAsset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateAssetResponse createAsset(CreateAssetRequest createAssetRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAssetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateAssetRequest, CreateAssetResponse>()
                .withOperationName("CreateAsset").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createAssetRequest)
                .withMarshaller(new CreateAssetRequestMarshaller(protocolFactory)));
    }

    /**
     * Creates a new MediaPackage VOD PackagingConfiguration resource.
     *
     * @param createPackagingConfigurationRequest
     *        A new MediaPackage VOD PackagingConfiguration resource configuration.
     * @return Result of the CreatePackagingConfiguration operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.CreatePackagingConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreatePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePackagingConfigurationResponse createPackagingConfiguration(
            CreatePackagingConfigurationRequest createPackagingConfigurationRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePackagingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CreatePackagingConfigurationRequest, CreatePackagingConfigurationResponse>()
                        .withOperationName("CreatePackagingConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(createPackagingConfigurationRequest)
                        .withMarshaller(new CreatePackagingConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * Creates a new MediaPackage VOD PackagingGroup resource.
     *
     * @param createPackagingGroupRequest
     *        A new MediaPackage VOD PackagingGroup resource configuration.
     * @return Result of the CreatePackagingGroup operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.CreatePackagingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/CreatePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePackagingGroupResponse createPackagingGroup(CreatePackagingGroupRequest createPackagingGroupRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePackagingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreatePackagingGroupRequest, CreatePackagingGroupResponse>()
                .withOperationName("CreatePackagingGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createPackagingGroupRequest)
                .withMarshaller(new CreatePackagingGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * Deletes an existing MediaPackage VOD Asset resource.
     *
     * @param deleteAssetRequest
     * @return Result of the DeleteAsset operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DeleteAsset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeleteAsset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteAssetResponse deleteAsset(DeleteAssetRequest deleteAssetRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAssetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteAssetRequest, DeleteAssetResponse>()
                .withOperationName("DeleteAsset").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteAssetRequest)
                .withMarshaller(new DeleteAssetRequestMarshaller(protocolFactory)));
    }

    /**
     * Deletes a MediaPackage VOD PackagingConfiguration resource.
     *
     * @param deletePackagingConfigurationRequest
     * @return Result of the DeletePackagingConfiguration operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DeletePackagingConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeletePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeletePackagingConfigurationResponse deletePackagingConfiguration(
            DeletePackagingConfigurationRequest deletePackagingConfigurationRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePackagingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeletePackagingConfigurationRequest, DeletePackagingConfigurationResponse>()
                        .withOperationName("DeletePackagingConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deletePackagingConfigurationRequest)
                        .withMarshaller(new DeletePackagingConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * Deletes a MediaPackage VOD PackagingGroup resource.
     *
     * @param deletePackagingGroupRequest
     * @return Result of the DeletePackagingGroup operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DeletePackagingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DeletePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeletePackagingGroupResponse deletePackagingGroup(DeletePackagingGroupRequest deletePackagingGroupRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePackagingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeletePackagingGroupRequest, DeletePackagingGroupResponse>()
                .withOperationName("DeletePackagingGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deletePackagingGroupRequest)
                .withMarshaller(new DeletePackagingGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * Returns a description of a MediaPackage VOD Asset resource.
     *
     * @param describeAssetRequest
     * @return Result of the DescribeAsset operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DescribeAsset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribeAsset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeAssetResponse describeAsset(DescribeAssetRequest describeAssetRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeAssetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeAssetRequest, DescribeAssetResponse>()
                .withOperationName("DescribeAsset").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeAssetRequest)
                .withMarshaller(new DescribeAssetRequestMarshaller(protocolFactory)));
    }

    /**
     * Returns a description of a MediaPackage VOD PackagingConfiguration resource.
     *
     * @param describePackagingConfigurationRequest
     * @return Result of the DescribePackagingConfiguration operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DescribePackagingConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribePackagingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribePackagingConfigurationResponse describePackagingConfiguration(
            DescribePackagingConfigurationRequest describePackagingConfigurationRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePackagingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribePackagingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribePackagingConfigurationRequest, DescribePackagingConfigurationResponse>()
                        .withOperationName("DescribePackagingConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describePackagingConfigurationRequest)
                        .withMarshaller(new DescribePackagingConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * Returns a description of a MediaPackage VOD PackagingGroup resource.
     *
     * @param describePackagingGroupRequest
     * @return Result of the DescribePackagingGroup operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.DescribePackagingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/DescribePackagingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribePackagingGroupResponse describePackagingGroup(DescribePackagingGroupRequest describePackagingGroupRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePackagingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribePackagingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribePackagingGroupRequest, DescribePackagingGroupResponse>()
                .withOperationName("DescribePackagingGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describePackagingGroupRequest)
                .withMarshaller(new DescribePackagingGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * Returns a collection of MediaPackage VOD Asset resources.
     *
     * @param listAssetsRequest
     * @return Result of the ListAssets operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.ListAssets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListAssets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListAssetsResponse listAssets(ListAssetsRequest listAssetsRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAssetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAssetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListAssetsRequest, ListAssetsResponse>()
                .withOperationName("ListAssets").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listAssetsRequest)
                .withMarshaller(new ListAssetsRequestMarshaller(protocolFactory)));
    }

    /**
     * Returns a collection of MediaPackage VOD Asset resources.<br/>
     * <p>
     * This is a variant of {@link #listAssets(software.amazon.awssdk.services.mediapackagevod.model.ListAssetsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListAssetsIterable responses = client.listAssetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediapackagevod.paginators.ListAssetsIterable responses = client.listAssetsPaginator(request);
     *     for (software.amazon.awssdk.services.mediapackagevod.model.ListAssetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListAssetsIterable responses = client.listAssetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAssets(software.amazon.awssdk.services.mediapackagevod.model.ListAssetsRequest)} operation.</b>
     * </p>
     *
     * @param listAssetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.ListAssets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListAssets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListAssetsIterable listAssetsPaginator(ListAssetsRequest listAssetsRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        return new ListAssetsIterable(this, applyPaginatorUserAgent(listAssetsRequest));
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingConfiguration resources.
     *
     * @param listPackagingConfigurationsRequest
     * @return Result of the ListPackagingConfigurations operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.ListPackagingConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPackagingConfigurationsResponse listPackagingConfigurations(
            ListPackagingConfigurationsRequest listPackagingConfigurationsRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPackagingConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPackagingConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListPackagingConfigurationsRequest, ListPackagingConfigurationsResponse>()
                        .withOperationName("ListPackagingConfigurations").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listPackagingConfigurationsRequest)
                        .withMarshaller(new ListPackagingConfigurationsRequestMarshaller(protocolFactory)));
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingConfiguration resources.<br/>
     * <p>
     * This is a variant of
     * {@link #listPackagingConfigurations(software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingConfigurationsIterable responses = client.listPackagingConfigurationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingConfigurationsIterable responses = client
     *             .listPackagingConfigurationsPaginator(request);
     *     for (software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingConfigurationsIterable responses = client.listPackagingConfigurationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPackagingConfigurations(software.amazon.awssdk.services.mediapackagevod.model.ListPackagingConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPackagingConfigurationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.ListPackagingConfigurations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPackagingConfigurationsIterable listPackagingConfigurationsPaginator(
            ListPackagingConfigurationsRequest listPackagingConfigurationsRequest) throws UnprocessableEntityException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaPackageVodException {
        return new ListPackagingConfigurationsIterable(this, applyPaginatorUserAgent(listPackagingConfigurationsRequest));
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingGroup resources.
     *
     * @param listPackagingGroupsRequest
     * @return Result of the ListPackagingGroups operation returned by the service.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.ListPackagingGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPackagingGroupsResponse listPackagingGroups(ListPackagingGroupsRequest listPackagingGroupsRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaPackageVodException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPackagingGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPackagingGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListPackagingGroupsRequest, ListPackagingGroupsResponse>()
                .withOperationName("ListPackagingGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listPackagingGroupsRequest)
                .withMarshaller(new ListPackagingGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * Returns a collection of MediaPackage VOD PackagingGroup resources.<br/>
     * <p>
     * This is a variant of
     * {@link #listPackagingGroups(software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingGroupsIterable responses = client.listPackagingGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingGroupsIterable responses = client
     *             .listPackagingGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediapackagevod.paginators.ListPackagingGroupsIterable responses = client.listPackagingGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPackagingGroups(software.amazon.awssdk.services.mediapackagevod.model.ListPackagingGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPackagingGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws UnprocessableEntityException
     *         The parameters sent in the request are not valid.
     * @throws InternalServerErrorException
     *         An unexpected error occurred.
     * @throws ForbiddenException
     *         The client is not authorized to access the requested resource.
     * @throws NotFoundException
     *         The requested resource does not exist.
     * @throws ServiceUnavailableException
     *         An unexpected error occurred.
     * @throws TooManyRequestsException
     *         The client has exceeded their resource or throttling limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaPackageVodException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaPackageVodClient.ListPackagingGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediapackage-vod-2018-11-07/ListPackagingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPackagingGroupsIterable listPackagingGroupsPaginator(ListPackagingGroupsRequest listPackagingGroupsRequest)
            throws UnprocessableEntityException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaPackageVodException {
        return new ListPackagingGroupsIterable(this, applyPaginatorUserAgent(listPackagingGroupsRequest));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaPackageVodException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnprocessableEntityException")
                                .exceptionBuilderSupplier(UnprocessableEntityException::builder).httpStatusCode(422).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MediaPackageVodRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
