/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediapackagevod.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * A configuration for accessing an external Secure Packager and Encoder Key Exchange (SPEKE) service that will provide
 * encryption keys.
 */
@Generated("software.amazon.awssdk:codegen")
public final class SpekeKeyProvider implements SdkPojo, Serializable,
        ToCopyableBuilder<SpekeKeyProvider.Builder, SpekeKeyProvider> {
    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(SpekeKeyProvider::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<List<String>> SYSTEM_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SystemIds")
            .getter(getter(SpekeKeyProvider::systemIds))
            .setter(setter(Builder::systemIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("systemIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Url")
            .getter(getter(SpekeKeyProvider::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("url").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ROLE_ARN_FIELD,
            SYSTEM_IDS_FIELD, URL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String roleArn;

    private final List<String> systemIds;

    private final String url;

    private SpekeKeyProvider(BuilderImpl builder) {
        this.roleArn = builder.roleArn;
        this.systemIds = builder.systemIds;
        this.url = builder.url;
    }

    /**
     * An Amazon Resource Name (ARN) of an IAM role that AWS Elemental MediaPackage will assume when accessing the key
     * provider service.
     *
     * @return An Amazon Resource Name (ARN) of an IAM role that AWS Elemental MediaPackage will assume when accessing
     *         the key provider service.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the SystemIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSystemIds() {
        return systemIds != null && !(systemIds instanceof SdkAutoConstructList);
    }

    /**
     * The system IDs to include in key requests.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSystemIds} method.
     * </p>
     * 
     * @return The system IDs to include in key requests.
     */
    public final List<String> systemIds() {
        return systemIds;
    }

    /**
     * The URL of the external key provider service.
     * 
     * @return The URL of the external key provider service.
     */
    public final String url() {
        return url;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasSystemIds() ? systemIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(url());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SpekeKeyProvider)) {
            return false;
        }
        SpekeKeyProvider other = (SpekeKeyProvider) obj;
        return Objects.equals(roleArn(), other.roleArn()) && hasSystemIds() == other.hasSystemIds()
                && Objects.equals(systemIds(), other.systemIds()) && Objects.equals(url(), other.url());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SpekeKeyProvider").add("RoleArn", roleArn())
                .add("SystemIds", hasSystemIds() ? systemIds() : null).add("Url", url()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "SystemIds":
            return Optional.ofNullable(clazz.cast(systemIds()));
        case "Url":
            return Optional.ofNullable(clazz.cast(url()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SpekeKeyProvider, T> g) {
        return obj -> g.apply((SpekeKeyProvider) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SpekeKeyProvider> {
        /**
         * An Amazon Resource Name (ARN) of an IAM role that AWS Elemental MediaPackage will assume when accessing the
         * key provider service.
         *
         * @param roleArn
         *        An Amazon Resource Name (ARN) of an IAM role that AWS Elemental MediaPackage will assume when
         *        accessing the key provider service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * The system IDs to include in key requests.
         * 
         * @param systemIds
         *        The system IDs to include in key requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder systemIds(Collection<String> systemIds);

        /**
         * The system IDs to include in key requests.
         * 
         * @param systemIds
         *        The system IDs to include in key requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder systemIds(String... systemIds);

        /**
         * The URL of the external key provider service.
         * 
         * @param url
         *        The URL of the external key provider service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);
    }

    static final class BuilderImpl implements Builder {
        private String roleArn;

        private List<String> systemIds = DefaultSdkAutoConstructList.getInstance();

        private String url;

        private BuilderImpl() {
        }

        private BuilderImpl(SpekeKeyProvider model) {
            roleArn(model.roleArn);
            systemIds(model.systemIds);
            url(model.url);
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final Collection<String> getSystemIds() {
            if (systemIds instanceof SdkAutoConstructList) {
                return null;
            }
            return systemIds;
        }

        public final void setSystemIds(Collection<String> systemIds) {
            this.systemIds = ___listOf__stringCopier.copy(systemIds);
        }

        @Override
        public final Builder systemIds(Collection<String> systemIds) {
            this.systemIds = ___listOf__stringCopier.copy(systemIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder systemIds(String... systemIds) {
            systemIds(Arrays.asList(systemIds));
            return this;
        }

        public final String getUrl() {
            return url;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        @Override
        public SpekeKeyProvider build() {
            return new SpekeKeyProvider(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
