/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ram.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a resource share in RAM.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceShare implements SdkPojo, Serializable, ToCopyableBuilder<ResourceShare.Builder, ResourceShare> {
    private static final SdkField<String> RESOURCE_SHARE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceShareArn").getter(getter(ResourceShare::resourceShareArn))
            .setter(setter(Builder::resourceShareArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceShareArn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ResourceShare::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> OWNING_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("owningAccountId").getter(getter(ResourceShare::owningAccountId))
            .setter(setter(Builder::owningAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("owningAccountId").build()).build();

    private static final SdkField<Boolean> ALLOW_EXTERNAL_PRINCIPALS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("allowExternalPrincipals").getter(getter(ResourceShare::allowExternalPrincipals))
            .setter(setter(Builder::allowExternalPrincipals))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allowExternalPrincipals").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(ResourceShare::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusMessage").getter(getter(ResourceShare::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusMessage").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(ResourceShare::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationTime").getter(getter(ResourceShare::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedTime").getter(getter(ResourceShare::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedTime").build()).build();

    private static final SdkField<String> FEATURE_SET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("featureSet").getter(getter(ResourceShare::featureSetAsString)).setter(setter(Builder::featureSet))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("featureSet").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_SHARE_ARN_FIELD,
            NAME_FIELD, OWNING_ACCOUNT_ID_FIELD, ALLOW_EXTERNAL_PRINCIPALS_FIELD, STATUS_FIELD, STATUS_MESSAGE_FIELD, TAGS_FIELD,
            CREATION_TIME_FIELD, LAST_UPDATED_TIME_FIELD, FEATURE_SET_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resourceShareArn;

    private final String name;

    private final String owningAccountId;

    private final Boolean allowExternalPrincipals;

    private final String status;

    private final String statusMessage;

    private final List<Tag> tags;

    private final Instant creationTime;

    private final Instant lastUpdatedTime;

    private final String featureSet;

    private ResourceShare(BuilderImpl builder) {
        this.resourceShareArn = builder.resourceShareArn;
        this.name = builder.name;
        this.owningAccountId = builder.owningAccountId;
        this.allowExternalPrincipals = builder.allowExternalPrincipals;
        this.status = builder.status;
        this.statusMessage = builder.statusMessage;
        this.tags = builder.tags;
        this.creationTime = builder.creationTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.featureSet = builder.featureSet;
    }

    /**
     * <p>
     * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
     * (ARN)</a> of the resource share
     * </p>
     * 
     * @return The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource
     *         Name (ARN)</a> of the resource share
     */
    public final String resourceShareArn() {
        return resourceShareArn;
    }

    /**
     * <p>
     * The name of the resource share.
     * </p>
     * 
     * @return The name of the resource share.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services account that owns the resource share.
     * </p>
     * 
     * @return The ID of the Amazon Web Services account that owns the resource share.
     */
    public final String owningAccountId() {
        return owningAccountId;
    }

    /**
     * <p>
     * Indicates whether principals outside your organization in Organizations can be associated with a resource share.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>True</code> – the resource share can be shared with any Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>False</code> – the resource share can be shared with only accounts in the same organization as the account
     * that owns the resource share.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Indicates whether principals outside your organization in Organizations can be associated with a resource
     *         share.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>True</code> – the resource share can be shared with any Amazon Web Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>False</code> – the resource share can be shared with only accounts in the same organization as the
     *         account that owns the resource share.
     *         </p>
     *         </li>
     */
    public final Boolean allowExternalPrincipals() {
        return allowExternalPrincipals;
    }

    /**
     * <p>
     * The current status of the resource share.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceShareStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the resource share.
     * @see ResourceShareStatus
     */
    public final ResourceShareStatus status() {
        return ResourceShareStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the resource share.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceShareStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the resource share.
     * @see ResourceShareStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A message about the status of the resource share.
     * </p>
     * 
     * @return A message about the status of the resource share.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tag key and value pairs attached to the resource share.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tag key and value pairs attached to the resource share.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The date and time when the resource share was created.
     * </p>
     * 
     * @return The date and time when the resource share was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The date and time when the resource share was last updated.
     * </p>
     * 
     * @return The date and time when the resource share was last updated.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * Indicates what features are available for this resource share. This parameter can have one of the following
     * values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to all
     * principals you share the resource share with. You can modify these resource shares in RAM using the console or
     * APIs. This resource share might have been created by RAM, or it might have been <b>CREATED_FROM_POLICY</b> and
     * then promoted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based policy. That
     * policy did not match any existing managed permissions, so RAM created this customer managed permission
     * automatically on the customer's behalf based on the attached policy document. This type of resource share is
     * visible only to the Amazon Web Services account that created it. You can't modify it in RAM unless you promote
     * it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but the
     * customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in progress. This value
     * changes to <code>STANDARD</code> when complete.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #featureSet} will
     * return {@link ResourceShareFeatureSet#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #featureSetAsString}.
     * </p>
     * 
     * @return Indicates what features are available for this resource share. This parameter can have one of the
     *         following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to
     *         all principals you share the resource share with. You can modify these resource shares in RAM using the
     *         console or APIs. This resource share might have been created by RAM, or it might have been
     *         <b>CREATED_FROM_POLICY</b> and then promoted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based
     *         policy. That policy did not match any existing managed permissions, so RAM created this customer managed
     *         permission automatically on the customer's behalf based on the attached policy document. This type of
     *         resource share is visible only to the Amazon Web Services account that created it. You can't modify it in
     *         RAM unless you promote it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but
     *         the customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in
     *         progress. This value changes to <code>STANDARD</code> when complete.
     *         </p>
     *         </li>
     * @see ResourceShareFeatureSet
     */
    public final ResourceShareFeatureSet featureSet() {
        return ResourceShareFeatureSet.fromValue(featureSet);
    }

    /**
     * <p>
     * Indicates what features are available for this resource share. This parameter can have one of the following
     * values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to all
     * principals you share the resource share with. You can modify these resource shares in RAM using the console or
     * APIs. This resource share might have been created by RAM, or it might have been <b>CREATED_FROM_POLICY</b> and
     * then promoted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based policy. That
     * policy did not match any existing managed permissions, so RAM created this customer managed permission
     * automatically on the customer's behalf based on the attached policy document. This type of resource share is
     * visible only to the Amazon Web Services account that created it. You can't modify it in RAM unless you promote
     * it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but the
     * customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in progress. This value
     * changes to <code>STANDARD</code> when complete.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #featureSet} will
     * return {@link ResourceShareFeatureSet#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #featureSetAsString}.
     * </p>
     * 
     * @return Indicates what features are available for this resource share. This parameter can have one of the
     *         following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to
     *         all principals you share the resource share with. You can modify these resource shares in RAM using the
     *         console or APIs. This resource share might have been created by RAM, or it might have been
     *         <b>CREATED_FROM_POLICY</b> and then promoted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based
     *         policy. That policy did not match any existing managed permissions, so RAM created this customer managed
     *         permission automatically on the customer's behalf based on the attached policy document. This type of
     *         resource share is visible only to the Amazon Web Services account that created it. You can't modify it in
     *         RAM unless you promote it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but
     *         the customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in
     *         progress. This value changes to <code>STANDARD</code> when complete.
     *         </p>
     *         </li>
     * @see ResourceShareFeatureSet
     */
    public final String featureSetAsString() {
        return featureSet;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceShareArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(owningAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(allowExternalPrincipals());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(featureSetAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceShare)) {
            return false;
        }
        ResourceShare other = (ResourceShare) obj;
        return Objects.equals(resourceShareArn(), other.resourceShareArn()) && Objects.equals(name(), other.name())
                && Objects.equals(owningAccountId(), other.owningAccountId())
                && Objects.equals(allowExternalPrincipals(), other.allowExternalPrincipals())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusMessage(), other.statusMessage()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(featureSetAsString(), other.featureSetAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceShare").add("ResourceShareArn", resourceShareArn()).add("Name", name())
                .add("OwningAccountId", owningAccountId()).add("AllowExternalPrincipals", allowExternalPrincipals())
                .add("Status", statusAsString()).add("StatusMessage", statusMessage()).add("Tags", hasTags() ? tags() : null)
                .add("CreationTime", creationTime()).add("LastUpdatedTime", lastUpdatedTime())
                .add("FeatureSet", featureSetAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "resourceShareArn":
            return Optional.ofNullable(clazz.cast(resourceShareArn()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "owningAccountId":
            return Optional.ofNullable(clazz.cast(owningAccountId()));
        case "allowExternalPrincipals":
            return Optional.ofNullable(clazz.cast(allowExternalPrincipals()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "creationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "lastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "featureSet":
            return Optional.ofNullable(clazz.cast(featureSetAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResourceShare, T> g) {
        return obj -> g.apply((ResourceShare) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceShare> {
        /**
         * <p>
         * The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Name
         * (ARN)</a> of the resource share
         * </p>
         * 
         * @param resourceShareArn
         *        The <a href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
         *        Resource Name (ARN)</a> of the resource share
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceShareArn(String resourceShareArn);

        /**
         * <p>
         * The name of the resource share.
         * </p>
         * 
         * @param name
         *        The name of the resource share.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of the Amazon Web Services account that owns the resource share.
         * </p>
         * 
         * @param owningAccountId
         *        The ID of the Amazon Web Services account that owns the resource share.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owningAccountId(String owningAccountId);

        /**
         * <p>
         * Indicates whether principals outside your organization in Organizations can be associated with a resource
         * share.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>True</code> – the resource share can be shared with any Amazon Web Services account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>False</code> – the resource share can be shared with only accounts in the same organization as the
         * account that owns the resource share.
         * </p>
         * </li>
         * </ul>
         * 
         * @param allowExternalPrincipals
         *        Indicates whether principals outside your organization in Organizations can be associated with a
         *        resource share.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>True</code> – the resource share can be shared with any Amazon Web Services account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>False</code> – the resource share can be shared with only accounts in the same organization as
         *        the account that owns the resource share.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowExternalPrincipals(Boolean allowExternalPrincipals);

        /**
         * <p>
         * The current status of the resource share.
         * </p>
         * 
         * @param status
         *        The current status of the resource share.
         * @see ResourceShareStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceShareStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the resource share.
         * </p>
         * 
         * @param status
         *        The current status of the resource share.
         * @see ResourceShareStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceShareStatus
         */
        Builder status(ResourceShareStatus status);

        /**
         * <p>
         * A message about the status of the resource share.
         * </p>
         * 
         * @param statusMessage
         *        A message about the status of the resource share.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * The tag key and value pairs attached to the resource share.
         * </p>
         * 
         * @param tags
         *        The tag key and value pairs attached to the resource share.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tag key and value pairs attached to the resource share.
         * </p>
         * 
         * @param tags
         *        The tag key and value pairs attached to the resource share.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tag key and value pairs attached to the resource share.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ram.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ram.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ram.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ram.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The date and time when the resource share was created.
         * </p>
         * 
         * @param creationTime
         *        The date and time when the resource share was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The date and time when the resource share was last updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The date and time when the resource share was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * Indicates what features are available for this resource share. This parameter can have one of the following
         * values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to all
         * principals you share the resource share with. You can modify these resource shares in RAM using the console
         * or APIs. This resource share might have been created by RAM, or it might have been <b>CREATED_FROM_POLICY</b>
         * and then promoted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based policy.
         * That policy did not match any existing managed permissions, so RAM created this customer managed permission
         * automatically on the customer's behalf based on the attached policy document. This type of resource share is
         * visible only to the Amazon Web Services account that created it. You can't modify it in RAM unless you
         * promote it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but the
         * customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in progress. This
         * value changes to <code>STANDARD</code> when complete.
         * </p>
         * </li>
         * </ul>
         * 
         * @param featureSet
         *        Indicates what features are available for this resource share. This parameter can have one of the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible
         *        to all principals you share the resource share with. You can modify these resource shares in RAM using
         *        the console or APIs. This resource share might have been created by RAM, or it might have been
         *        <b>CREATED_FROM_POLICY</b> and then promoted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based
         *        policy. That policy did not match any existing managed permissions, so RAM created this customer
         *        managed permission automatically on the customer's behalf based on the attached policy document. This
         *        type of resource share is visible only to the Amazon Web Services account that created it. You can't
         *        modify it in RAM unless you promote it. For more information, see
         *        <a>PromoteResourceShareCreatedFromPolicy</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>,
         *        but the customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in
         *        progress. This value changes to <code>STANDARD</code> when complete.
         *        </p>
         *        </li>
         * @see ResourceShareFeatureSet
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceShareFeatureSet
         */
        Builder featureSet(String featureSet);

        /**
         * <p>
         * Indicates what features are available for this resource share. This parameter can have one of the following
         * values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible to all
         * principals you share the resource share with. You can modify these resource shares in RAM using the console
         * or APIs. This resource share might have been created by RAM, or it might have been <b>CREATED_FROM_POLICY</b>
         * and then promoted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based policy.
         * That policy did not match any existing managed permissions, so RAM created this customer managed permission
         * automatically on the customer's behalf based on the attached policy document. This type of resource share is
         * visible only to the Amazon Web Services account that created it. You can't modify it in RAM unless you
         * promote it. For more information, see <a>PromoteResourceShareCreatedFromPolicy</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>, but the
         * customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in progress. This
         * value changes to <code>STANDARD</code> when complete.
         * </p>
         * </li>
         * </ul>
         * 
         * @param featureSet
         *        Indicates what features are available for this resource share. This parameter can have one of the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>STANDARD</b> – A resource share that supports all functionality. These resource shares are visible
         *        to all principals you share the resource share with. You can modify these resource shares in RAM using
         *        the console or APIs. This resource share might have been created by RAM, or it might have been
         *        <b>CREATED_FROM_POLICY</b> and then promoted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CREATED_FROM_POLICY</b> – The customer manually shared a resource by attaching a resource-based
         *        policy. That policy did not match any existing managed permissions, so RAM created this customer
         *        managed permission automatically on the customer's behalf based on the attached policy document. This
         *        type of resource share is visible only to the Amazon Web Services account that created it. You can't
         *        modify it in RAM unless you promote it. For more information, see
         *        <a>PromoteResourceShareCreatedFromPolicy</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PROMOTING_TO_STANDARD</b> – This resource share was originally <code>CREATED_FROM_POLICY</code>,
         *        but the customer ran the <a>PromoteResourceShareCreatedFromPolicy</a> and that operation is still in
         *        progress. This value changes to <code>STANDARD</code> when complete.
         *        </p>
         *        </li>
         * @see ResourceShareFeatureSet
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceShareFeatureSet
         */
        Builder featureSet(ResourceShareFeatureSet featureSet);
    }

    static final class BuilderImpl implements Builder {
        private String resourceShareArn;

        private String name;

        private String owningAccountId;

        private Boolean allowExternalPrincipals;

        private String status;

        private String statusMessage;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Instant creationTime;

        private Instant lastUpdatedTime;

        private String featureSet;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceShare model) {
            resourceShareArn(model.resourceShareArn);
            name(model.name);
            owningAccountId(model.owningAccountId);
            allowExternalPrincipals(model.allowExternalPrincipals);
            status(model.status);
            statusMessage(model.statusMessage);
            tags(model.tags);
            creationTime(model.creationTime);
            lastUpdatedTime(model.lastUpdatedTime);
            featureSet(model.featureSet);
        }

        public final String getResourceShareArn() {
            return resourceShareArn;
        }

        public final void setResourceShareArn(String resourceShareArn) {
            this.resourceShareArn = resourceShareArn;
        }

        @Override
        public final Builder resourceShareArn(String resourceShareArn) {
            this.resourceShareArn = resourceShareArn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getOwningAccountId() {
            return owningAccountId;
        }

        public final void setOwningAccountId(String owningAccountId) {
            this.owningAccountId = owningAccountId;
        }

        @Override
        public final Builder owningAccountId(String owningAccountId) {
            this.owningAccountId = owningAccountId;
            return this;
        }

        public final Boolean getAllowExternalPrincipals() {
            return allowExternalPrincipals;
        }

        public final void setAllowExternalPrincipals(Boolean allowExternalPrincipals) {
            this.allowExternalPrincipals = allowExternalPrincipals;
        }

        @Override
        public final Builder allowExternalPrincipals(Boolean allowExternalPrincipals) {
            this.allowExternalPrincipals = allowExternalPrincipals;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ResourceShareStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final String getFeatureSet() {
            return featureSet;
        }

        public final void setFeatureSet(String featureSet) {
            this.featureSet = featureSet;
        }

        @Override
        public final Builder featureSet(String featureSet) {
            this.featureSet = featureSet;
            return this;
        }

        @Override
        public final Builder featureSet(ResourceShareFeatureSet featureSet) {
            this.featureSet(featureSet == null ? null : featureSet.toString());
            return this;
        }

        @Override
        public ResourceShare build() {
            return new ResourceShare(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
