/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rekognition.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class DetectLabelsRequest extends RekognitionRequest implements
        ToCopyableBuilder<DetectLabelsRequest.Builder, DetectLabelsRequest> {
    private static final SdkField<Image> IMAGE_FIELD = SdkField.<Image> builder(MarshallingType.SDK_POJO).memberName("Image")
            .getter(getter(DetectLabelsRequest::image)).setter(setter(Builder::image)).constructor(Image::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Image").build()).build();

    private static final SdkField<Integer> MAX_LABELS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxLabels").getter(getter(DetectLabelsRequest::maxLabels)).setter(setter(Builder::maxLabels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxLabels").build()).build();

    private static final SdkField<Float> MIN_CONFIDENCE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("MinConfidence").getter(getter(DetectLabelsRequest::minConfidence))
            .setter(setter(Builder::minConfidence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinConfidence").build()).build();

    private static final SdkField<List<String>> FEATURES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Features")
            .getter(getter(DetectLabelsRequest::featuresAsStrings))
            .setter(setter(Builder::featuresWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Features").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DetectLabelsSettings> SETTINGS_FIELD = SdkField
            .<DetectLabelsSettings> builder(MarshallingType.SDK_POJO).memberName("Settings")
            .getter(getter(DetectLabelsRequest::settings)).setter(setter(Builder::settings))
            .constructor(DetectLabelsSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Settings").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_FIELD, MAX_LABELS_FIELD,
            MIN_CONFIDENCE_FIELD, FEATURES_FIELD, SETTINGS_FIELD));

    private final Image image;

    private final Integer maxLabels;

    private final Float minConfidence;

    private final List<String> features;

    private final DetectLabelsSettings settings;

    private DetectLabelsRequest(BuilderImpl builder) {
        super(builder);
        this.image = builder.image;
        this.maxLabels = builder.maxLabels;
        this.minConfidence = builder.minConfidence;
        this.features = builder.features;
        this.settings = builder.settings;
    }

    /**
     * <p>
     * The input image as base64-encoded bytes or an S3 object. If you use the AWS CLI to call Amazon Rekognition
     * operations, passing image bytes is not supported. Images stored in an S3 Bucket do not need to be base64-encoded.
     * </p>
     * <p>
     * If you are using an AWS SDK to call Amazon Rekognition, you might not need to base64-encode image bytes passed
     * using the <code>Bytes</code> field. For more information, see Images in the Amazon Rekognition developer guide.
     * </p>
     * 
     * @return The input image as base64-encoded bytes or an S3 object. If you use the AWS CLI to call Amazon
     *         Rekognition operations, passing image bytes is not supported. Images stored in an S3 Bucket do not need
     *         to be base64-encoded.</p>
     *         <p>
     *         If you are using an AWS SDK to call Amazon Rekognition, you might not need to base64-encode image bytes
     *         passed using the <code>Bytes</code> field. For more information, see Images in the Amazon Rekognition
     *         developer guide.
     */
    public final Image image() {
        return image;
    }

    /**
     * <p>
     * Maximum number of labels you want the service to return in the response. The service returns the specified number
     * of highest confidence labels. Only valid when GENERAL_LABELS is specified as a feature type in the Feature input
     * parameter.
     * </p>
     * 
     * @return Maximum number of labels you want the service to return in the response. The service returns the
     *         specified number of highest confidence labels. Only valid when GENERAL_LABELS is specified as a feature
     *         type in the Feature input parameter.
     */
    public final Integer maxLabels() {
        return maxLabels;
    }

    /**
     * <p>
     * Specifies the minimum confidence level for the labels to return. Amazon Rekognition doesn't return any labels
     * with confidence lower than this specified value.
     * </p>
     * <p>
     * If <code>MinConfidence</code> is not specified, the operation returns labels with a confidence values greater
     * than or equal to 55 percent. Only valid when GENERAL_LABELS is specified as a feature type in the Feature input
     * parameter.
     * </p>
     * 
     * @return Specifies the minimum confidence level for the labels to return. Amazon Rekognition doesn't return any
     *         labels with confidence lower than this specified value.</p>
     *         <p>
     *         If <code>MinConfidence</code> is not specified, the operation returns labels with a confidence values
     *         greater than or equal to 55 percent. Only valid when GENERAL_LABELS is specified as a feature type in the
     *         Feature input parameter.
     */
    public final Float minConfidence() {
        return minConfidence;
    }

    /**
     * <p>
     * A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection feature, while
     * specifying IMAGE_PROPERTIES returns information regarding image color and quality. If no option is specified
     * GENERAL_LABELS is used by default.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatures} method.
     * </p>
     * 
     * @return A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection feature,
     *         while specifying IMAGE_PROPERTIES returns information regarding image color and quality. If no option is
     *         specified GENERAL_LABELS is used by default.
     */
    public final List<DetectLabelsFeatureName> features() {
        return DetectLabelsFeatureListCopier.copyStringToEnum(features);
    }

    /**
     * For responses, this returns true if the service returned a value for the Features property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFeatures() {
        return features != null && !(features instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection feature, while
     * specifying IMAGE_PROPERTIES returns information regarding image color and quality. If no option is specified
     * GENERAL_LABELS is used by default.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatures} method.
     * </p>
     * 
     * @return A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection feature,
     *         while specifying IMAGE_PROPERTIES returns information regarding image color and quality. If no option is
     *         specified GENERAL_LABELS is used by default.
     */
    public final List<String> featuresAsStrings() {
        return features;
    }

    /**
     * <p>
     * A list of the filters to be applied to returned detected labels and image properties. Specified filters can be
     * inclusive, exclusive, or a combination of both. Filters can be used for individual labels or label categories.
     * The exact label names or label categories must be supplied. For a full list of labels and label categories, see
     * <a href="https://docs.aws.amazon.com/rekognition/latest/dg/labels.html">Detecting labels</a>.
     * </p>
     * 
     * @return A list of the filters to be applied to returned detected labels and image properties. Specified filters
     *         can be inclusive, exclusive, or a combination of both. Filters can be used for individual labels or label
     *         categories. The exact label names or label categories must be supplied. For a full list of labels and
     *         label categories, see <a href="https://docs.aws.amazon.com/rekognition/latest/dg/labels.html">Detecting
     *         labels</a>.
     */
    public final DetectLabelsSettings settings() {
        return settings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(maxLabels());
        hashCode = 31 * hashCode + Objects.hashCode(minConfidence());
        hashCode = 31 * hashCode + Objects.hashCode(hasFeatures() ? featuresAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(settings());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DetectLabelsRequest)) {
            return false;
        }
        DetectLabelsRequest other = (DetectLabelsRequest) obj;
        return Objects.equals(image(), other.image()) && Objects.equals(maxLabels(), other.maxLabels())
                && Objects.equals(minConfidence(), other.minConfidence()) && hasFeatures() == other.hasFeatures()
                && Objects.equals(featuresAsStrings(), other.featuresAsStrings()) && Objects.equals(settings(), other.settings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DetectLabelsRequest").add("Image", image()).add("MaxLabels", maxLabels())
                .add("MinConfidence", minConfidence()).add("Features", hasFeatures() ? featuresAsStrings() : null)
                .add("Settings", settings()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Image":
            return Optional.ofNullable(clazz.cast(image()));
        case "MaxLabels":
            return Optional.ofNullable(clazz.cast(maxLabels()));
        case "MinConfidence":
            return Optional.ofNullable(clazz.cast(minConfidence()));
        case "Features":
            return Optional.ofNullable(clazz.cast(featuresAsStrings()));
        case "Settings":
            return Optional.ofNullable(clazz.cast(settings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DetectLabelsRequest, T> g) {
        return obj -> g.apply((DetectLabelsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends RekognitionRequest.Builder, SdkPojo, CopyableBuilder<Builder, DetectLabelsRequest> {
        /**
         * <p>
         * The input image as base64-encoded bytes or an S3 object. If you use the AWS CLI to call Amazon Rekognition
         * operations, passing image bytes is not supported. Images stored in an S3 Bucket do not need to be
         * base64-encoded.
         * </p>
         * <p>
         * If you are using an AWS SDK to call Amazon Rekognition, you might not need to base64-encode image bytes
         * passed using the <code>Bytes</code> field. For more information, see Images in the Amazon Rekognition
         * developer guide.
         * </p>
         * 
         * @param image
         *        The input image as base64-encoded bytes or an S3 object. If you use the AWS CLI to call Amazon
         *        Rekognition operations, passing image bytes is not supported. Images stored in an S3 Bucket do not
         *        need to be base64-encoded.</p>
         *        <p>
         *        If you are using an AWS SDK to call Amazon Rekognition, you might not need to base64-encode image
         *        bytes passed using the <code>Bytes</code> field. For more information, see Images in the Amazon
         *        Rekognition developer guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(Image image);

        /**
         * <p>
         * The input image as base64-encoded bytes or an S3 object. If you use the AWS CLI to call Amazon Rekognition
         * operations, passing image bytes is not supported. Images stored in an S3 Bucket do not need to be
         * base64-encoded.
         * </p>
         * <p>
         * If you are using an AWS SDK to call Amazon Rekognition, you might not need to base64-encode image bytes
         * passed using the <code>Bytes</code> field. For more information, see Images in the Amazon Rekognition
         * developer guide.
         * </p>
         * This is a convenience method that creates an instance of the {@link Image.Builder} avoiding the need to
         * create one manually via {@link Image#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Image.Builder#build()} is called immediately and its result is
         * passed to {@link #image(Image)}.
         * 
         * @param image
         *        a consumer that will call methods on {@link Image.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #image(Image)
         */
        default Builder image(Consumer<Image.Builder> image) {
            return image(Image.builder().applyMutation(image).build());
        }

        /**
         * <p>
         * Maximum number of labels you want the service to return in the response. The service returns the specified
         * number of highest confidence labels. Only valid when GENERAL_LABELS is specified as a feature type in the
         * Feature input parameter.
         * </p>
         * 
         * @param maxLabels
         *        Maximum number of labels you want the service to return in the response. The service returns the
         *        specified number of highest confidence labels. Only valid when GENERAL_LABELS is specified as a
         *        feature type in the Feature input parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxLabels(Integer maxLabels);

        /**
         * <p>
         * Specifies the minimum confidence level for the labels to return. Amazon Rekognition doesn't return any labels
         * with confidence lower than this specified value.
         * </p>
         * <p>
         * If <code>MinConfidence</code> is not specified, the operation returns labels with a confidence values greater
         * than or equal to 55 percent. Only valid when GENERAL_LABELS is specified as a feature type in the Feature
         * input parameter.
         * </p>
         * 
         * @param minConfidence
         *        Specifies the minimum confidence level for the labels to return. Amazon Rekognition doesn't return any
         *        labels with confidence lower than this specified value.</p>
         *        <p>
         *        If <code>MinConfidence</code> is not specified, the operation returns labels with a confidence values
         *        greater than or equal to 55 percent. Only valid when GENERAL_LABELS is specified as a feature type in
         *        the Feature input parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minConfidence(Float minConfidence);

        /**
         * <p>
         * A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection feature, while
         * specifying IMAGE_PROPERTIES returns information regarding image color and quality. If no option is specified
         * GENERAL_LABELS is used by default.
         * </p>
         * 
         * @param features
         *        A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection
         *        feature, while specifying IMAGE_PROPERTIES returns information regarding image color and quality. If
         *        no option is specified GENERAL_LABELS is used by default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featuresWithStrings(Collection<String> features);

        /**
         * <p>
         * A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection feature, while
         * specifying IMAGE_PROPERTIES returns information regarding image color and quality. If no option is specified
         * GENERAL_LABELS is used by default.
         * </p>
         * 
         * @param features
         *        A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection
         *        feature, while specifying IMAGE_PROPERTIES returns information regarding image color and quality. If
         *        no option is specified GENERAL_LABELS is used by default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featuresWithStrings(String... features);

        /**
         * <p>
         * A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection feature, while
         * specifying IMAGE_PROPERTIES returns information regarding image color and quality. If no option is specified
         * GENERAL_LABELS is used by default.
         * </p>
         * 
         * @param features
         *        A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection
         *        feature, while specifying IMAGE_PROPERTIES returns information regarding image color and quality. If
         *        no option is specified GENERAL_LABELS is used by default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder features(Collection<DetectLabelsFeatureName> features);

        /**
         * <p>
         * A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection feature, while
         * specifying IMAGE_PROPERTIES returns information regarding image color and quality. If no option is specified
         * GENERAL_LABELS is used by default.
         * </p>
         * 
         * @param features
         *        A list of the types of analysis to perform. Specifying GENERAL_LABELS uses the label detection
         *        feature, while specifying IMAGE_PROPERTIES returns information regarding image color and quality. If
         *        no option is specified GENERAL_LABELS is used by default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder features(DetectLabelsFeatureName... features);

        /**
         * <p>
         * A list of the filters to be applied to returned detected labels and image properties. Specified filters can
         * be inclusive, exclusive, or a combination of both. Filters can be used for individual labels or label
         * categories. The exact label names or label categories must be supplied. For a full list of labels and label
         * categories, see <a href="https://docs.aws.amazon.com/rekognition/latest/dg/labels.html">Detecting labels</a>.
         * </p>
         * 
         * @param settings
         *        A list of the filters to be applied to returned detected labels and image properties. Specified
         *        filters can be inclusive, exclusive, or a combination of both. Filters can be used for individual
         *        labels or label categories. The exact label names or label categories must be supplied. For a full
         *        list of labels and label categories, see <a
         *        href="https://docs.aws.amazon.com/rekognition/latest/dg/labels.html">Detecting labels</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settings(DetectLabelsSettings settings);

        /**
         * <p>
         * A list of the filters to be applied to returned detected labels and image properties. Specified filters can
         * be inclusive, exclusive, or a combination of both. Filters can be used for individual labels or label
         * categories. The exact label names or label categories must be supplied. For a full list of labels and label
         * categories, see <a href="https://docs.aws.amazon.com/rekognition/latest/dg/labels.html">Detecting labels</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link DetectLabelsSettings.Builder} avoiding
         * the need to create one manually via {@link DetectLabelsSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DetectLabelsSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #settings(DetectLabelsSettings)}.
         * 
         * @param settings
         *        a consumer that will call methods on {@link DetectLabelsSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #settings(DetectLabelsSettings)
         */
        default Builder settings(Consumer<DetectLabelsSettings.Builder> settings) {
            return settings(DetectLabelsSettings.builder().applyMutation(settings).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends RekognitionRequest.BuilderImpl implements Builder {
        private Image image;

        private Integer maxLabels;

        private Float minConfidence;

        private List<String> features = DefaultSdkAutoConstructList.getInstance();

        private DetectLabelsSettings settings;

        private BuilderImpl() {
        }

        private BuilderImpl(DetectLabelsRequest model) {
            super(model);
            image(model.image);
            maxLabels(model.maxLabels);
            minConfidence(model.minConfidence);
            featuresWithStrings(model.features);
            settings(model.settings);
        }

        public final Image.Builder getImage() {
            return image != null ? image.toBuilder() : null;
        }

        public final void setImage(Image.BuilderImpl image) {
            this.image = image != null ? image.build() : null;
        }

        @Override
        public final Builder image(Image image) {
            this.image = image;
            return this;
        }

        public final Integer getMaxLabels() {
            return maxLabels;
        }

        public final void setMaxLabels(Integer maxLabels) {
            this.maxLabels = maxLabels;
        }

        @Override
        public final Builder maxLabels(Integer maxLabels) {
            this.maxLabels = maxLabels;
            return this;
        }

        public final Float getMinConfidence() {
            return minConfidence;
        }

        public final void setMinConfidence(Float minConfidence) {
            this.minConfidence = minConfidence;
        }

        @Override
        public final Builder minConfidence(Float minConfidence) {
            this.minConfidence = minConfidence;
            return this;
        }

        public final Collection<String> getFeatures() {
            if (features instanceof SdkAutoConstructList) {
                return null;
            }
            return features;
        }

        public final void setFeatures(Collection<String> features) {
            this.features = DetectLabelsFeatureListCopier.copy(features);
        }

        @Override
        public final Builder featuresWithStrings(Collection<String> features) {
            this.features = DetectLabelsFeatureListCopier.copy(features);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featuresWithStrings(String... features) {
            featuresWithStrings(Arrays.asList(features));
            return this;
        }

        @Override
        public final Builder features(Collection<DetectLabelsFeatureName> features) {
            this.features = DetectLabelsFeatureListCopier.copyEnumToString(features);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder features(DetectLabelsFeatureName... features) {
            features(Arrays.asList(features));
            return this;
        }

        public final DetectLabelsSettings.Builder getSettings() {
            return settings != null ? settings.toBuilder() : null;
        }

        public final void setSettings(DetectLabelsSettings.BuilderImpl settings) {
            this.settings = settings != null ? settings.build() : null;
        }

        @Override
        @Transient
        public final Builder settings(DetectLabelsSettings settings) {
            this.settings = settings;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DetectLabelsRequest build() {
            return new DetectLabelsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
