/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.RequiredTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The encryption settings for an S3 Metadata journal table or inventory table configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetadataTableEncryptionConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<MetadataTableEncryptionConfiguration.Builder, MetadataTableEncryptionConfiguration> {
    private static final SdkField<String> SSE_ALGORITHM_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SseAlgorithm")
            .getter(getter(MetadataTableEncryptionConfiguration::sseAlgorithmAsString))
            .setter(setter(Builder::sseAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SseAlgorithm")
                    .unmarshallLocationName("SseAlgorithm").build(), RequiredTrait.create()).build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyArn")
            .getter(getter(MetadataTableEncryptionConfiguration::kmsKeyArn))
            .setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyArn")
                    .unmarshallLocationName("KmsKeyArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SSE_ALGORITHM_FIELD,
            KMS_KEY_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String sseAlgorithm;

    private final String kmsKeyArn;

    private MetadataTableEncryptionConfiguration(BuilderImpl builder) {
        this.sseAlgorithm = builder.sseAlgorithm;
        this.kmsKeyArn = builder.kmsKeyArn;
    }

    /**
     * <p>
     * The encryption type specified for a metadata table. To specify server-side encryption with Key Management Service
     * (KMS) keys (SSE-KMS), use the <code>aws:kms</code> value. To specify server-side encryption with Amazon S3
     * managed keys (SSE-S3), use the <code>AES256</code> value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sseAlgorithm} will
     * return {@link TableSseAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sseAlgorithmAsString}.
     * </p>
     * 
     * @return The encryption type specified for a metadata table. To specify server-side encryption with Key Management
     *         Service (KMS) keys (SSE-KMS), use the <code>aws:kms</code> value. To specify server-side encryption with
     *         Amazon S3 managed keys (SSE-S3), use the <code>AES256</code> value.
     * @see TableSseAlgorithm
     */
    public final TableSseAlgorithm sseAlgorithm() {
        return TableSseAlgorithm.fromValue(sseAlgorithm);
    }

    /**
     * <p>
     * The encryption type specified for a metadata table. To specify server-side encryption with Key Management Service
     * (KMS) keys (SSE-KMS), use the <code>aws:kms</code> value. To specify server-side encryption with Amazon S3
     * managed keys (SSE-S3), use the <code>AES256</code> value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sseAlgorithm} will
     * return {@link TableSseAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sseAlgorithmAsString}.
     * </p>
     * 
     * @return The encryption type specified for a metadata table. To specify server-side encryption with Key Management
     *         Service (KMS) keys (SSE-KMS), use the <code>aws:kms</code> value. To specify server-side encryption with
     *         Amazon S3 managed keys (SSE-S3), use the <code>AES256</code> value.
     * @see TableSseAlgorithm
     */
    public final String sseAlgorithmAsString() {
        return sseAlgorithm;
    }

    /**
     * <p>
     * If server-side encryption with Key Management Service (KMS) keys (SSE-KMS) is specified, you must also specify
     * the KMS key Amazon Resource Name (ARN). You must specify a customer-managed KMS key that's located in the same
     * Region as the general purpose bucket that corresponds to the metadata table configuration.
     * </p>
     * 
     * @return If server-side encryption with Key Management Service (KMS) keys (SSE-KMS) is specified, you must also
     *         specify the KMS key Amazon Resource Name (ARN). You must specify a customer-managed KMS key that's
     *         located in the same Region as the general purpose bucket that corresponds to the metadata table
     *         configuration.
     */
    public final String kmsKeyArn() {
        return kmsKeyArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sseAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetadataTableEncryptionConfiguration)) {
            return false;
        }
        MetadataTableEncryptionConfiguration other = (MetadataTableEncryptionConfiguration) obj;
        return Objects.equals(sseAlgorithmAsString(), other.sseAlgorithmAsString())
                && Objects.equals(kmsKeyArn(), other.kmsKeyArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MetadataTableEncryptionConfiguration").add("SseAlgorithm", sseAlgorithmAsString())
                .add("KmsKeyArn", kmsKeyArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SseAlgorithm":
            return Optional.ofNullable(clazz.cast(sseAlgorithmAsString()));
        case "KmsKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SseAlgorithm", SSE_ALGORITHM_FIELD);
        map.put("KmsKeyArn", KMS_KEY_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MetadataTableEncryptionConfiguration, T> g) {
        return obj -> g.apply((MetadataTableEncryptionConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetadataTableEncryptionConfiguration> {
        /**
         * <p>
         * The encryption type specified for a metadata table. To specify server-side encryption with Key Management
         * Service (KMS) keys (SSE-KMS), use the <code>aws:kms</code> value. To specify server-side encryption with
         * Amazon S3 managed keys (SSE-S3), use the <code>AES256</code> value.
         * </p>
         * 
         * @param sseAlgorithm
         *        The encryption type specified for a metadata table. To specify server-side encryption with Key
         *        Management Service (KMS) keys (SSE-KMS), use the <code>aws:kms</code> value. To specify server-side
         *        encryption with Amazon S3 managed keys (SSE-S3), use the <code>AES256</code> value.
         * @see TableSseAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableSseAlgorithm
         */
        Builder sseAlgorithm(String sseAlgorithm);

        /**
         * <p>
         * The encryption type specified for a metadata table. To specify server-side encryption with Key Management
         * Service (KMS) keys (SSE-KMS), use the <code>aws:kms</code> value. To specify server-side encryption with
         * Amazon S3 managed keys (SSE-S3), use the <code>AES256</code> value.
         * </p>
         * 
         * @param sseAlgorithm
         *        The encryption type specified for a metadata table. To specify server-side encryption with Key
         *        Management Service (KMS) keys (SSE-KMS), use the <code>aws:kms</code> value. To specify server-side
         *        encryption with Amazon S3 managed keys (SSE-S3), use the <code>AES256</code> value.
         * @see TableSseAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableSseAlgorithm
         */
        Builder sseAlgorithm(TableSseAlgorithm sseAlgorithm);

        /**
         * <p>
         * If server-side encryption with Key Management Service (KMS) keys (SSE-KMS) is specified, you must also
         * specify the KMS key Amazon Resource Name (ARN). You must specify a customer-managed KMS key that's located in
         * the same Region as the general purpose bucket that corresponds to the metadata table configuration.
         * </p>
         * 
         * @param kmsKeyArn
         *        If server-side encryption with Key Management Service (KMS) keys (SSE-KMS) is specified, you must also
         *        specify the KMS key Amazon Resource Name (ARN). You must specify a customer-managed KMS key that's
         *        located in the same Region as the general purpose bucket that corresponds to the metadata table
         *        configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);
    }

    static final class BuilderImpl implements Builder {
        private String sseAlgorithm;

        private String kmsKeyArn;

        private BuilderImpl() {
        }

        private BuilderImpl(MetadataTableEncryptionConfiguration model) {
            sseAlgorithm(model.sseAlgorithm);
            kmsKeyArn(model.kmsKeyArn);
        }

        public final String getSseAlgorithm() {
            return sseAlgorithm;
        }

        public final void setSseAlgorithm(String sseAlgorithm) {
            this.sseAlgorithm = sseAlgorithm;
        }

        @Override
        public final Builder sseAlgorithm(String sseAlgorithm) {
            this.sseAlgorithm = sseAlgorithm;
            return this;
        }

        @Override
        public final Builder sseAlgorithm(TableSseAlgorithm sseAlgorithm) {
            this.sseAlgorithm(sseAlgorithm == null ? null : sseAlgorithm.toString());
            return this;
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        @Override
        public MetadataTableEncryptionConfiguration build() {
            return new MetadataTableEncryptionConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
