/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemakergeospatial;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.AsyncResponseTransformerUtils;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sagemakergeospatial.model.AccessDeniedException;
import software.amazon.awssdk.services.sagemakergeospatial.model.ConflictException;
import software.amazon.awssdk.services.sagemakergeospatial.model.DeleteEarthObservationJobRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.DeleteEarthObservationJobResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.DeleteVectorEnrichmentJobRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.DeleteVectorEnrichmentJobResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.ExportEarthObservationJobRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.ExportEarthObservationJobResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.ExportVectorEnrichmentJobRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.ExportVectorEnrichmentJobResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.GetEarthObservationJobRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.GetEarthObservationJobResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.GetRasterDataCollectionRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.GetRasterDataCollectionResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.GetTileRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.GetTileResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.GetVectorEnrichmentJobRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.GetVectorEnrichmentJobResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.InternalServerException;
import software.amazon.awssdk.services.sagemakergeospatial.model.ListEarthObservationJobsRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.ListEarthObservationJobsResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.ListRasterDataCollectionsRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.ListRasterDataCollectionsResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.ListVectorEnrichmentJobsRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.ListVectorEnrichmentJobsResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sagemakergeospatial.model.SageMakerGeospatialException;
import software.amazon.awssdk.services.sagemakergeospatial.model.SageMakerGeospatialRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.SearchRasterDataCollectionRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.SearchRasterDataCollectionResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.sagemakergeospatial.model.StartEarthObservationJobRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.StartEarthObservationJobResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.StartVectorEnrichmentJobRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.StartVectorEnrichmentJobResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.StopEarthObservationJobRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.StopEarthObservationJobResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.StopVectorEnrichmentJobRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.StopVectorEnrichmentJobResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.TagResourceRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.TagResourceResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.ThrottlingException;
import software.amazon.awssdk.services.sagemakergeospatial.model.UntagResourceRequest;
import software.amazon.awssdk.services.sagemakergeospatial.model.UntagResourceResponse;
import software.amazon.awssdk.services.sagemakergeospatial.model.ValidationException;
import software.amazon.awssdk.services.sagemakergeospatial.paginators.ListEarthObservationJobsPublisher;
import software.amazon.awssdk.services.sagemakergeospatial.paginators.ListRasterDataCollectionsPublisher;
import software.amazon.awssdk.services.sagemakergeospatial.paginators.ListVectorEnrichmentJobsPublisher;
import software.amazon.awssdk.services.sagemakergeospatial.paginators.SearchRasterDataCollectionPublisher;
import software.amazon.awssdk.services.sagemakergeospatial.transform.DeleteEarthObservationJobRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.DeleteVectorEnrichmentJobRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.ExportEarthObservationJobRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.ExportVectorEnrichmentJobRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.GetEarthObservationJobRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.GetRasterDataCollectionRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.GetTileRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.GetVectorEnrichmentJobRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.ListEarthObservationJobsRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.ListRasterDataCollectionsRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.ListVectorEnrichmentJobsRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.SearchRasterDataCollectionRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.StartEarthObservationJobRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.StartVectorEnrichmentJobRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.StopEarthObservationJobRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.StopVectorEnrichmentJobRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.sagemakergeospatial.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Pair;

/**
 * Internal implementation of {@link SageMakerGeospatialAsyncClient}.
 *
 * @see SageMakerGeospatialAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSageMakerGeospatialAsyncClient implements SageMakerGeospatialAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSageMakerGeospatialAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final SageMakerGeospatialServiceClientConfiguration serviceClientConfiguration;

    protected DefaultSageMakerGeospatialAsyncClient(SageMakerGeospatialServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Use this operation to delete an Earth Observation job.
     * </p>
     *
     * @param deleteEarthObservationJobRequest
     * @return A Java Future containing the result of the DeleteEarthObservationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.DeleteEarthObservationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/DeleteEarthObservationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEarthObservationJobResponse> deleteEarthObservationJob(
            DeleteEarthObservationJobRequest deleteEarthObservationJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEarthObservationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEarthObservationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEarthObservationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEarthObservationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEarthObservationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEarthObservationJobRequest, DeleteEarthObservationJobResponse>()
                            .withOperationName("DeleteEarthObservationJob")
                            .withMarshaller(new DeleteEarthObservationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteEarthObservationJobRequest));
            CompletableFuture<DeleteEarthObservationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to delete a Vector Enrichment job.
     * </p>
     *
     * @param deleteVectorEnrichmentJobRequest
     * @return A Java Future containing the result of the DeleteVectorEnrichmentJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.DeleteVectorEnrichmentJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/DeleteVectorEnrichmentJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVectorEnrichmentJobResponse> deleteVectorEnrichmentJob(
            DeleteVectorEnrichmentJobRequest deleteVectorEnrichmentJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVectorEnrichmentJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVectorEnrichmentJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVectorEnrichmentJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVectorEnrichmentJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVectorEnrichmentJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVectorEnrichmentJobRequest, DeleteVectorEnrichmentJobResponse>()
                            .withOperationName("DeleteVectorEnrichmentJob")
                            .withMarshaller(new DeleteVectorEnrichmentJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteVectorEnrichmentJobRequest));
            CompletableFuture<DeleteVectorEnrichmentJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to export results of an Earth Observation job and optionally source images used as input to
     * the EOJ to an Amazon S3 location.
     * </p>
     *
     * @param exportEarthObservationJobRequest
     * @return A Java Future containing the result of the ExportEarthObservationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You have exceeded the service quota.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.ExportEarthObservationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/ExportEarthObservationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportEarthObservationJobResponse> exportEarthObservationJob(
            ExportEarthObservationJobRequest exportEarthObservationJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportEarthObservationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportEarthObservationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportEarthObservationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportEarthObservationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportEarthObservationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportEarthObservationJobRequest, ExportEarthObservationJobResponse>()
                            .withOperationName("ExportEarthObservationJob")
                            .withMarshaller(new ExportEarthObservationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportEarthObservationJobRequest));
            CompletableFuture<ExportEarthObservationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to copy results of a Vector Enrichment job to an Amazon S3 location.
     * </p>
     *
     * @param exportVectorEnrichmentJobRequest
     * @return A Java Future containing the result of the ExportVectorEnrichmentJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You have exceeded the service quota.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.ExportVectorEnrichmentJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/ExportVectorEnrichmentJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportVectorEnrichmentJobResponse> exportVectorEnrichmentJob(
            ExportVectorEnrichmentJobRequest exportVectorEnrichmentJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportVectorEnrichmentJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportVectorEnrichmentJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportVectorEnrichmentJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportVectorEnrichmentJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportVectorEnrichmentJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportVectorEnrichmentJobRequest, ExportVectorEnrichmentJobResponse>()
                            .withOperationName("ExportVectorEnrichmentJob")
                            .withMarshaller(new ExportVectorEnrichmentJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportVectorEnrichmentJobRequest));
            CompletableFuture<ExportVectorEnrichmentJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the details for a previously initiated Earth Observation job.
     * </p>
     *
     * @param getEarthObservationJobRequest
     * @return A Java Future containing the result of the GetEarthObservationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.GetEarthObservationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/GetEarthObservationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEarthObservationJobResponse> getEarthObservationJob(
            GetEarthObservationJobRequest getEarthObservationJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEarthObservationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEarthObservationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEarthObservationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEarthObservationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEarthObservationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEarthObservationJobRequest, GetEarthObservationJobResponse>()
                            .withOperationName("GetEarthObservationJob")
                            .withMarshaller(new GetEarthObservationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEarthObservationJobRequest));
            CompletableFuture<GetEarthObservationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to get details of a specific raster data collection.
     * </p>
     *
     * @param getRasterDataCollectionRequest
     * @return A Java Future containing the result of the GetRasterDataCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.GetRasterDataCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/GetRasterDataCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRasterDataCollectionResponse> getRasterDataCollection(
            GetRasterDataCollectionRequest getRasterDataCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRasterDataCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRasterDataCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRasterDataCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRasterDataCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRasterDataCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRasterDataCollectionRequest, GetRasterDataCollectionResponse>()
                            .withOperationName("GetRasterDataCollection")
                            .withMarshaller(new GetRasterDataCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRasterDataCollectionRequest));
            CompletableFuture<GetRasterDataCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a web mercator tile for the given Earth Observation job.
     * </p>
     *
     * @param getTileRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The output binary file.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.GetTile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/GetTile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> getTile(GetTileRequest getTileRequest,
            AsyncResponseTransformer<GetTileResponse, ReturnT> asyncResponseTransformer) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTile");
            Pair<AsyncResponseTransformer<GetTileResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetTileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<GetTileRequest, GetTileResponse>().withOperationName("GetTile")
                            .withMarshaller(new GetTileRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTileRequest), asyncResponseTransformer);
            AsyncResponseTransformer<GetTileResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<GetTileResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details of a Vector Enrichment Job for a given job Amazon Resource Name (ARN).
     * </p>
     *
     * @param getVectorEnrichmentJobRequest
     * @return A Java Future containing the result of the GetVectorEnrichmentJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.GetVectorEnrichmentJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/GetVectorEnrichmentJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetVectorEnrichmentJobResponse> getVectorEnrichmentJob(
            GetVectorEnrichmentJobRequest getVectorEnrichmentJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVectorEnrichmentJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVectorEnrichmentJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVectorEnrichmentJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetVectorEnrichmentJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetVectorEnrichmentJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVectorEnrichmentJobRequest, GetVectorEnrichmentJobResponse>()
                            .withOperationName("GetVectorEnrichmentJob")
                            .withMarshaller(new GetVectorEnrichmentJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getVectorEnrichmentJobRequest));
            CompletableFuture<GetVectorEnrichmentJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to get a list of the Earth Observation jobs associated with the calling Amazon Web Services
     * account.
     * </p>
     *
     * @param listEarthObservationJobsRequest
     * @return A Java Future containing the result of the ListEarthObservationJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.ListEarthObservationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/ListEarthObservationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEarthObservationJobsResponse> listEarthObservationJobs(
            ListEarthObservationJobsRequest listEarthObservationJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEarthObservationJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEarthObservationJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEarthObservationJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEarthObservationJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEarthObservationJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEarthObservationJobsRequest, ListEarthObservationJobsResponse>()
                            .withOperationName("ListEarthObservationJobs")
                            .withMarshaller(new ListEarthObservationJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEarthObservationJobsRequest));
            CompletableFuture<ListEarthObservationJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to get a list of the Earth Observation jobs associated with the calling Amazon Web Services
     * account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEarthObservationJobs(software.amazon.awssdk.services.sagemakergeospatial.model.ListEarthObservationJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sagemakergeospatial.paginators.ListEarthObservationJobsPublisher publisher = client.listEarthObservationJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sagemakergeospatial.paginators.ListEarthObservationJobsPublisher publisher = client.listEarthObservationJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sagemakergeospatial.model.ListEarthObservationJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sagemakergeospatial.model.ListEarthObservationJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEarthObservationJobs(software.amazon.awssdk.services.sagemakergeospatial.model.ListEarthObservationJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEarthObservationJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.ListEarthObservationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/ListEarthObservationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListEarthObservationJobsPublisher listEarthObservationJobsPaginator(
            ListEarthObservationJobsRequest listEarthObservationJobsRequest) {
        return new ListEarthObservationJobsPublisher(this, applyPaginatorUserAgent(listEarthObservationJobsRequest));
    }

    /**
     * <p>
     * Use this operation to get raster data collections.
     * </p>
     *
     * @param listRasterDataCollectionsRequest
     * @return A Java Future containing the result of the ListRasterDataCollections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.ListRasterDataCollections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/ListRasterDataCollections"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRasterDataCollectionsResponse> listRasterDataCollections(
            ListRasterDataCollectionsRequest listRasterDataCollectionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRasterDataCollectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRasterDataCollections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRasterDataCollectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRasterDataCollectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRasterDataCollectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRasterDataCollectionsRequest, ListRasterDataCollectionsResponse>()
                            .withOperationName("ListRasterDataCollections")
                            .withMarshaller(new ListRasterDataCollectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRasterDataCollectionsRequest));
            CompletableFuture<ListRasterDataCollectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to get raster data collections.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRasterDataCollections(software.amazon.awssdk.services.sagemakergeospatial.model.ListRasterDataCollectionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sagemakergeospatial.paginators.ListRasterDataCollectionsPublisher publisher = client.listRasterDataCollectionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sagemakergeospatial.paginators.ListRasterDataCollectionsPublisher publisher = client.listRasterDataCollectionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sagemakergeospatial.model.ListRasterDataCollectionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sagemakergeospatial.model.ListRasterDataCollectionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRasterDataCollections(software.amazon.awssdk.services.sagemakergeospatial.model.ListRasterDataCollectionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRasterDataCollectionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.ListRasterDataCollections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/ListRasterDataCollections"
     *      target="_top">AWS API Documentation</a>
     */
    public ListRasterDataCollectionsPublisher listRasterDataCollectionsPaginator(
            ListRasterDataCollectionsRequest listRasterDataCollectionsRequest) {
        return new ListRasterDataCollectionsPublisher(this, applyPaginatorUserAgent(listRasterDataCollectionsRequest));
    }

    /**
     * <p>
     * Lists the tags attached to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of vector enrichment jobs.
     * </p>
     *
     * @param listVectorEnrichmentJobsRequest
     * @return A Java Future containing the result of the ListVectorEnrichmentJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.ListVectorEnrichmentJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/ListVectorEnrichmentJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVectorEnrichmentJobsResponse> listVectorEnrichmentJobs(
            ListVectorEnrichmentJobsRequest listVectorEnrichmentJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVectorEnrichmentJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVectorEnrichmentJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVectorEnrichmentJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVectorEnrichmentJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVectorEnrichmentJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVectorEnrichmentJobsRequest, ListVectorEnrichmentJobsResponse>()
                            .withOperationName("ListVectorEnrichmentJobs")
                            .withMarshaller(new ListVectorEnrichmentJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listVectorEnrichmentJobsRequest));
            CompletableFuture<ListVectorEnrichmentJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of vector enrichment jobs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVectorEnrichmentJobs(software.amazon.awssdk.services.sagemakergeospatial.model.ListVectorEnrichmentJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sagemakergeospatial.paginators.ListVectorEnrichmentJobsPublisher publisher = client.listVectorEnrichmentJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sagemakergeospatial.paginators.ListVectorEnrichmentJobsPublisher publisher = client.listVectorEnrichmentJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sagemakergeospatial.model.ListVectorEnrichmentJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sagemakergeospatial.model.ListVectorEnrichmentJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVectorEnrichmentJobs(software.amazon.awssdk.services.sagemakergeospatial.model.ListVectorEnrichmentJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVectorEnrichmentJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.ListVectorEnrichmentJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/ListVectorEnrichmentJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListVectorEnrichmentJobsPublisher listVectorEnrichmentJobsPaginator(
            ListVectorEnrichmentJobsRequest listVectorEnrichmentJobsRequest) {
        return new ListVectorEnrichmentJobsPublisher(this, applyPaginatorUserAgent(listVectorEnrichmentJobsRequest));
    }

    /**
     * <p>
     * Allows you run image query on a specific raster data collection to get a list of the satellite imagery matching
     * the selected filters.
     * </p>
     *
     * @param searchRasterDataCollectionRequest
     * @return A Java Future containing the result of the SearchRasterDataCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.SearchRasterDataCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/SearchRasterDataCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchRasterDataCollectionResponse> searchRasterDataCollection(
            SearchRasterDataCollectionRequest searchRasterDataCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchRasterDataCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchRasterDataCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchRasterDataCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchRasterDataCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchRasterDataCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchRasterDataCollectionRequest, SearchRasterDataCollectionResponse>()
                            .withOperationName("SearchRasterDataCollection")
                            .withMarshaller(new SearchRasterDataCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchRasterDataCollectionRequest));
            CompletableFuture<SearchRasterDataCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you run image query on a specific raster data collection to get a list of the satellite imagery matching
     * the selected filters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #searchRasterDataCollection(software.amazon.awssdk.services.sagemakergeospatial.model.SearchRasterDataCollectionRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sagemakergeospatial.paginators.SearchRasterDataCollectionPublisher publisher = client.searchRasterDataCollectionPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sagemakergeospatial.paginators.SearchRasterDataCollectionPublisher publisher = client.searchRasterDataCollectionPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.sagemakergeospatial.model.SearchRasterDataCollectionResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.sagemakergeospatial.model.SearchRasterDataCollectionResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchRasterDataCollection(software.amazon.awssdk.services.sagemakergeospatial.model.SearchRasterDataCollectionRequest)}
     * operation.</b>
     * </p>
     *
     * @param searchRasterDataCollectionRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.SearchRasterDataCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/SearchRasterDataCollection"
     *      target="_top">AWS API Documentation</a>
     */
    public SearchRasterDataCollectionPublisher searchRasterDataCollectionPaginator(
            SearchRasterDataCollectionRequest searchRasterDataCollectionRequest) {
        return new SearchRasterDataCollectionPublisher(this, applyPaginatorUserAgent(searchRasterDataCollectionRequest));
    }

    /**
     * <p>
     * Use this operation to create an Earth observation job.
     * </p>
     *
     * @param startEarthObservationJobRequest
     * @return A Java Future containing the result of the StartEarthObservationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You have exceeded the service quota.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.StartEarthObservationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/StartEarthObservationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartEarthObservationJobResponse> startEarthObservationJob(
            StartEarthObservationJobRequest startEarthObservationJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startEarthObservationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartEarthObservationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartEarthObservationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartEarthObservationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartEarthObservationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartEarthObservationJobRequest, StartEarthObservationJobResponse>()
                            .withOperationName("StartEarthObservationJob")
                            .withMarshaller(new StartEarthObservationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startEarthObservationJobRequest));
            CompletableFuture<StartEarthObservationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a Vector Enrichment job for the supplied job type. Currently, there are two supported job types: reverse
     * geocoding and map matching.
     * </p>
     *
     * @param startVectorEnrichmentJobRequest
     * @return A Java Future containing the result of the StartVectorEnrichmentJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You have exceeded the service quota.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.StartVectorEnrichmentJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/StartVectorEnrichmentJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartVectorEnrichmentJobResponse> startVectorEnrichmentJob(
            StartVectorEnrichmentJobRequest startVectorEnrichmentJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startVectorEnrichmentJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartVectorEnrichmentJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartVectorEnrichmentJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartVectorEnrichmentJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartVectorEnrichmentJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartVectorEnrichmentJobRequest, StartVectorEnrichmentJobResponse>()
                            .withOperationName("StartVectorEnrichmentJob")
                            .withMarshaller(new StartVectorEnrichmentJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startVectorEnrichmentJobRequest));
            CompletableFuture<StartVectorEnrichmentJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to stop an existing earth observation job.
     * </p>
     *
     * @param stopEarthObservationJobRequest
     * @return A Java Future containing the result of the StopEarthObservationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.StopEarthObservationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/StopEarthObservationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopEarthObservationJobResponse> stopEarthObservationJob(
            StopEarthObservationJobRequest stopEarthObservationJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopEarthObservationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopEarthObservationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopEarthObservationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopEarthObservationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopEarthObservationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopEarthObservationJobRequest, StopEarthObservationJobResponse>()
                            .withOperationName("StopEarthObservationJob")
                            .withMarshaller(new StopEarthObservationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopEarthObservationJobRequest));
            CompletableFuture<StopEarthObservationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops the Vector Enrichment job for a given job ARN.
     * </p>
     *
     * @param stopVectorEnrichmentJobRequest
     * @return A Java Future containing the result of the StopVectorEnrichmentJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.StopVectorEnrichmentJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/StopVectorEnrichmentJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopVectorEnrichmentJobResponse> stopVectorEnrichmentJob(
            StopVectorEnrichmentJobRequest stopVectorEnrichmentJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopVectorEnrichmentJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopVectorEnrichmentJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopVectorEnrichmentJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopVectorEnrichmentJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopVectorEnrichmentJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopVectorEnrichmentJobRequest, StopVectorEnrichmentJobResponse>()
                            .withOperationName("StopVectorEnrichmentJob")
                            .withMarshaller(new StopVectorEnrichmentJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopVectorEnrichmentJobRequest));
            CompletableFuture<StopVectorEnrichmentJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The resource you want to tag.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The resource you want to untag.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ResourceNotFoundException The request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerGeospatialException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerGeospatialAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-geospatial-2020-05-27/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker Geospatial");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SageMakerGeospatialServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SageMakerGeospatialException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends SageMakerGeospatialRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
