/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sqs.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SendMessageRequest extends SqsRequest implements
        ToCopyableBuilder<SendMessageRequest.Builder, SendMessageRequest> {
    private static final SdkField<String> QUEUE_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("QueueUrl").getter(getter(SendMessageRequest::queueUrl)).setter(setter(Builder::queueUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueueUrl").build()).build();

    private static final SdkField<String> MESSAGE_BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageBody").getter(getter(SendMessageRequest::messageBody)).setter(setter(Builder::messageBody))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageBody").build()).build();

    private static final SdkField<Integer> DELAY_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("DelaySeconds").getter(getter(SendMessageRequest::delaySeconds)).setter(setter(Builder::delaySeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DelaySeconds").build()).build();

    private static final SdkField<Map<String, MessageAttributeValue>> MESSAGE_ATTRIBUTES_FIELD = SdkField
            .<Map<String, MessageAttributeValue>> builder(MarshallingType.MAP)
            .memberName("MessageAttributes")
            .getter(getter(SendMessageRequest::messageAttributes))
            .setter(setter(Builder::messageAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<MessageAttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageAttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).isFlattened(true).build()).build();

    private static final SdkField<Map<String, MessageSystemAttributeValue>> MESSAGE_SYSTEM_ATTRIBUTES_FIELD = SdkField
            .<Map<String, MessageSystemAttributeValue>> builder(MarshallingType.MAP)
            .memberName("MessageSystemAttributes")
            .getter(getter(SendMessageRequest::messageSystemAttributesAsStrings))
            .setter(setter(Builder::messageSystemAttributesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageSystemAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<MessageSystemAttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageSystemAttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).isFlattened(true).build()).build();

    private static final SdkField<String> MESSAGE_DEDUPLICATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageDeduplicationId").getter(getter(SendMessageRequest::messageDeduplicationId))
            .setter(setter(Builder::messageDeduplicationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageDeduplicationId").build())
            .build();

    private static final SdkField<String> MESSAGE_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageGroupId").getter(getter(SendMessageRequest::messageGroupId))
            .setter(setter(Builder::messageGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageGroupId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUEUE_URL_FIELD,
            MESSAGE_BODY_FIELD, DELAY_SECONDS_FIELD, MESSAGE_ATTRIBUTES_FIELD, MESSAGE_SYSTEM_ATTRIBUTES_FIELD,
            MESSAGE_DEDUPLICATION_ID_FIELD, MESSAGE_GROUP_ID_FIELD));

    private final String queueUrl;

    private final String messageBody;

    private final Integer delaySeconds;

    private final Map<String, MessageAttributeValue> messageAttributes;

    private final Map<String, MessageSystemAttributeValue> messageSystemAttributes;

    private final String messageDeduplicationId;

    private final String messageGroupId;

    private SendMessageRequest(BuilderImpl builder) {
        super(builder);
        this.queueUrl = builder.queueUrl;
        this.messageBody = builder.messageBody;
        this.delaySeconds = builder.delaySeconds;
        this.messageAttributes = builder.messageAttributes;
        this.messageSystemAttributes = builder.messageSystemAttributes;
        this.messageDeduplicationId = builder.messageDeduplicationId;
        this.messageGroupId = builder.messageGroupId;
    }

    /**
     * <p>
     * The URL of the Amazon SQS queue to which a message is sent.
     * </p>
     * <p>
     * Queue URLs and names are case-sensitive.
     * </p>
     * 
     * @return The URL of the Amazon SQS queue to which a message is sent.</p>
     *         <p>
     *         Queue URLs and names are case-sensitive.
     */
    public final String queueUrl() {
        return queueUrl;
    }

    /**
     * <p>
     * The message to send. The minimum size is one character. The maximum size is 256 KiB.
     * </p>
     * <important>
     * <p>
     * A message can include only XML, JSON, and unformatted text. The following Unicode characters are allowed:
     * </p>
     * <p>
     * <code>#x9</code> | <code>#xA</code> | <code>#xD</code> | <code>#x20</code> to <code>#xD7FF</code> |
     * <code>#xE000</code> to <code>#xFFFD</code> | <code>#x10000</code> to <code>#x10FFFF</code>
     * </p>
     * <p>
     * Any characters not included in this list will be rejected. For more information, see the <a
     * href="http://www.w3.org/TR/REC-xml/#charsets">W3C specification for characters</a>.
     * </p>
     * </important>
     * 
     * @return The message to send. The minimum size is one character. The maximum size is 256 KiB.</p> <important>
     *         <p>
     *         A message can include only XML, JSON, and unformatted text. The following Unicode characters are allowed:
     *         </p>
     *         <p>
     *         <code>#x9</code> | <code>#xA</code> | <code>#xD</code> | <code>#x20</code> to <code>#xD7FF</code> |
     *         <code>#xE000</code> to <code>#xFFFD</code> | <code>#x10000</code> to <code>#x10FFFF</code>
     *         </p>
     *         <p>
     *         Any characters not included in this list will be rejected. For more information, see the <a
     *         href="http://www.w3.org/TR/REC-xml/#charsets">W3C specification for characters</a>.
     *         </p>
     */
    public final String messageBody() {
        return messageBody;
    }

    /**
     * <p>
     * The length of time, in seconds, for which to delay a specific message. Valid values: 0 to 900. Maximum: 15
     * minutes. Messages with a positive <code>DelaySeconds</code> value become available for processing after the delay
     * period is finished. If you don't specify a value, the default value for the queue applies.
     * </p>
     * <note>
     * <p>
     * When you set <code>FifoQueue</code>, you can't set <code>DelaySeconds</code> per message. You can set this
     * parameter only on a queue level.
     * </p>
     * </note>
     * 
     * @return The length of time, in seconds, for which to delay a specific message. Valid values: 0 to 900. Maximum:
     *         15 minutes. Messages with a positive <code>DelaySeconds</code> value become available for processing
     *         after the delay period is finished. If you don't specify a value, the default value for the queue
     *         applies. </p> <note>
     *         <p>
     *         When you set <code>FifoQueue</code>, you can't set <code>DelaySeconds</code> per message. You can set
     *         this parameter only on a queue level.
     *         </p>
     */
    public final Integer delaySeconds() {
        return delaySeconds;
    }

    /**
     * For responses, this returns true if the service returned a value for the MessageAttributes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMessageAttributes() {
        return messageAttributes != null && !(messageAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
     * >Amazon SQS message attributes</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMessageAttributes} method.
     * </p>
     * 
     * @return Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For
     *         more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
     *         >Amazon SQS message attributes</a> in the <i>Amazon SQS Developer Guide</i>.
     */
    public final Map<String, MessageAttributeValue> messageAttributes() {
        return messageAttributes;
    }

    /**
     * <p>
     * The message system attribute to send. Each message system attribute consists of a <code>Name</code>,
     * <code>Type</code>, and <code>Value</code>.
     * </p>
     * <important>
     * <ul>
     * <li>
     * <p>
     * Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
     * <code>String</code> and its value must be a correctly formatted X-Ray trace header string.
     * </p>
     * </li>
     * <li>
     * <p>
     * The size of a message system attribute doesn't count towards the total size of a message.
     * </p>
     * </li>
     * </ul>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMessageSystemAttributes} method.
     * </p>
     * 
     * @return The message system attribute to send. Each message system attribute consists of a <code>Name</code>,
     *         <code>Type</code>, and <code>Value</code>.</p> <important>
     *         <ul>
     *         <li>
     *         <p>
     *         Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
     *         <code>String</code> and its value must be a correctly formatted X-Ray trace header string.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The size of a message system attribute doesn't count towards the total size of a message.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<MessageSystemAttributeNameForSends, MessageSystemAttributeValue> messageSystemAttributes() {
        return MessageBodySystemAttributeMapCopier.copyStringToEnum(messageSystemAttributes);
    }

    /**
     * For responses, this returns true if the service returned a value for the MessageSystemAttributes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasMessageSystemAttributes() {
        return messageSystemAttributes != null && !(messageSystemAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The message system attribute to send. Each message system attribute consists of a <code>Name</code>,
     * <code>Type</code>, and <code>Value</code>.
     * </p>
     * <important>
     * <ul>
     * <li>
     * <p>
     * Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
     * <code>String</code> and its value must be a correctly formatted X-Ray trace header string.
     * </p>
     * </li>
     * <li>
     * <p>
     * The size of a message system attribute doesn't count towards the total size of a message.
     * </p>
     * </li>
     * </ul>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMessageSystemAttributes} method.
     * </p>
     * 
     * @return The message system attribute to send. Each message system attribute consists of a <code>Name</code>,
     *         <code>Type</code>, and <code>Value</code>.</p> <important>
     *         <ul>
     *         <li>
     *         <p>
     *         Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
     *         <code>String</code> and its value must be a correctly formatted X-Ray trace header string.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The size of a message system attribute doesn't count towards the total size of a message.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<String, MessageSystemAttributeValue> messageSystemAttributesAsStrings() {
        return messageSystemAttributes;
    }

    /**
     * <p>
     * This parameter applies only to FIFO (first-in-first-out) queues.
     * </p>
     * <p>
     * The token used for deduplication of sent messages. If a message with a particular
     * <code>MessageDeduplicationId</code> is sent successfully, any messages sent with the same
     * <code>MessageDeduplicationId</code> are accepted successfully but aren't delivered during the 5-minute
     * deduplication interval. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
     * > Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Every message must have a unique <code>MessageDeduplicationId</code>,
     * </p>
     * <ul>
     * <li>
     * <p>
     * You may provide a <code>MessageDeduplicationId</code> explicitly.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
     * <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
     * <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the message).
     * </p>
     * </li>
     * <li>
     * <p>
     * If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
     * <code>ContentBasedDeduplication</code> set, the action fails with an error.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code> overrides
     * the generated one.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
     * deduplication interval are treated as duplicates and only one copy of the message is delivered.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with a
     * <code>MessageDeduplicationId</code> that is the same as the one generated for the first
     * <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the message
     * is delivered.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be useful for
     * troubleshooting delivery issues).
     * </p>
     * <p>
     * If a message is sent successfully but the acknowledgement is lost and the message is resent with the same
     * <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SQS can't detect duplicate messages.
     * </p>
     * <p>
     * Amazon SQS continues to keep track of the message deduplication ID even after the message is received and
     * deleted.
     * </p>
     * </note>
     * <p>
     * The maximum length of <code>MessageDeduplicationId</code> is 128 characters. <code>MessageDeduplicationId</code>
     * can contain alphanumeric characters (<code>a-z</code>, <code>A-Z</code>, <code>0-9</code>) and punctuation (
     * <code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
     * </p>
     * <p>
     * For best practices of using <code>MessageDeduplicationId</code>, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagededuplicationid-property.html"
     * >Using the MessageDeduplicationId Property</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * 
     * @return This parameter applies only to FIFO (first-in-first-out) queues.</p>
     *         <p>
     *         The token used for deduplication of sent messages. If a message with a particular
     *         <code>MessageDeduplicationId</code> is sent successfully, any messages sent with the same
     *         <code>MessageDeduplicationId</code> are accepted successfully but aren't delivered during the 5-minute
     *         deduplication interval. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
     *         > Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Every message must have a unique <code>MessageDeduplicationId</code>,
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You may provide a <code>MessageDeduplicationId</code> explicitly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
     *         <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
     *         <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
     *         message).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
     *         <code>ContentBasedDeduplication</code> set, the action fails with an error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
     *         overrides the generated one.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
     *         deduplication interval are treated as duplicates and only one copy of the message is delivered.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with
     *         a <code>MessageDeduplicationId</code> that is the same as the one generated for the first
     *         <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the
     *         message is delivered.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be useful
     *         for troubleshooting delivery issues).
     *         </p>
     *         <p>
     *         If a message is sent successfully but the acknowledgement is lost and the message is resent with the same
     *         <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SQS can't detect duplicate
     *         messages.
     *         </p>
     *         <p>
     *         Amazon SQS continues to keep track of the message deduplication ID even after the message is received and
     *         deleted.
     *         </p>
     *         </note>
     *         <p>
     *         The maximum length of <code>MessageDeduplicationId</code> is 128 characters.
     *         <code>MessageDeduplicationId</code> can contain alphanumeric characters (<code>a-z</code>,
     *         <code>A-Z</code>, <code>0-9</code>) and punctuation (
     *         <code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
     *         </p>
     *         <p>
     *         For best practices of using <code>MessageDeduplicationId</code>, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagededuplicationid-property.html"
     *         >Using the MessageDeduplicationId Property</a> in the <i>Amazon SQS Developer Guide</i>.
     */
    public final String messageDeduplicationId() {
        return messageDeduplicationId;
    }

    /**
     * <p>
     * This parameter applies only to FIFO (first-in-first-out) queues.
     * </p>
     * <p>
     * The tag that specifies that a message belongs to a specific message group. Messages that belong to the same
     * message group are processed in a FIFO manner (however, messages in different message groups might be processed
     * out of order). To interleave multiple ordered streams within a single queue, use <code>MessageGroupId</code>
     * values (for example, session data for multiple users). In this scenario, multiple consumers can process the
     * queue, but the session data of each user is processed in a FIFO fashion.
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
     * <code>MessageGroupId</code>, the action fails.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReceiveMessage</code> might return messages with multiple <code>MessageGroupId</code> values. For each
     * <code>MessageGroupId</code>, the messages are sorted by time sent. The caller can't specify a
     * <code>MessageGroupId</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The length of <code>MessageGroupId</code> is 128 characters. Valid values: alphanumeric characters and
     * punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
     * </p>
     * <p>
     * For best practices of using <code>MessageGroupId</code>, see <a href=
     * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagegroupid-property.html"
     * >Using the MessageGroupId Property</a> in the <i>Amazon SQS Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * <code>MessageGroupId</code> is required for FIFO queues. You can't use it for Standard queues.
     * </p>
     * </important>
     * 
     * @return This parameter applies only to FIFO (first-in-first-out) queues.</p>
     *         <p>
     *         The tag that specifies that a message belongs to a specific message group. Messages that belong to the
     *         same message group are processed in a FIFO manner (however, messages in different message groups might be
     *         processed out of order). To interleave multiple ordered streams within a single queue, use
     *         <code>MessageGroupId</code> values (for example, session data for multiple users). In this scenario,
     *         multiple consumers can process the queue, but the session data of each user is processed in a FIFO
     *         fashion.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
     *         <code>MessageGroupId</code>, the action fails.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReceiveMessage</code> might return messages with multiple <code>MessageGroupId</code> values. For
     *         each <code>MessageGroupId</code>, the messages are sorted by time sent. The caller can't specify a
     *         <code>MessageGroupId</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The length of <code>MessageGroupId</code> is 128 characters. Valid values: alphanumeric characters and
     *         punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
     *         </p>
     *         <p>
     *         For best practices of using <code>MessageGroupId</code>, see <a href=
     *         "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagegroupid-property.html"
     *         >Using the MessageGroupId Property</a> in the <i>Amazon SQS Developer Guide</i>.
     *         </p>
     *         <important>
     *         <p>
     *         <code>MessageGroupId</code> is required for FIFO queues. You can't use it for Standard queues.
     *         </p>
     */
    public final String messageGroupId() {
        return messageGroupId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(queueUrl());
        hashCode = 31 * hashCode + Objects.hashCode(messageBody());
        hashCode = 31 * hashCode + Objects.hashCode(delaySeconds());
        hashCode = 31 * hashCode + Objects.hashCode(hasMessageAttributes() ? messageAttributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMessageSystemAttributes() ? messageSystemAttributesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(messageDeduplicationId());
        hashCode = 31 * hashCode + Objects.hashCode(messageGroupId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendMessageRequest)) {
            return false;
        }
        SendMessageRequest other = (SendMessageRequest) obj;
        return Objects.equals(queueUrl(), other.queueUrl()) && Objects.equals(messageBody(), other.messageBody())
                && Objects.equals(delaySeconds(), other.delaySeconds()) && hasMessageAttributes() == other.hasMessageAttributes()
                && Objects.equals(messageAttributes(), other.messageAttributes())
                && hasMessageSystemAttributes() == other.hasMessageSystemAttributes()
                && Objects.equals(messageSystemAttributesAsStrings(), other.messageSystemAttributesAsStrings())
                && Objects.equals(messageDeduplicationId(), other.messageDeduplicationId())
                && Objects.equals(messageGroupId(), other.messageGroupId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SendMessageRequest").add("QueueUrl", queueUrl()).add("MessageBody", messageBody())
                .add("DelaySeconds", delaySeconds())
                .add("MessageAttributes", hasMessageAttributes() ? messageAttributes() : null)
                .add("MessageSystemAttributes", hasMessageSystemAttributes() ? messageSystemAttributesAsStrings() : null)
                .add("MessageDeduplicationId", messageDeduplicationId()).add("MessageGroupId", messageGroupId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "QueueUrl":
            return Optional.ofNullable(clazz.cast(queueUrl()));
        case "MessageBody":
            return Optional.ofNullable(clazz.cast(messageBody()));
        case "DelaySeconds":
            return Optional.ofNullable(clazz.cast(delaySeconds()));
        case "MessageAttributes":
            return Optional.ofNullable(clazz.cast(messageAttributes()));
        case "MessageSystemAttributes":
            return Optional.ofNullable(clazz.cast(messageSystemAttributesAsStrings()));
        case "MessageDeduplicationId":
            return Optional.ofNullable(clazz.cast(messageDeduplicationId()));
        case "MessageGroupId":
            return Optional.ofNullable(clazz.cast(messageGroupId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SendMessageRequest, T> g) {
        return obj -> g.apply((SendMessageRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SqsRequest.Builder, SdkPojo, CopyableBuilder<Builder, SendMessageRequest> {
        /**
         * <p>
         * The URL of the Amazon SQS queue to which a message is sent.
         * </p>
         * <p>
         * Queue URLs and names are case-sensitive.
         * </p>
         * 
         * @param queueUrl
         *        The URL of the Amazon SQS queue to which a message is sent.</p>
         *        <p>
         *        Queue URLs and names are case-sensitive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queueUrl(String queueUrl);

        /**
         * <p>
         * The message to send. The minimum size is one character. The maximum size is 256 KiB.
         * </p>
         * <important>
         * <p>
         * A message can include only XML, JSON, and unformatted text. The following Unicode characters are allowed:
         * </p>
         * <p>
         * <code>#x9</code> | <code>#xA</code> | <code>#xD</code> | <code>#x20</code> to <code>#xD7FF</code> |
         * <code>#xE000</code> to <code>#xFFFD</code> | <code>#x10000</code> to <code>#x10FFFF</code>
         * </p>
         * <p>
         * Any characters not included in this list will be rejected. For more information, see the <a
         * href="http://www.w3.org/TR/REC-xml/#charsets">W3C specification for characters</a>.
         * </p>
         * </important>
         * 
         * @param messageBody
         *        The message to send. The minimum size is one character. The maximum size is 256 KiB.</p> <important>
         *        <p>
         *        A message can include only XML, JSON, and unformatted text. The following Unicode characters are
         *        allowed:
         *        </p>
         *        <p>
         *        <code>#x9</code> | <code>#xA</code> | <code>#xD</code> | <code>#x20</code> to <code>#xD7FF</code> |
         *        <code>#xE000</code> to <code>#xFFFD</code> | <code>#x10000</code> to <code>#x10FFFF</code>
         *        </p>
         *        <p>
         *        Any characters not included in this list will be rejected. For more information, see the <a
         *        href="http://www.w3.org/TR/REC-xml/#charsets">W3C specification for characters</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageBody(String messageBody);

        /**
         * <p>
         * The length of time, in seconds, for which to delay a specific message. Valid values: 0 to 900. Maximum: 15
         * minutes. Messages with a positive <code>DelaySeconds</code> value become available for processing after the
         * delay period is finished. If you don't specify a value, the default value for the queue applies.
         * </p>
         * <note>
         * <p>
         * When you set <code>FifoQueue</code>, you can't set <code>DelaySeconds</code> per message. You can set this
         * parameter only on a queue level.
         * </p>
         * </note>
         * 
         * @param delaySeconds
         *        The length of time, in seconds, for which to delay a specific message. Valid values: 0 to 900.
         *        Maximum: 15 minutes. Messages with a positive <code>DelaySeconds</code> value become available for
         *        processing after the delay period is finished. If you don't specify a value, the default value for the
         *        queue applies. </p> <note>
         *        <p>
         *        When you set <code>FifoQueue</code>, you can't set <code>DelaySeconds</code> per message. You can set
         *        this parameter only on a queue level.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder delaySeconds(Integer delaySeconds);

        /**
         * <p>
         * Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For more
         * information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
         * >Amazon SQS message attributes</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * 
         * @param messageAttributes
         *        Each message attribute consists of a <code>Name</code>, <code>Type</code>, and <code>Value</code>. For
         *        more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/sqs-message-metadata.html#sqs-message-attributes"
         *        >Amazon SQS message attributes</a> in the <i>Amazon SQS Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes);

        /**
         * <p>
         * The message system attribute to send. Each message system attribute consists of a <code>Name</code>,
         * <code>Type</code>, and <code>Value</code>.
         * </p>
         * <important>
         * <ul>
         * <li>
         * <p>
         * Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
         * <code>String</code> and its value must be a correctly formatted X-Ray trace header string.
         * </p>
         * </li>
         * <li>
         * <p>
         * The size of a message system attribute doesn't count towards the total size of a message.
         * </p>
         * </li>
         * </ul>
         * </important>
         * 
         * @param messageSystemAttributes
         *        The message system attribute to send. Each message system attribute consists of a <code>Name</code>,
         *        <code>Type</code>, and <code>Value</code>.</p> <important>
         *        <ul>
         *        <li>
         *        <p>
         *        Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must
         *        be <code>String</code> and its value must be a correctly formatted X-Ray trace header string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The size of a message system attribute doesn't count towards the total size of a message.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageSystemAttributesWithStrings(Map<String, MessageSystemAttributeValue> messageSystemAttributes);

        /**
         * <p>
         * The message system attribute to send. Each message system attribute consists of a <code>Name</code>,
         * <code>Type</code>, and <code>Value</code>.
         * </p>
         * <important>
         * <ul>
         * <li>
         * <p>
         * Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must be
         * <code>String</code> and its value must be a correctly formatted X-Ray trace header string.
         * </p>
         * </li>
         * <li>
         * <p>
         * The size of a message system attribute doesn't count towards the total size of a message.
         * </p>
         * </li>
         * </ul>
         * </important>
         * 
         * @param messageSystemAttributes
         *        The message system attribute to send. Each message system attribute consists of a <code>Name</code>,
         *        <code>Type</code>, and <code>Value</code>.</p> <important>
         *        <ul>
         *        <li>
         *        <p>
         *        Currently, the only supported message system attribute is <code>AWSTraceHeader</code>. Its type must
         *        be <code>String</code> and its value must be a correctly formatted X-Ray trace header string.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The size of a message system attribute doesn't count towards the total size of a message.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageSystemAttributes(
                Map<MessageSystemAttributeNameForSends, MessageSystemAttributeValue> messageSystemAttributes);

        /**
         * <p>
         * This parameter applies only to FIFO (first-in-first-out) queues.
         * </p>
         * <p>
         * The token used for deduplication of sent messages. If a message with a particular
         * <code>MessageDeduplicationId</code> is sent successfully, any messages sent with the same
         * <code>MessageDeduplicationId</code> are accepted successfully but aren't delivered during the 5-minute
         * deduplication interval. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
         * > Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Every message must have a unique <code>MessageDeduplicationId</code>,
         * </p>
         * <ul>
         * <li>
         * <p>
         * You may provide a <code>MessageDeduplicationId</code> explicitly.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
         * <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
         * <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the message).
         * </p>
         * </li>
         * <li>
         * <p>
         * If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
         * <code>ContentBasedDeduplication</code> set, the action fails with an error.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
         * overrides the generated one.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within the
         * deduplication interval are treated as duplicates and only one copy of the message is delivered.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message with a
         * <code>MessageDeduplicationId</code> that is the same as the one generated for the first
         * <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of the
         * message is delivered.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be useful for
         * troubleshooting delivery issues).
         * </p>
         * <p>
         * If a message is sent successfully but the acknowledgement is lost and the message is resent with the same
         * <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SQS can't detect duplicate
         * messages.
         * </p>
         * <p>
         * Amazon SQS continues to keep track of the message deduplication ID even after the message is received and
         * deleted.
         * </p>
         * </note>
         * <p>
         * The maximum length of <code>MessageDeduplicationId</code> is 128 characters.
         * <code>MessageDeduplicationId</code> can contain alphanumeric characters (<code>a-z</code>, <code>A-Z</code>,
         * <code>0-9</code>) and punctuation (<code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
         * </p>
         * <p>
         * For best practices of using <code>MessageDeduplicationId</code>, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagededuplicationid-property.html"
         * >Using the MessageDeduplicationId Property</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * 
         * @param messageDeduplicationId
         *        This parameter applies only to FIFO (first-in-first-out) queues.</p>
         *        <p>
         *        The token used for deduplication of sent messages. If a message with a particular
         *        <code>MessageDeduplicationId</code> is sent successfully, any messages sent with the same
         *        <code>MessageDeduplicationId</code> are accepted successfully but aren't delivered during the 5-minute
         *        deduplication interval. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/FIFO-queues-exactly-once-processing.html"
         *        > Exactly-once processing</a> in the <i>Amazon SQS Developer Guide</i>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Every message must have a unique <code>MessageDeduplicationId</code>,
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You may provide a <code>MessageDeduplicationId</code> explicitly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you aren't able to provide a <code>MessageDeduplicationId</code> and you enable
         *        <code>ContentBasedDeduplication</code> for your queue, Amazon SQS uses a SHA-256 hash to generate the
         *        <code>MessageDeduplicationId</code> using the body of the message (but not the attributes of the
         *        message).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you don't provide a <code>MessageDeduplicationId</code> and the queue doesn't have
         *        <code>ContentBasedDeduplication</code> set, the action fails with an error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the queue has <code>ContentBasedDeduplication</code> set, your <code>MessageDeduplicationId</code>
         *        overrides the generated one.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        When <code>ContentBasedDeduplication</code> is in effect, messages with identical content sent within
         *        the deduplication interval are treated as duplicates and only one copy of the message is delivered.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you send one message with <code>ContentBasedDeduplication</code> enabled and then another message
         *        with a <code>MessageDeduplicationId</code> that is the same as the one generated for the first
         *        <code>MessageDeduplicationId</code>, the two messages are treated as duplicates and only one copy of
         *        the message is delivered.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The <code>MessageDeduplicationId</code> is available to the consumer of the message (this can be
         *        useful for troubleshooting delivery issues).
         *        </p>
         *        <p>
         *        If a message is sent successfully but the acknowledgement is lost and the message is resent with the
         *        same <code>MessageDeduplicationId</code> after the deduplication interval, Amazon SQS can't detect
         *        duplicate messages.
         *        </p>
         *        <p>
         *        Amazon SQS continues to keep track of the message deduplication ID even after the message is received
         *        and deleted.
         *        </p>
         *        </note>
         *        <p>
         *        The maximum length of <code>MessageDeduplicationId</code> is 128 characters.
         *        <code>MessageDeduplicationId</code> can contain alphanumeric characters (<code>a-z</code>,
         *        <code>A-Z</code>, <code>0-9</code>) and punctuation (
         *        <code>!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~</code>).
         *        </p>
         *        <p>
         *        For best practices of using <code>MessageDeduplicationId</code>, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagededuplicationid-property.html"
         *        >Using the MessageDeduplicationId Property</a> in the <i>Amazon SQS Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageDeduplicationId(String messageDeduplicationId);

        /**
         * <p>
         * This parameter applies only to FIFO (first-in-first-out) queues.
         * </p>
         * <p>
         * The tag that specifies that a message belongs to a specific message group. Messages that belong to the same
         * message group are processed in a FIFO manner (however, messages in different message groups might be
         * processed out of order). To interleave multiple ordered streams within a single queue, use
         * <code>MessageGroupId</code> values (for example, session data for multiple users). In this scenario, multiple
         * consumers can process the queue, but the session data of each user is processed in a FIFO fashion.
         * </p>
         * <ul>
         * <li>
         * <p>
         * You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
         * <code>MessageGroupId</code>, the action fails.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReceiveMessage</code> might return messages with multiple <code>MessageGroupId</code> values. For each
         * <code>MessageGroupId</code>, the messages are sorted by time sent. The caller can't specify a
         * <code>MessageGroupId</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The length of <code>MessageGroupId</code> is 128 characters. Valid values: alphanumeric characters and
         * punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
         * </p>
         * <p>
         * For best practices of using <code>MessageGroupId</code>, see <a href=
         * "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagegroupid-property.html"
         * >Using the MessageGroupId Property</a> in the <i>Amazon SQS Developer Guide</i>.
         * </p>
         * <important>
         * <p>
         * <code>MessageGroupId</code> is required for FIFO queues. You can't use it for Standard queues.
         * </p>
         * </important>
         * 
         * @param messageGroupId
         *        This parameter applies only to FIFO (first-in-first-out) queues.</p>
         *        <p>
         *        The tag that specifies that a message belongs to a specific message group. Messages that belong to the
         *        same message group are processed in a FIFO manner (however, messages in different message groups might
         *        be processed out of order). To interleave multiple ordered streams within a single queue, use
         *        <code>MessageGroupId</code> values (for example, session data for multiple users). In this scenario,
         *        multiple consumers can process the queue, but the session data of each user is processed in a FIFO
         *        fashion.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You must associate a non-empty <code>MessageGroupId</code> with a message. If you don't provide a
         *        <code>MessageGroupId</code>, the action fails.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReceiveMessage</code> might return messages with multiple <code>MessageGroupId</code> values.
         *        For each <code>MessageGroupId</code>, the messages are sorted by time sent. The caller can't specify a
         *        <code>MessageGroupId</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The length of <code>MessageGroupId</code> is 128 characters. Valid values: alphanumeric characters and
         *        punctuation <code>(!"#$%&amp;'()*+,-./:;&lt;=&gt;?@[\]^_`{|}~)</code>.
         *        </p>
         *        <p>
         *        For best practices of using <code>MessageGroupId</code>, see <a href=
         *        "https://docs.aws.amazon.com/AWSSimpleQueueService/latest/SQSDeveloperGuide/using-messagegroupid-property.html"
         *        >Using the MessageGroupId Property</a> in the <i>Amazon SQS Developer Guide</i>.
         *        </p>
         *        <important>
         *        <p>
         *        <code>MessageGroupId</code> is required for FIFO queues. You can't use it for Standard queues.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageGroupId(String messageGroupId);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SqsRequest.BuilderImpl implements Builder {
        private String queueUrl;

        private String messageBody;

        private Integer delaySeconds;

        private Map<String, MessageAttributeValue> messageAttributes = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, MessageSystemAttributeValue> messageSystemAttributes = DefaultSdkAutoConstructMap.getInstance();

        private String messageDeduplicationId;

        private String messageGroupId;

        private BuilderImpl() {
        }

        private BuilderImpl(SendMessageRequest model) {
            super(model);
            queueUrl(model.queueUrl);
            messageBody(model.messageBody);
            delaySeconds(model.delaySeconds);
            messageAttributes(model.messageAttributes);
            messageSystemAttributesWithStrings(model.messageSystemAttributes);
            messageDeduplicationId(model.messageDeduplicationId);
            messageGroupId(model.messageGroupId);
        }

        public final String getQueueUrl() {
            return queueUrl;
        }

        public final void setQueueUrl(String queueUrl) {
            this.queueUrl = queueUrl;
        }

        @Override
        public final Builder queueUrl(String queueUrl) {
            this.queueUrl = queueUrl;
            return this;
        }

        public final String getMessageBody() {
            return messageBody;
        }

        public final void setMessageBody(String messageBody) {
            this.messageBody = messageBody;
        }

        @Override
        public final Builder messageBody(String messageBody) {
            this.messageBody = messageBody;
            return this;
        }

        public final Integer getDelaySeconds() {
            return delaySeconds;
        }

        public final void setDelaySeconds(Integer delaySeconds) {
            this.delaySeconds = delaySeconds;
        }

        @Override
        public final Builder delaySeconds(Integer delaySeconds) {
            this.delaySeconds = delaySeconds;
            return this;
        }

        public final Map<String, MessageAttributeValue.Builder> getMessageAttributes() {
            Map<String, MessageAttributeValue.Builder> result = MessageBodyAttributeMapCopier
                    .copyToBuilder(this.messageAttributes);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setMessageAttributes(Map<String, MessageAttributeValue.BuilderImpl> messageAttributes) {
            this.messageAttributes = MessageBodyAttributeMapCopier.copyFromBuilder(messageAttributes);
        }

        @Override
        public final Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes) {
            this.messageAttributes = MessageBodyAttributeMapCopier.copy(messageAttributes);
            return this;
        }

        public final Map<String, MessageSystemAttributeValue.Builder> getMessageSystemAttributes() {
            Map<String, MessageSystemAttributeValue.Builder> result = MessageBodySystemAttributeMapCopier
                    .copyToBuilder(this.messageSystemAttributes);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setMessageSystemAttributes(Map<String, MessageSystemAttributeValue.BuilderImpl> messageSystemAttributes) {
            this.messageSystemAttributes = MessageBodySystemAttributeMapCopier.copyFromBuilder(messageSystemAttributes);
        }

        @Override
        public final Builder messageSystemAttributesWithStrings(Map<String, MessageSystemAttributeValue> messageSystemAttributes) {
            this.messageSystemAttributes = MessageBodySystemAttributeMapCopier.copy(messageSystemAttributes);
            return this;
        }

        @Override
        public final Builder messageSystemAttributes(
                Map<MessageSystemAttributeNameForSends, MessageSystemAttributeValue> messageSystemAttributes) {
            this.messageSystemAttributes = MessageBodySystemAttributeMapCopier.copyEnumToString(messageSystemAttributes);
            return this;
        }

        public final String getMessageDeduplicationId() {
            return messageDeduplicationId;
        }

        public final void setMessageDeduplicationId(String messageDeduplicationId) {
            this.messageDeduplicationId = messageDeduplicationId;
        }

        @Override
        public final Builder messageDeduplicationId(String messageDeduplicationId) {
            this.messageDeduplicationId = messageDeduplicationId;
            return this;
        }

        public final String getMessageGroupId() {
            return messageGroupId;
        }

        public final void setMessageGroupId(String messageGroupId) {
            this.messageGroupId = messageGroupId;
        }

        @Override
        public final Builder messageGroupId(String messageGroupId) {
            this.messageGroupId = messageGroupId;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SendMessageRequest build() {
            return new SendMessageRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
