package com.atlassian.logging.log4j.layout;

import com.atlassian.logging.log4j.layout.json.DefaultJsonDataProvider;
import com.atlassian.logging.log4j.layout.json.JsonDataProvider;
import com.atlassian.logging.log4j.layout.json.JsonLayoutHelper;
import org.apache.logging.log4j.core.Core;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginBuilderAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginBuilderFactory;
import org.apache.logging.log4j.core.layout.AbstractStringLayout;

import java.nio.charset.Charset;

/**
 * Produces json logs based on this specification:
 * https://pug.jira-dev.com/wiki/display/CP/RFC+-+Consistent+JSON+logging+format+for+application+logs
 * <br>
 * This layout can eliminate unnecessary stacktrace lines
 * <br>
 * Settings:
 * <br>
 * * dataProvider - path.to.DataProvider - defaults to JsonDataProvider.  Must extend JsonDataProviderInterface
 * This class is used to inject data into the json structure.  Please see {@link com.atlassian.logging.log4j.layout.json.DefaultJsonDataProvider} for
 * default implementation.
 * <br>
 * * FilterFrames - List of class/package names to be filtered out from stacktrace.
 * Please see {@link com.atlassian.logging.log4j.StackTraceCompressor.Builder#filteredFrames(String)} for the syntax.
 * <p>
 * NOTE: does not seem conform https://pug.jira-dev.com/wiki/display/CC/RFC+-+Consistent+JSON+logging+format+for+application+logs
 * (for example, produces serviceId field that MUST NOT be part of message)
 */
@Plugin(name = "AtlassianJsonLayout", category = Core.CATEGORY_NAME, elementType = Layout.ELEMENT_TYPE, printObject = true)
public class AtlassianJsonLayout extends AbstractStringLayout {

    protected final JsonLayoutHelper layoutHelper;

    @Override
    public String toString() {
        return "AtlassianJsonLayout{" +
                "layoutHelper=" + layoutHelper +
                '}';
    }

    private AtlassianJsonLayout(Charset charset, JsonLayoutHelper layoutHelper) {
        super(charset);
        this.layoutHelper = layoutHelper;
        this.layoutHelper.initialise();
    }

    @PluginBuilderFactory
    public static <B extends Builder<B>> B newBuilder() {
        return new Builder<B>().asBuilder();
    }

    @Override
    public String toSerializable(LogEvent event) {
        return layoutHelper.format(event);
    }

    public static class Builder<B extends Builder<B>> extends AbstractStringLayout.Builder<B>
            implements org.apache.logging.log4j.core.util.Builder<AtlassianJsonLayout> {

        @PluginBuilderAttribute
        private String suppressedFields = "";
        @PluginBuilderAttribute
        private String dataProvider = "";
        @PluginBuilderAttribute
        private boolean filteringApplied = true;
        @PluginBuilderAttribute
        private int minimumLines = 6;
        @PluginBuilderAttribute
        private boolean showEludedSummary = false;
        @PluginBuilderAttribute
        private String filteredFrames = "";
        @PluginBuilderAttribute
        private String environmentConfigFilename = "";
        @PluginBuilderAttribute
        private boolean includeLocation = false;
        @PluginBuilderAttribute
        private String additionalFields = "";

        /**
         * Suppresses output of specified fields
         *
         * @param suppressedFields field names separated by commas
         */
        public B setSuppressedFields(String suppressedFields) {
            this.suppressedFields = suppressedFields;
            return asBuilder();
        }

        /**
         * Used for log4j2 configuration.
         *
         * @param dataProvider Fully qualified class name of the {@link JsonDataProvider}
         */
        public B setDataProvider(String dataProvider) {
            this.dataProvider = dataProvider;
            return asBuilder();
        }


        /**
         * Controls whether filtering will be applied to via this pattern
         *
         * @param filteringApplied the boolean flag
         */
        public B setFilteringApplied(boolean filteringApplied) {
            this.filteringApplied = filteringApplied;
            return asBuilder();
        }


        /**
         * Sets how many first N lines of the stack trace are always shown
         *
         * @param minimumLines how many lines to always show
         */
        public B setMinimumLines(int minimumLines) {
            this.minimumLines = minimumLines;
            return asBuilder();
        }

        /**
         * Allows the eluded line to be shown in horizontal summary form
         *
         * @param showEludedSummary true if they are to be shown
         */
        public B setShowEludedSummary(boolean showEludedSummary) {
            this.showEludedSummary = showEludedSummary;
            return asBuilder();
        }

        /**
         * Any stack frame starting with <code>"at "</code> + <code>filter</code> will not be written to the log.
         * <br>
         * You can specify multiples frames separated by commas eg "org.apache.tomcat, org.hibernate"
         *
         * @param filteredFrames class name(s) or package name(s) to be filtered
         */
        public B setFilteredFrames(String filteredFrames) {
            this.filteredFrames = filteredFrames;
            return asBuilder();
        }

        /**
         * The name of a properties file that alters what is used as the <code>env</code>
         * json attribute. By default in {@link DefaultJsonDataProvider} the environment
         * is the <code>studio.env</code> system property.
         * <br>
         * Each key in the properties file either starts with <code>copy-[mode].</code> or <code>split-[mode].</code>.
         * where [mode] can be "prefix" or "suffix".
         * For "split", the log message will only be sent (split) to the alternate environment.
         * For "copy", the log message will be sent to the original and alternate environment.
         * The remainder of the key gives the suffix/prefix of a logger depending on mode. e.g. <code>com.atlassian.foo</code>.
         * The value of that key is a suffix to add to the environment attribute e.g. <code>-clean</code>.
         *
         * @param environmentConfigFilename name of config file (on classpath)
         */
        public B setEnvironmentConfigFilename(String environmentConfigFilename) {
            this.environmentConfigFilename = environmentConfigFilename;
            return asBuilder();
        }

        /**
         * Log4j can include the call location that made the logger call.  In a Splunk world this
         * is dubious given its high cost on every message and low value of information.  So we set this off
         * by default and if a product wants it on, then they can set it in their log4j config
         *
         * @param includeLocation flag on whether to include location
         * @see LogEvent#getSource()
         */
        public B setIncludeLocation(boolean includeLocation) {
            this.includeLocation = includeLocation;
            return asBuilder();
        }

        /**
         * Set additional fields that will be used in events
         * Fields needs to be passed as string in format: "field1:value1,field:value2,..."
         *
         * @param additionalFields fields to be added to event
         */
        public B setAdditionalFields(String additionalFields) {
            this.additionalFields = additionalFields;
            return asBuilder();
        }

        @Override
        public AtlassianJsonLayout build() {
            JsonLayoutHelper layoutHelper = new JsonLayoutHelper();
            if (!suppressedFields.isEmpty()) {
                layoutHelper.setSuppressedFields(suppressedFields);
            }
            if (!dataProvider.isEmpty()) {
                this.setDataProviderObject(layoutHelper, dataProvider);
            }
            layoutHelper.setFilteringApplied(filteringApplied);
            layoutHelper.setMinimumLines(minimumLines);
            layoutHelper.setShowEludedSummary(showEludedSummary);
            if (!filteredFrames.isEmpty()) {
                layoutHelper.setFilteredFrames(filteredFrames);
            }
            if (!environmentConfigFilename.isEmpty()) {
                layoutHelper.setEnvironmentConfigFilename(environmentConfigFilename);
            }
            layoutHelper.setIncludeLocation(includeLocation);
            if (!additionalFields.isEmpty()) {
                layoutHelper.setAdditionalFields(additionalFields);
            }
            return new AtlassianJsonLayout(getCharset(), layoutHelper);
        }

        private void setDataProviderObject(JsonLayoutHelper layoutHelper, final String dataProviderClazz) {
            try {
                layoutHelper.setDataProvider((JsonDataProvider) Class.forName(dataProviderClazz).newInstance());
            } catch (ClassNotFoundException e) {
                throw new RuntimeException("JsonDataProvider implementation not found", e);
            } catch (InstantiationException | IllegalAccessException e) {
                throw new RuntimeException("Failed to instantiate JsonDataProvider implementation", e);
            }
        }
    }
}
