package com.atlassian.logging.log4j.layout.patterns;

import com.atlassian.logging.log4j.CommonConstants;
import com.atlassian.logging.log4j.NewLineSupport;
import com.atlassian.logging.log4j.StackTraceCompressor;
import com.atlassian.logging.log4j.StackTraceInfo;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.pattern.ConverterKeys;
import org.apache.logging.log4j.core.pattern.LogEventPatternConverter;
import org.apache.logging.log4j.core.pattern.PatternConverter;

@Plugin(name = "StackTraceFilteringPatternConverter", category = PatternConverter.CATEGORY)
@ConverterKeys({"stf"})
public class StackTraceFilteringPatternConverter extends LogEventPatternConverter {

    StackTraceCompressor stackTraceCompressor;
    StackTraceFilteringPatternConverterOptions converterOptions;

    StackTraceFilteringPatternConverter(String[] options) {
        super("StackTraceFilteringPatternConverter", "StackTraceFilteringPatternConverter");
        StackTraceFilteringPatternConverterOptions converterOptions = StackTraceFilteringPatternConverterOptions.newInstance(options);
        this.converterOptions = converterOptions;
        stackTraceCompressor = StackTraceCompressor.defaultBuilder(converterOptions.getMinimumLines(), converterOptions.isShowEludedSummary())
                .filteredFrames(converterOptions.getFilteredFrames())
                .filteredEveryThingAfterFrames(converterOptions.getFilterEveryThingAfterFrames())
                .filteredEveryThingAfterMessage(converterOptions.getFilterEverythingAfterMessage())
                .markerAtFrames(converterOptions.getMarkerAtFrames())
                .markerAtMessage(converterOptions.getMarkerAtMessage())
                .replacementToken(converterOptions.getFilterReplacementToken()).build();
    }

    public static StackTraceFilteringPatternConverter newInstance(final String[] options) {
        return new StackTraceFilteringPatternConverter(options);
    }

    @Override
    public void format(LogEvent event, StringBuilder toAppendTo) {
        Throwable throwable = event.getThrown();
        if (throwable != null) {
            toAppendTo.append(formatStackTrace(event, throwable));
        }
    }


    private String formatStackTrace(LogEvent event, Throwable throwable) {
        StringBuffer buffer = new StringBuffer();
        String[] stackTraceLines = getThrowableStrRep(throwable);

        // if they have turned us off or its a debug event then go au natural
        if (!converterOptions.isFilteringApplied() || (Level.DEBUG.equals(event.getLevel()) && !converterOptions.isFilteringAppliedToDebugLevel())) {
            outputPlainThrowable(buffer, stackTraceLines);
        } else {
            stackTraceCompressor.filterStackTrace(buffer, stackTraceLines);
        }

        return buffer.toString();
    }

    private void outputPlainThrowable(StringBuffer buffer, String[] stackTraceLines) {
        NewLineSupport.join(buffer, stackTraceLines);
    }

    protected String[] getThrowableStrRep(final Throwable throwable) {
        return new StackTraceInfo(throwable, CommonConstants.DEFAULT_NEW_LINE_PREFIX, converterOptions.isStackTracePackagingExamined()).getThrowableStrRep();
    }
}
