/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bamboo.accesstoken;

import com.atlassian.bamboo.accesstoken.AccessToken;
import com.atlassian.bamboo.accesstoken.AccessTokenDao;
import com.atlassian.bamboo.accesstoken.AccessTokenGenerator;
import com.atlassian.bamboo.accesstoken.AccessTokenPermission;
import com.atlassian.bamboo.accesstoken.AccessTokenService;
import com.atlassian.bamboo.accesstoken.AccessTokenServiceUtils;
import com.atlassian.bamboo.accesstoken.CreateAccessTokenRequest;
import com.atlassian.bamboo.accesstoken.MutableAccessToken;
import com.atlassian.bamboo.accesstoken.MutableAccessTokenImpl;
import com.atlassian.bamboo.accesstoken.RawAccessToken;
import com.atlassian.bamboo.accesstoken.SimpleRawAccessToken;
import com.atlassian.bamboo.accesstoken.event.AccessTokenCreatedEvent;
import com.atlassian.bamboo.accesstoken.event.AccessTokenDeletedEvent;
import com.atlassian.bamboo.accesstoken.event.AllUserAccessTokensDeletedEvent;
import com.atlassian.bamboo.accesstoken.exception.CreateTokenFailedException;
import com.atlassian.bamboo.accesstoken.exception.TokenLimitExceededException;
import com.atlassian.bamboo.accesstoken.exception.TokenWithNameExistsException;
import com.atlassian.bamboo.core.BambooObject;
import com.atlassian.bamboo.exception.UnauthorisedException;
import com.atlassian.bamboo.persister.AuditLogService;
import com.atlassian.bamboo.security.AccessTokenContextHolder;
import com.atlassian.bamboo.security.BambooPermissionManager;
import com.atlassian.bamboo.security.acegi.acls.BambooPermission;
import com.atlassian.bamboo.user.BambooAuthenticationContext;
import com.atlassian.bamboo.user.BambooUser;
import com.atlassian.bamboo.user.BambooUserManager;
import com.atlassian.bamboo.util.BambooStringUtils;
import com.atlassian.bamboo.utils.SystemProperty;
import com.atlassian.event.api.EventPublisher;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.acegisecurity.acls.Permission;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.springframework.util.CollectionUtils;

public class DefaultAccessTokenService
implements AccessTokenService {
    private static final int MAX_TOKEN_NAME_LENGTH = 255;
    private static final Logger log = Logger.getLogger(DefaultAccessTokenService.class);
    private static int MAX_GENERATE_TOKEN_ID_RETRY = 10;
    private final AccessTokenDao accessTokenDao;
    private final AccessTokenGenerator accessTokenGenerator;
    private final BambooAuthenticationContext bambooAuthenticationContext;
    private final BambooPermissionManager bambooPermissionManager;
    private final EventPublisher eventPublisher;
    private final BambooUserManager bambooUserManager;
    private final AuditLogService auditLogService;

    public DefaultAccessTokenService(AccessTokenDao accessTokenDao, AccessTokenGenerator accessTokenGenerator, BambooAuthenticationContext bambooAuthenticationContext, BambooPermissionManager bambooPermissionManager, EventPublisher eventPublisher, BambooUserManager bambooUserManager, AuditLogService auditLogService) {
        this.accessTokenDao = accessTokenDao;
        this.accessTokenGenerator = accessTokenGenerator;
        this.bambooAuthenticationContext = bambooAuthenticationContext;
        this.bambooPermissionManager = bambooPermissionManager;
        this.eventPublisher = eventPublisher;
        this.bambooUserManager = bambooUserManager;
        this.auditLogService = auditLogService;
    }

    public Optional<AccessToken> authenticate(@NotNull String rawToken) {
        this.validateNotEmpty(rawToken, "rawToken");
        if (!this.accessTokenGenerator.isValidToken(rawToken)) {
            log.debug((Object)"Can not authenticate - token is not valid");
            return Optional.empty();
        }
        String tokenId = this.accessTokenGenerator.getId(rawToken);
        Optional byTokenId = this.accessTokenDao.findByTokenId(tokenId);
        if (byTokenId.isPresent()) {
            MutableAccessToken mutableAccessToken = (MutableAccessToken)byTokenId.get();
            BambooUser bambooUser = this.bambooUserManager.getBambooUser(mutableAccessToken.getUserName());
            if (bambooUser != null && bambooUser.isEnabled()) {
                if (this.accessTokenGenerator.authenticateToken(rawToken, mutableAccessToken.getHashedToken())) {
                    return Optional.of(mutableAccessToken.toAccessToken());
                }
                log.warn((Object)String.format("Someone tried to authenticate with token id %s and invalid secret", tokenId));
            } else {
                log.info((Object)String.format("User correlated with token id %s does not exist or is disabled", mutableAccessToken.getTokenId()));
            }
        } else {
            log.info((Object)String.format("Token with id %s does not exist", tokenId));
        }
        return Optional.empty();
    }

    @NotNull
    public RawAccessToken create(@NotNull CreateAccessTokenRequest createAccessTokenRequest) throws TokenLimitExceededException, CreateTokenFailedException, TokenWithNameExistsException {
        String requestUserName = createAccessTokenRequest.getUserName();
        String requestTokenName = createAccessTokenRequest.getName();
        Set requestPermissions = createAccessTokenRequest.getPermissions();
        this.validateNotEmpty(requestUserName, "requestUserName");
        this.validateNotEmpty(requestTokenName, "requestTokenName");
        this.validateTokenNameMaxLenght(requestTokenName);
        this.validateNotEmpty(requestPermissions, "requestPermissions");
        AccessTokenServiceUtils.validateRequestedPermissionsDependencies(requestPermissions);
        if (!this.isCurrentUser(requestUserName)) {
            throw new UnauthorisedException(String.format("User %s tried to create rawToken for %s. Creating tokens are allowed only for current user.", this.bambooAuthenticationContext.getUserName(), requestUserName));
        }
        this.validateNotAuthorisedByRestrictedToken(requestUserName);
        long userTokensCount = this.accessTokenDao.countByUserName(requestUserName);
        if (userTokensCount >= SystemProperty.MAX_TOKENS_PER_USER.getTypedValue()) {
            throw new TokenLimitExceededException(String.format("User %s has %s tokens. Max number of tokens is : %s, can not create new token", requestUserName, userTokensCount, SystemProperty.MAX_TOKENS_PER_USER.getTypedValue()));
        }
        if (this.accessTokenDao.tokenWithNameAndUserNameExists(requestUserName, requestTokenName)) {
            throw new TokenWithNameExistsException(String.format("Token with name %s already exists", requestTokenName));
        }
        int attempts = 0;
        while (attempts++ < MAX_GENERATE_TOKEN_ID_RETRY) {
            String token = this.accessTokenGenerator.generateToken();
            String tokenId = this.accessTokenGenerator.getId(token);
            Optional byTokenId = this.accessTokenDao.findByTokenId(tokenId);
            if (byTokenId.isPresent()) continue;
            String hashedToken = this.accessTokenGenerator.hashToken(token);
            MutableAccessToken mutableAccessToken = this.saveToken(tokenId, hashedToken, createAccessTokenRequest.getName(), requestUserName, requestPermissions);
            SimpleRawAccessToken simpleRawAccessToken = new SimpleRawAccessToken(mutableAccessToken.getCreationDate(), mutableAccessToken.getTokenId(), mutableAccessToken.getName(), mutableAccessToken.getUserName(), mutableAccessToken.getPermissions(), token);
            this.eventPublisher.publish((Object)new AccessTokenCreatedEvent(simpleRawAccessToken.toAccessToken()));
            this.auditLogService.log(String.format("New access token '%s' has been created for user %s with permissions: [%s]", mutableAccessToken.getName(), requestUserName, BambooStringUtils.enumSetToString((Set)mutableAccessToken.getPermissions())));
            return simpleRawAccessToken;
        }
        throw new CreateTokenFailedException("There was an error creating an access token");
    }

    @NotNull
    public List<AccessToken> getAccessTokensByUserName(@NotNull String userName) {
        this.validateNotEmpty(userName, "userName");
        this.validateCurrentUserPermissionToGetOrDelete(userName);
        return this.accessTokenDao.findByUserName(userName).stream().map(AccessToken::toAccessToken).sorted(Comparator.comparing(o -> o.getName().toUpperCase())).collect(Collectors.toList());
    }

    public void deleteByTokenId(@NotNull String tokenId) {
        this.validateNotEmpty(tokenId, "tokenId");
        Optional byTokenId = this.accessTokenDao.findByTokenId(tokenId);
        byTokenId.ifPresent(mutableAccessToken -> {
            this.validateCurrentUserPermissionToGetOrDelete(mutableAccessToken.getUserName());
            this.validateNotAuthorisedByRestrictedToken(mutableAccessToken.getUserName());
            this.accessTokenDao.delete((BambooObject)mutableAccessToken);
            String currentUser = this.bambooAuthenticationContext.getUserName();
            this.auditLogService.log(String.format("Access token '%s' for user %s has been revoked", mutableAccessToken.getName(), mutableAccessToken.getUserName()));
            this.eventPublisher.publish((Object)new AccessTokenDeletedEvent(mutableAccessToken.toAccessToken(), StringUtils.defaultString((String)currentUser)));
        });
    }

    public void deleteByUserName(@NotNull String userName) throws UnauthorisedException {
        this.validateCurrentUserPermissionToGetOrDelete(userName);
        this.validateNotAuthorisedByRestrictedToken(userName);
        this.accessTokenDao.deleteByUserName(userName);
        this.auditLogService.log(String.format("All access tokens of user %s has been revoked", userName));
        this.eventPublisher.publish((Object)new AllUserAccessTokensDeletedEvent(userName));
    }

    private MutableAccessToken saveToken(String tokenId, String hashedToken, String tokenName, String requestUserName, Iterable<AccessTokenPermission> permissions) {
        MutableAccessTokenImpl mutableAccessToken = new MutableAccessTokenImpl();
        mutableAccessToken.setTokenId(tokenId);
        mutableAccessToken.setHashedToken(hashedToken);
        mutableAccessToken.setUserName(requestUserName);
        mutableAccessToken.setName(tokenName);
        mutableAccessToken.setPermissions(permissions);
        this.accessTokenDao.save((BambooObject)mutableAccessToken);
        return mutableAccessToken;
    }

    private void validateCurrentUserPermissionToGetOrDelete(String userName) {
        if (!this.isAdminOrRestrictedAdmin() && !this.isCurrentUser(userName)) {
            String currentUser = this.bambooAuthenticationContext.getUserName();
            throw new UnauthorisedException(String.format("User %s is not authorised to perform get or delete for %s's access tokens", currentUser, userName));
        }
    }

    private void validateNotAuthorisedByRestrictedToken(String userName) {
        AccessToken accessToken;
        Optional<AccessToken> accessTokenOptional = AccessTokenContextHolder.getAccessToken();
        if (accessTokenOptional.isPresent() && !(accessToken = accessTokenOptional.get()).getPermissions().contains(AccessTokenPermission.USER)) {
            throw new UnauthorisedException(String.format("Operation not allowed due to permission restrictions of the access token: user %s token %s", userName, accessToken.getName()));
        }
    }

    private boolean isAdminOrRestrictedAdmin() {
        return this.bambooPermissionManager.hasGlobalPermission((Permission)BambooPermission.ADMINISTRATION) || this.bambooPermissionManager.hasGlobalPermission((Permission)BambooPermission.RESTRICTEDADMINISTRATION);
    }

    private boolean isCurrentUser(String userName) {
        String currentUserName = this.bambooAuthenticationContext.getUserName();
        return Objects.equals(currentUserName, userName);
    }

    private void validateNotEmpty(String value, String field) {
        if (StringUtils.isEmpty((CharSequence)value)) {
            throw new IllegalArgumentException(String.format("Field %s can not be empty", field));
        }
    }

    private void validateNotEmpty(Collection<?> value, String field) {
        if (CollectionUtils.isEmpty(value)) {
            throw new IllegalArgumentException(String.format("Field %s can not be empty", field));
        }
    }

    private void validateTokenNameMaxLenght(String tokenName) {
        if (tokenName.length() > 255) {
            throw new IllegalArgumentException("Token name cannot be longer than 255characters");
        }
    }
}

