/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bamboo.crypto.instance;

import com.atlassian.bamboo.bandana.BandanaAccessVerifier;
import com.atlassian.bamboo.bandana.PlanAwareBandanaContext;
import com.atlassian.bamboo.fileserver.SystemDirectory;
import com.atlassian.bamboo.utils.BambooFiles;
import com.atlassian.bandana.BandanaContext;
import com.atlassian.bandana.BandanaManager;
import com.atlassian.security.random.DefaultSecureRandomService;
import com.atlassian.security.random.SecureRandomService;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.log4j.Logger;
import org.bouncycastle.crypto.BlockCipher;
import org.bouncycastle.pqc.math.linearalgebra.ByteUtils;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

class InstanceSecretStorage {
    private static final Logger log = Logger.getLogger(InstanceSecretStorage.class);
    private final SecureRandomService secureRandomService = DefaultSecureRandomService.getInstance();
    private static final String PROPERTY_INSTANCE_CIPHER_KEY = BandanaAccessVerifier.createRestrictedKey("instance.cipher.key");
    private static final String PROPERTY_INSTANCE_CIPHER_IV = BandanaAccessVerifier.createRestrictedKey("instance.cipher.iv");
    private final BandanaManager bandanaManager;
    static final int KEY_LENGTH_BITS = 256;
    private final Map<Integer, Pair<byte[], byte[]>> keysAndIvs = new HashMap<Integer, Pair<byte[], byte[]>>();

    InstanceSecretStorage(BandanaManager bandanaManager) {
        this.bandanaManager = bandanaManager;
    }

    private Pair<byte[], byte[]> readKeyAndIv(int cipherDataId) {
        byte[] dbKey = this.getKeyFromDb(cipherDataId);
        byte[] fsKey = this.getKeyFromFs(cipherDataId);
        byte[] iv = this.getIvFromDb(cipherDataId);
        if (dbKey == null && fsKey == null && iv == null) {
            return null;
        }
        if (!InstanceSecretStorage.isDataComplete(dbKey, fsKey, iv)) {
            throw new IllegalArgumentException("Unable to read cipher data for " + cipherDataId);
        }
        byte[] key = this.restoreKey(dbKey, fsKey);
        return Pair.of((Object)key, (Object)iv);
    }

    private static boolean isDataComplete(byte[] dbKey, byte[] fsKey, byte[] iv) {
        if (dbKey != null && fsKey != null && iv != null) {
            return true;
        }
        if (dbKey == null) {
            log.fatal((Object)"Database part of instance key is unavailable");
        }
        if (fsKey == null) {
            log.fatal((Object)"Filesystem part of instance key is unavailable");
        }
        if (iv == null) {
            log.fatal((Object)"Cipher initialisation vector is unavailable");
        }
        return false;
    }

    @NotNull
    public Pair<byte[], byte[]> getKeyAndIv(BlockCipher blockCipher, int cipherDataId, boolean onlyIfPresent) {
        return this.getCipherData(cipherDataId, onlyIfPresent, blockCipher.getBlockSize(), false);
    }

    @NotNull
    public Pair<byte[], byte[]> getKeyAndIvForceReload(BlockCipher blockCipher, int cipherDataId, boolean onlyIfPresent) {
        return this.getCipherData(cipherDataId, onlyIfPresent, blockCipher.getBlockSize(), true);
    }

    @NotNull
    private Pair<byte[], byte[]> getCipherData(int cipherDataId, boolean onlyIfPresent, int blockSize, boolean forceReload) {
        Pair<byte[], byte[]> cipherData;
        if (onlyIfPresent) {
            cipherData = this.getCipherData(cipherDataId, forceReload);
            Preconditions.checkArgument((cipherData != null ? 1 : 0) != 0, (Object)("Unknown cipher data id " + cipherDataId));
        } else {
            cipherData = this.getOrCreateCipherData(blockSize, cipherDataId, forceReload);
        }
        return cipherData;
    }

    @NotNull
    private synchronized Pair<byte[], byte[]> getOrCreateCipherData(int blockSizeBytes, int cipherDataId, boolean forceReload) {
        Pair<byte[], byte[]> keyAndIv = this.getCipherData(cipherDataId, forceReload);
        if (keyAndIv == null) {
            keyAndIv = this.readKeyAndIv(cipherDataId);
            if (keyAndIv == null) {
                this.generateCipherData(blockSizeBytes, cipherDataId);
                keyAndIv = this.readKeyAndIv(cipherDataId);
            }
            Preconditions.checkState((keyAndIv != null ? 1 : 0) != 0, (Object)"Expected crypto data to be available, but it was not.");
            this.keysAndIvs.put(cipherDataId, keyAndIv);
        }
        return keyAndIv;
    }

    private synchronized void loadCipherData(int cipherDataId) {
        Pair<byte[], byte[]> keyAndIv = this.readKeyAndIv(cipherDataId);
        this.keysAndIvs.put(cipherDataId, keyAndIv);
    }

    private synchronized Pair<byte[], byte[]> getCipherData(int cipherDataId, boolean forceReload) {
        if (forceReload || !this.keysAndIvs.containsKey(cipherDataId)) {
            this.loadCipherData(cipherDataId);
        }
        return this.keysAndIvs.get(cipherDataId);
    }

    private void generateCipherData(int blockSizeBytes, int cipherDataId) {
        log.info((Object)("Initialising instance cryptography for cipher data id " + cipherDataId + "..."));
        log.info((Object)"Generating instance cipher key...");
        byte[] key = InstanceSecretStorage.newByteArrayForKey();
        this.secureRandomService.nextBytes(key);
        this.storeKeyInDbAndFs(key, cipherDataId);
        this.storeIv(blockSizeBytes, cipherDataId);
        log.info((Object)"Instance cryptography initialised.");
    }

    private void storeIv(int blockSizeBytes, int cipherDataId) {
        log.info((Object)"Generating instance initialisation vector...");
        byte[] iv = new byte[blockSizeBytes];
        this.secureRandomService.nextBytes(iv);
        InstanceSecretStorage.setValueByteArray(this.bandanaManager, InstanceSecretStorage.getInstanceCipherIvPropertyName(cipherDataId), iv);
    }

    @Nullable
    private byte[] getKeyFromFs(int cipherDataId) {
        Path keyFile = this.getKeyFile(cipherDataId);
        if (!Files.exists(keyFile, new LinkOption[0])) {
            return null;
        }
        try {
            return Files.readAllBytes(keyFile);
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Nullable
    private byte[] getKeyFromDb(int cipherDataId) {
        return InstanceSecretStorage.getValueByteArray(this.bandanaManager, InstanceSecretStorage.getInstanceCipherKeyPropertyName(cipherDataId));
    }

    @Nullable
    private byte[] getIvFromDb(int cipherDataId) {
        return InstanceSecretStorage.getValueByteArray(this.bandanaManager, InstanceSecretStorage.getInstanceCipherIvPropertyName(cipherDataId));
    }

    private void storeKeyInDbAndFs(byte[] key, int cipherDataId) {
        Path keyFile = this.createKeyFile(cipherDataId);
        log.info((Object)"Securely storing instance key...");
        byte[] random = InstanceSecretStorage.newByteArrayForKey();
        this.secureRandomService.nextBytes(random);
        byte[] fsKeyPart = ByteUtils.xor((byte[])random, (byte[])key);
        try {
            Files.write(keyFile, fsKeyPart, new OpenOption[0]);
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
        InstanceSecretStorage.setValueByteArray(this.bandanaManager, InstanceSecretStorage.getInstanceCipherKeyPropertyName(cipherDataId), random);
    }

    private Path createKeyFile(int cipherDataId) {
        Path keyFile = this.getKeyFile(cipherDataId);
        try {
            Path parent = keyFile.getParent();
            Files.createDirectories(parent, new FileAttribute[0]);
            BambooFiles.setAccessibleOnlyByOwner((Path)parent);
            Files.write(keyFile, new byte[0], new OpenOption[0]);
            BambooFiles.setAccessibleOnlyByOwner((Path)keyFile);
            return keyFile;
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private Path getKeyFile(int cipherDataId) {
        Path instanceCipherDirectory = this.getInstanceCipherDirectory();
        return instanceCipherDirectory.resolve("cipher.key_" + cipherDataId);
    }

    @VisibleForTesting
    protected Path getInstanceCipherDirectory() {
        Path configDirectory = Paths.get(SystemDirectory.getConfigDirectory().getAbsolutePath(), new String[0]);
        return configDirectory.resolve("cipher");
    }

    private static byte[] newByteArrayForKey() {
        return new byte[32];
    }

    private byte[] restoreKey(byte[] dbKey, byte[] fsKey) {
        return ByteUtils.xor((byte[])dbKey, (byte[])fsKey);
    }

    @Contract(value="null -> null; !null -> !null")
    private static byte[] stringToBytes(String value) {
        if (value == null) {
            return null;
        }
        return Base64.getDecoder().decode(value);
    }

    @NotNull
    private static String bytesToString(@NotNull byte[] bytes) {
        return Base64.getEncoder().encodeToString(bytes);
    }

    private static void setValueByteArray(BandanaManager bandanaManager, String key, byte[] value) {
        bandanaManager.setValue((BandanaContext)PlanAwareBandanaContext.GLOBAL_CONTEXT, key, (Object)InstanceSecretStorage.bytesToString(value));
    }

    @Nullable
    private static byte[] getValueByteArray(BandanaManager bandanaManager, @NotNull String key) {
        String value = (String)bandanaManager.getValue((BandanaContext)PlanAwareBandanaContext.GLOBAL_CONTEXT, key);
        return InstanceSecretStorage.stringToBytes(value);
    }

    @VisibleForTesting
    static String getInstanceCipherKeyPropertyName(int cipherDataId) {
        return PROPERTY_INSTANCE_CIPHER_KEY + "_" + cipherDataId;
    }

    @VisibleForTesting
    static String getInstanceCipherIvPropertyName(int cipherDataId) {
        return PROPERTY_INSTANCE_CIPHER_IV + "_" + cipherDataId;
    }
}

