/*
 * Copyright © 2012 - 2013 Atlassian Corporation Pty Ltd.
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0.
 * Unless required by applicable law or agreed to in writing, software distributed under the License is
 * distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under the License.
 */

package com.atlassian.jira.testkit.client;

import com.sun.jersey.api.client.GenericType;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;
import org.codehaus.jackson.annotate.JsonAutoDetect;

import java.util.List;

/**
 * Use this class from func/selenium/page-object tests that need to manipulate SearchRequests.
 *
 * See <code>com.atlassian.jira.testkit.plugin.SearchRequestBackdoor</code> in jira-testkit-plugin for backend.
 *
 * @since v5.0
 */
public class SearchRequestControl extends BackdoorControl<SearchRequestControl>
{
    private static final GenericType<List<SearchBean>> SEARCH_LIST_TYPE = new GenericType<List<SearchBean>>(){};

    public SearchRequestControl(JIRAEnvironmentData environmentData)
    {
        super(environmentData);
    }

    /**
     * Creates a SearchRequest with the specified query and returns the id of the object created. The share level of
     * the filter will be Private.
     *
     * @param username  the user to save the search for
     * @param searchJql the JQL of the search
     * @param searchName the name of the search
     * @param searchDescription the description of the search
     *
     * @return String id of the SearchRequest created
     */
    public String createFilter(String username, String searchJql, String searchName, String searchDescription)
    {
        return createFilter(username, searchJql, searchName, searchDescription, "");
    }

    /**
     * Creates a SearchRequest with the specified query and returns the id of the object created. The share level of
     * the filter will be determined by the JSON share string. See SharePermissionUtils for how the JSON is handled.
     *
     * @param username  the user to save the search for
     * @param searchJql the JQL of the search
     * @param searchName the name of the search
     * @param searchDescription the description of the search
     * @param jsonShareString a JSON array of permissions to apply to the saved search
     *
     * @return String id of the SearchRequest created
     */
    public String createFilter(String username, String searchJql, String searchName, String searchDescription,
            String jsonShareString)
    {
        SearchBean searchBean = new SearchBean(username, searchJql, searchName, searchDescription, jsonShareString);
        return createResource().path("filter").post(String.class, searchBean);
    }

    public List<SearchBean> getOwnedFilters(String username)
    {
        return createResource().path("filter").path("my").queryParam("username", username).get(SEARCH_LIST_TYPE);
    }
    
    @JsonAutoDetect
    public static class SearchBean
    {
        public String username;
        public String searchJql;
        public String searchName;
        public String searchDescription;
        public String jsonShareString;
        public boolean favourite;
        public long favouriteCount;

        public SearchBean()
        {
        }
        
        public SearchBean(String username, String searchJql, String searchName, String searchDescription,
                String jsonShareString)
        {
            this.username = username;
            this.searchJql = searchJql;
            this.searchName = searchName;
            this.searchDescription = searchDescription;
            this.jsonShareString = jsonShareString;
        }

        public String getUsername()
        {
            return username;
        }

        public SearchBean setUsername(String username)
        {
            this.username = username;
            return this;
        }

        public String getSearchJql()
        {
            return searchJql;
        }

        public SearchBean setSearchJql(String searchJql)
        {
            this.searchJql = searchJql;
            return this;
        }

        public String getSearchName()
        {
            return searchName;
        }

        public SearchBean setSearchName(String searchName)
        {
            this.searchName = searchName;
            return this;
        }

        public String getSearchDescription()
        {
            return searchDescription;
        }

        public SearchBean setSearchDescription(String searchDescription)
        {
            this.searchDescription = searchDescription;
            return this;
        }

        public String getJsonShareString()
        {
            return jsonShareString;
        }

        public SearchBean setJsonShareString(String jsonShareString)
        {
            this.jsonShareString = jsonShareString;
            return this;
        }

        public boolean isFavourite()
        {
            return favourite;
        }

        public SearchBean setFavourite(boolean favourite)
        {
            this.favourite = favourite;
            return this;
        }

        public long getFavouriteCount()
        {
            return favouriteCount;
        }

        public SearchBean setFavouriteCount(long favouriteCount)
        {
            this.favouriteCount = favouriteCount;
            return this;
        }

        @Override
        public String toString()
        {
            return ToStringBuilder.reflectionToString(this, ToStringStyle.SHORT_PREFIX_STYLE);
        }
    }
}
