package com.atlassian.mail.server;

import com.atlassian.mail.server.auth.AuthenticationContext;
import com.atlassian.mail.server.auth.Credentials;
import com.atlassian.mail.server.auth.UserPasswordCredentials;
import org.apache.commons.lang3.StringUtils;

import javax.mail.MessagingException;
import javax.mail.Service;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;

/**
 * Internal backward-compatible authenticato which supports default user/password authentication
 *
 */
final class InternalAuthenticationContext implements AuthenticationContext {

    private String userName = null;

    private String password = null;

    public InternalAuthenticationContext() {
        super();
    }

    public InternalAuthenticationContext(String userName, String password) {
        this();
        setUserName(userName);
        setPassword(password);
    }

    String getUserName() {
        return userName;
    }

    String getPassword() {
        return password;
    }

    void setPassword(String password) {
        if (StringUtils.isNotBlank(password))
            this.password = password;
        else
            this.password = null;
    }

    void setUserName(String userName) {
        if (StringUtils.isNotBlank(userName))
            this.userName = userName;
        else
            this.userName = null;
    }

    @Override
    public void connectService(Service service) throws MessagingException {
        service.connect(this.userName, this.password);
    }

    @Override
    public Credentials getCredentials() {
        return new InternalUserPasswordCredentials();
    }

    @Override
    public boolean isAuthenticating() {
        return this.userName != null;
    }

    @Override
    public Properties preparePropertiesForSession(Properties properties) {
        String protocol = properties.getProperty("mail.transport.protocol");
        properties.put("mail." + protocol + ".auth", Boolean.toString(isAuthenticating()));
        return properties;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        InternalAuthenticationContext that = (InternalAuthenticationContext) o;
        return Objects.equals(userName, that.userName) &&
                Objects.equals(password, that.password);
    }

    @Override
    public int hashCode() {
        return Objects.hash(userName, password);
    }

    private final class InternalUserPasswordCredentials implements UserPasswordCredentials {
        @Override
        public String getUserName() {
            return userName;
        }

        @Override
        public String getPassword() {
            return password;
        }

        @Override
        public Optional<Properties> getProperties() {
            return Optional.ofNullable(null);
        }
    }
}
