/*
 * Copyright 2021 Expedia, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.expediagroup.graphql.client.jackson.types

import com.expediagroup.graphql.client.jackson.serializers.OptionalInputSerializer
import com.fasterxml.jackson.annotation.JsonCreator
import com.fasterxml.jackson.annotation.JsonValue
import com.fasterxml.jackson.databind.annotation.JsonSerialize

@JsonSerialize(using = OptionalInputSerializer::class)
sealed class OptionalInput<out T> {
    /**
     * Represents missing/undefined value.
     */
    @JsonSerialize(using = OptionalInputSerializer::class)
    object Undefined : OptionalInput<Nothing>() {
        override fun toString() = "Undefined"
    }

    /**
     * Wrapper holding explicitly specified value including NULL.
     */
    @JsonSerialize(using = OptionalInputSerializer::class)
    class Defined<out U> @JsonCreator constructor(@JsonValue val value: U?) : OptionalInput<U>() {
        override fun toString(): String = "Defined(value=$value)"
    }
}

/**
 * Jackson value filter that is used to determine whether to serialize the `OptionalInput` or not.
 */
class UndefinedFilter {
    override fun equals(other: Any?): Boolean = OptionalInput.Undefined == other
}
