/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ImmutableFloatSet;
import com.gs.collections.api.set.primitive.FloatSet;
import com.gs.collections.api.set.primitive.MutableFloatSet;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedFloatCollection;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableFloatSet}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link FloatIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveSet.stg.
 * </p>
 *
 * @see MutableFloatSet#asSynchronized()
 * @see MutableSet#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedFloatSet
        extends AbstractSynchronizedFloatCollection
        implements MutableFloatSet
{
    private static final long serialVersionUID = 1L;

    SynchronizedFloatSet(MutableFloatSet set)
    {
        super(set);
    }

    SynchronizedFloatSet(MutableFloatSet set, Object newLock)
    {
        super(set, newLock);
    }

    /**
     * This method will take a MutableFloatSet and wrap it directly in a SynchronizedFloatSet.
     */
    public static SynchronizedFloatSet of(MutableFloatSet set)
    {
        return new SynchronizedFloatSet(set);
    }

    /**
     * This method will take a MutableFloatSet and wrap it directly in a SynchronizedFloatSet.
     * Additionally, a developer specifies which lock to use with the collection.
     */
    public static SynchronizedFloatSet of(MutableFloatSet set, Object lock)
    {
        return new SynchronizedFloatSet(set, lock);
    }

    @GuardedBy("getLock()")
    private MutableFloatSet getMutableFloatSet()
    {
        return (MutableFloatSet) this.getFloatCollection();
    }

    @Override
    public SynchronizedFloatSet without(float element)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatSet().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedFloatSet with(float element)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatSet().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedFloatSet withAll(FloatIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatSet().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedFloatSet withoutAll(FloatIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatSet().removeAll(elements);
        }
        return this;
    }

    @Override
    public MutableFloatSet select(FloatPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatSet().select(predicate);
        }
    }

    @Override
    public MutableFloatSet reject(FloatPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatSet().reject(predicate);
        }
    }

    @Override
    public <V> MutableSet<V> collect(FloatToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatSet().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherSet)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatSet().equals(otherSet);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatSet().hashCode();
        }
    }

    @Override
    public LazyFloatIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyFloatIterableAdapter(this);
        }
    }

    @Override
    public MutableFloatSet asUnmodifiable()
    {
        return new UnmodifiableFloatSet(this);
    }

    @Override
    public MutableFloatSet asSynchronized()
    {
        return this;
    }

    public FloatSet freeze()
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatSet().freeze();
        }
    }

    @Override
    public ImmutableFloatSet toImmutable()
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatSet().toImmutable();
        }
    }

    public <T> T injectInto(T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatSet().injectInto(injectedValue, function);
        }
    }
}
