/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.internal.primitive;

import java.util.Collection;

import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectFloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatProcedure;
import com.gs.collections.api.collection.primitive.MutableFloatCollection;
import com.gs.collections.api.iterator.FloatIterator;

/**
 * The FloatIteratorIterate class provides implementations of the various iteration patterns for use with the {@link FloatIterator}.
 * This file was automatically generated from template file primitiveIteratorIterate.stg.
 *
 * @since 5.0
 */
public final class FloatIteratorIterate
{
    private FloatIteratorIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    public static void forEach(FloatIterator iterator, FloatProcedure procedure)
    {
        while (iterator.hasNext())
        {
            procedure.value(iterator.next());
        }
    }

    public static <R extends MutableFloatCollection> R select(
            FloatIterator iterator,
            FloatPredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            float item = iterator.next();
            if (predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <R extends MutableFloatCollection> R reject(
            FloatIterator iterator,
            FloatPredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            float item = iterator.next();
            if (!predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <V, R extends Collection<V>> R collect(
            FloatIterator iterator,
            FloatToObjectFunction<? extends V> function,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            float item = iterator.next();
            targetCollection.add(function.valueOf(item));
        }
        return targetCollection;
    }

    public static float detectIfNone(FloatIterator iterator, FloatPredicate predicate, float ifNone)
    {
        while (iterator.hasNext())
        {
            float item = iterator.next();
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public static int count(FloatIterator iterator, FloatPredicate predicate)
    {
        int count = 0;
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                count++;
            }
        }
        return count;
    }

    public static boolean anySatisfy(FloatIterator iterator, FloatPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return true;
            }
        }
        return false;
    }

    public static boolean allSatisfy(FloatIterator iterator, FloatPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (!predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static boolean noneSatisfy(FloatIterator iterator, FloatPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static <T> T injectInto(FloatIterator iterator, T injectedValue, ObjectFloatToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        while(iterator.hasNext())
        {
            result = function.valueOf(result, iterator.next());
        }
        return result;
    }

    public static double sum(FloatIterator iterator)
    {
        double sum = 0.0;
        while (iterator.hasNext())
        {
            sum += iterator.next();
        }
        return sum;
    }

    public static float max(FloatIterator iterator)
    {
        float max = iterator.next();
        while (iterator.hasNext())
        {
            float next = iterator.next();
            if (Float.compare(max, next) < 0)
            {
                max = next;
            }
        }
        return max;
    }

    public static float min(FloatIterator iterator)
    {
        float min = iterator.next();
        while (iterator.hasNext())
        {
            float next = iterator.next();
            if (Float.compare(next, min) < 0)
            {
                min = next;
            }
        }
        return min;
    }

}
