/*
 * Copyright 2014 ArcBees Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.gwtplatform.dispatch.rest.rebind.utils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.google.gwt.core.ext.UnableToCompleteException;
import com.gwtplatform.dispatch.rest.rebind.GeneratorWithInput;
import com.gwtplatform.dispatch.rest.rebind.GeneratorWithoutInput;
import com.gwtplatform.dispatch.rest.rebind.HasPriority;

public class Generators {
    private static final Comparator<HasPriority> COMPARATOR = new Comparator<HasPriority>() {
        @Override
        public int compare(HasPriority o1, HasPriority o2) {
            return o1.getPriority() - o2.getPriority();
        }
    };

    /**
     * Get the best suited generator for the given type.
     *
     * @throws UnableToCompleteException If no generators are found.
     */
    public static <T extends HasPriority & GeneratorWithInput<? super I, ?>, I> T getGenerator(
            Logger logger, Collection<T> generators, I input) throws UnableToCompleteException {
        T generator = findGenerator(generators, input);

        if (generator == null) {
            logger.die("Unable to find an appropriate generator for '%s'", input);
        }

        return generator;
    }

    /**
     * Find the best suited generator for the given type.
     *
     * @return the best suited generator for input or {@code null} if none are found.
     */
    public static <T extends HasPriority & GeneratorWithInput<? super I, ?>, I> T findGenerator(
            Collection<T> generators, I input) {
        List<T> sortedGenerators = sortGenerators(generators);

        for (T generator : sortedGenerators) {
            if (generator.canGenerate(input)) {
                return generator;
            }
        }

        return null;
    }

    /**
     * Find the best suited generator for the given type.
     *
     * @return the best suited generator or {@code null} if none are found.
     */
    public static <T extends HasPriority & GeneratorWithoutInput<?>> T findGeneratorWithoutInput(
            Collection<T> generators) {
        List<T> sortedGenerators = sortGenerators(generators);

        for (T generator : sortedGenerators) {
            if (generator.canGenerate()) {
                return generator;
            }
        }

        return null;
    }

    /**
     * Find all generators capable of handling the given input.
     *
     * @return the {@link Collection} containing all generators (or none) capable of handling <code>input</code>.
     */
    public static <G extends GeneratorWithInput<? super I, ?>, I> Collection<G> findGenerators(
            Collection<G> generators, I input) {
        List<G> collection = new ArrayList<G>();

        for (G generator : generators) {
            if (generator.canGenerate(input)) {
                collection.add(generator);
            }
        }

        return collection;
    }

    /**
     * Find all generators capable of handling the given input then execute them.
     *
     * @return the {@link Collection} containing all outputs generated by the consecutive execution of all valid
     * generators.
     */
    public static <G extends GeneratorWithInput<? super I, O>, I, O> Collection<O> executeAll(
            Collection<G> generators, I input) throws UnableToCompleteException {
        Collection<G> validGenerators = findGenerators(generators, input);
        List<O> outputs = new ArrayList<O>(validGenerators.size());

        for (G generator : generators) {
            O output = generator.generate(input);
            outputs.add(output);
        }

        return outputs;
    }

    /**
     * Sort the provided generators by weight without modifying the original collection.
     */
    private static <T extends HasPriority> List<T> sortGenerators(Collection<T> generators) {
        List<T> sortedGenerators = new ArrayList<T>(generators);
        Collections.sort(sortedGenerators, COMPARATOR);

        return sortedGenerators;
    }
}
