/*
 * Decompiled with CFR 0.152.
 */
package com.helger.peppol.utils;

import com.helger.commons.ValueEnforcer;
import com.helger.commons.annotation.Nonempty;
import com.helger.commons.annotation.ReturnsMutableCopy;
import com.helger.commons.collection.impl.CommonsArrayList;
import com.helger.commons.collection.impl.ICommonsList;
import com.helger.commons.functional.IToBooleanFunction;
import com.helger.commons.state.ETriState;
import com.helger.peppol.utils.CertificateRevocationChecker;
import com.helger.peppol.utils.EPeppolCertificateCheckResult;
import com.helger.peppol.utils.ERevocationCheckMode;
import com.helger.peppol.utils.PeppolKeyStoreHelper;
import java.io.IOException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.time.OffsetDateTime;
import java.util.Date;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import javax.annotation.concurrent.ThreadSafe;
import javax.security.auth.x500.X500Principal;
import net.jodah.expiringmap.ExpirationPolicy;
import net.jodah.expiringmap.ExpiringMap;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.x509.BasicConstraints;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public final class PeppolCertificateChecker {
    public static final boolean DEFAULT_CACHE_OSCP_RESULTS = true;
    private static final Logger LOGGER = LoggerFactory.getLogger(PeppolCertificateChecker.class);
    private static final AtomicBoolean CACHE_OCSP_RESULTS = new AtomicBoolean(true);
    private static final PeppolRevocationCache REVOCATION_CACHE_AP = new PeppolRevocationCache((IToBooleanFunction<X509Certificate>)((IToBooleanFunction)x509Certificate -> ((PeppolRevocationCheckBuilder)PeppolCertificateChecker.peppolRevocationCheck().certificate((X509Certificate)x509Certificate)).validCAsPeppolAP().build().isRevoked()));
    private static final PeppolRevocationCache REVOCATION_CACHE_SMP = new PeppolRevocationCache((IToBooleanFunction<X509Certificate>)((IToBooleanFunction)x509Certificate -> ((PeppolRevocationCheckBuilder)PeppolCertificateChecker.peppolRevocationCheck().certificate((X509Certificate)x509Certificate)).validCAsPeppolSMP().build().isRevoked()));
    private static final ICommonsList<X509Certificate> PEPPOL_AP_CA_CERTS = new CommonsArrayList();
    private static final ICommonsList<X500Principal> PEPPOL_AP_CA_ISSUERS = new CommonsArrayList();
    private static final ICommonsList<X509Certificate> PEPPOL_SMP_CA_CERTS = new CommonsArrayList();
    private static final ICommonsList<X500Principal> PEPPOL_SMP_CA_ISSUERS = new CommonsArrayList();

    private PeppolCertificateChecker() {
    }

    public static boolean isCacheOCSPResults() {
        return CACHE_OCSP_RESULTS.get();
    }

    public static void setCacheOCSPResults(boolean bl) {
        CACHE_OCSP_RESULTS.set(bl);
        LOGGER.info("Global PeppolCertificateChecker OCSP cache enabled: " + bl);
    }

    public static void clearOCSPCache() {
        REVOCATION_CACHE_AP.clearCache();
        REVOCATION_CACHE_SMP.clearCache();
        LOGGER.info("The PeppolCertificateChecker OCSP cache was cleared");
    }

    @Nonnull
    public static PeppolRevocationCache getRevocationCacheAP() {
        return REVOCATION_CACHE_AP;
    }

    @Nonnull
    public static PeppolRevocationCache getRevocationCacheSMP() {
        return REVOCATION_CACHE_SMP;
    }

    private static boolean _isCA(@Nonnull X509Certificate x509Certificate) {
        byte[] byArray = x509Certificate.getExtensionValue(Extension.basicConstraints.getId());
        if (byArray != null) {
            try {
                ASN1Sequence aSN1Sequence;
                BasicConstraints basicConstraints;
                ASN1Primitive aSN1Primitive = JcaX509ExtensionUtils.parseExtensionValue((byte[])byArray);
                if (aSN1Primitive instanceof ASN1Sequence && (basicConstraints = BasicConstraints.getInstance((Object)(aSN1Sequence = (ASN1Sequence)aSN1Primitive))) != null) {
                    return basicConstraints.isCA();
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return false;
    }

    public static void addTrustedPeppolAPCACertificate(@Nonnull X509Certificate x509Certificate) {
        ValueEnforcer.notNull((Object)x509Certificate, (String)"Certificate");
        if (!PeppolCertificateChecker._isCA(x509Certificate)) {
            throw new IllegalArgumentException("The provided AP certificate does not seem to be a CA: " + x509Certificate);
        }
        if (PEPPOL_AP_CA_CERTS.contains((Object)x509Certificate)) {
            throw new IllegalArgumentException("AP certificate is already trusted as Peppol AP CA: " + x509Certificate);
        }
        PEPPOL_AP_CA_CERTS.add((Object)x509Certificate);
        PEPPOL_AP_CA_ISSUERS.add((Object)x509Certificate.getSubjectX500Principal());
    }

    public static void clearTrustedPeppolAPCACertificates() {
        LOGGER.warn("Explicitly removing all " + PEPPOL_AP_CA_CERTS.size() + " entries from the list of trusted Peppol AP CA certificates");
        PEPPOL_AP_CA_CERTS.clear();
        PEPPOL_AP_CA_ISSUERS.clear();
    }

    public static void addTrustedPeppolSMPCACertificate(@Nonnull X509Certificate x509Certificate) {
        ValueEnforcer.notNull((Object)x509Certificate, (String)"Certificate");
        if (!PeppolCertificateChecker._isCA(x509Certificate)) {
            throw new IllegalArgumentException("The provided SMP certificate does not seem to be a CA: " + x509Certificate);
        }
        if (PEPPOL_SMP_CA_CERTS.contains((Object)x509Certificate)) {
            throw new IllegalArgumentException("SMP certificate is already trusted as Peppol AP CA: " + x509Certificate);
        }
        PEPPOL_SMP_CA_CERTS.add((Object)x509Certificate);
        PEPPOL_SMP_CA_ISSUERS.add((Object)x509Certificate.getSubjectX500Principal());
    }

    public static void clearTrustedPeppolSMPCACertificates() {
        LOGGER.warn("Explicitly removing all " + PEPPOL_SMP_CA_CERTS.size() + " entries from the list of trusted Peppol SMP CA certificates");
        PEPPOL_SMP_CA_CERTS.clear();
        PEPPOL_SMP_CA_ISSUERS.clear();
    }

    @Nonnull
    @Nonempty
    @ReturnsMutableCopy
    public static ICommonsList<X509Certificate> getAllPeppolAPCACertificates() {
        return (ICommonsList)PEPPOL_AP_CA_CERTS.getClone();
    }

    @Nonnull
    @Nonempty
    @ReturnsMutableCopy
    public static ICommonsList<X500Principal> getAllPeppolAPCAIssuers() {
        return (ICommonsList)PEPPOL_AP_CA_ISSUERS.getClone();
    }

    @Nonnull
    @Nonempty
    @ReturnsMutableCopy
    public static ICommonsList<X509Certificate> getAllPeppolSMPCACertificates() {
        return (ICommonsList)PEPPOL_SMP_CA_CERTS.getClone();
    }

    @Nonnull
    @Nonempty
    @ReturnsMutableCopy
    public static ICommonsList<X500Principal> getAllPeppolSMPCAIssuers() {
        return (ICommonsList)PEPPOL_SMP_CA_ISSUERS.getClone();
    }

    public static PeppolRevocationCheckBuilder peppolRevocationCheck() {
        return new PeppolRevocationCheckBuilder();
    }

    @Nonnull
    public static EPeppolCertificateCheckResult checkCertificate(@Nullable ICommonsList<X500Principal> iCommonsList, @Nullable PeppolRevocationCache peppolRevocationCache, @Nonnull CertificateRevocationChecker.AbstractRevocationCheckBuilder<?> abstractRevocationCheckBuilder) {
        X509Certificate x509Certificate;
        ValueEnforcer.notNull(abstractRevocationCheckBuilder, (String)"RevocationChecker");
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Running Peppol Certificate Check" + (String)(iCommonsList != null ? " against a list of " + iCommonsList.size() + " certificate issuers" : "") + (peppolRevocationCache != null ? "; a cache is provided" : "; not using a cache"));
        }
        if ((x509Certificate = abstractRevocationCheckBuilder.certificate()) == null) {
            LOGGER.warn("No Peppol Certificate was provided to the certificate check");
            return EPeppolCertificateCheckResult.NO_CERTIFICATE_PROVIDED;
        }
        Date date = abstractRevocationCheckBuilder.checkDate();
        try {
            if (date == null) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Checking the Peppol Certificate validity against the current date time");
                }
                x509Certificate.checkValidity();
            } else {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Checking the Peppol Certificate validity against the provided date time " + date);
                }
                x509Certificate.checkValidity(date);
            }
        }
        catch (CertificateNotYetValidException certificateNotYetValidException) {
            LOGGER.warn("The provided Peppol Certificate is not yet valid per " + (date == null ? "now" : date.toString()));
            return EPeppolCertificateCheckResult.NOT_YET_VALID;
        }
        catch (CertificateExpiredException certificateExpiredException) {
            LOGGER.warn("The provided Peppol Certificate is expired per " + (date == null ? "now" : date.toString()));
            return EPeppolCertificateCheckResult.EXPIRED;
        }
        if (iCommonsList != null) {
            X500Principal x500Principal = x509Certificate.getIssuerX500Principal();
            if (!iCommonsList.contains((Object)x500Principal)) {
                LOGGER.warn("The provided Peppol Certificate issuer '" + x500Principal + "' is not in the list of trusted issuers");
                return EPeppolCertificateCheckResult.UNSUPPORTED_ISSUER;
            }
        } else if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Not testing against known Peppol Certificate issuers");
        }
        if (peppolRevocationCache != null) {
            boolean bl;
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Testing if the Peppol Certificate is revoked, using a cache");
            }
            if (bl = peppolRevocationCache.isRevoked(x509Certificate)) {
                LOGGER.warn("The Peppol Certificate is revoked [caching used]");
                return EPeppolCertificateCheckResult.REVOKED;
            }
        } else {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Testing if the Peppol Certificate is revoked, without a cache");
            }
            if (abstractRevocationCheckBuilder.build().isRevoked()) {
                LOGGER.warn("The Peppol Certificate is revoked [no caching]");
                return EPeppolCertificateCheckResult.REVOKED;
            }
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("The Peppol Certificate seems to be valid");
        }
        return EPeppolCertificateCheckResult.VALID;
    }

    @Nonnull
    public static EPeppolCertificateCheckResult checkPeppolAPCertificate(@Nullable X509Certificate x509Certificate, @Nullable OffsetDateTime offsetDateTime, @Nonnull ETriState eTriState, @Nullable ERevocationCheckMode eRevocationCheckMode) {
        boolean bl = eTriState.isUndefined() ? PeppolCertificateChecker.isCacheOCSPResults() : eTriState.isTrue();
        return PeppolCertificateChecker.checkCertificate(PEPPOL_AP_CA_ISSUERS, bl ? REVOCATION_CACHE_AP : null, ((PeppolRevocationCheckBuilder)((PeppolRevocationCheckBuilder)PeppolCertificateChecker.peppolRevocationCheck().certificate(x509Certificate)).checkDate(offsetDateTime)).validCAsPeppolAP().checkMode(eRevocationCheckMode));
    }

    @Nonnull
    public static EPeppolCertificateCheckResult checkPeppolSMPCertificate(@Nullable X509Certificate x509Certificate, @Nullable OffsetDateTime offsetDateTime, @Nonnull ETriState eTriState, @Nullable ERevocationCheckMode eRevocationCheckMode) {
        boolean bl = eTriState.isUndefined() ? PeppolCertificateChecker.isCacheOCSPResults() : eTriState.isTrue();
        return PeppolCertificateChecker.checkCertificate(PEPPOL_SMP_CA_ISSUERS, bl ? REVOCATION_CACHE_SMP : null, ((PeppolRevocationCheckBuilder)((PeppolRevocationCheckBuilder)PeppolCertificateChecker.peppolRevocationCheck().certificate(x509Certificate)).checkDate(offsetDateTime)).validCAsPeppolSMP().checkMode(eRevocationCheckMode));
    }

    static {
        PeppolCertificateChecker.addTrustedPeppolAPCACertificate(PeppolKeyStoreHelper.Config2018.CERTIFICATE_PILOT_AP);
        PeppolCertificateChecker.addTrustedPeppolAPCACertificate(PeppolKeyStoreHelper.Config2018.CERTIFICATE_PRODUCTION_AP);
        PeppolCertificateChecker.addTrustedPeppolSMPCACertificate(PeppolKeyStoreHelper.Config2018.CERTIFICATE_PILOT_SMP);
        PeppolCertificateChecker.addTrustedPeppolSMPCACertificate(PeppolKeyStoreHelper.Config2018.CERTIFICATE_PRODUCTION_SMP);
    }

    @NotThreadSafe
    public static class PeppolRevocationCheckBuilder
    extends CertificateRevocationChecker.AbstractRevocationCheckBuilder<PeppolRevocationCheckBuilder> {
        @Nonnull
        public PeppolRevocationCheckBuilder validCAsPeppolAP() {
            return (PeppolRevocationCheckBuilder)this.validCAs((Iterable<? extends X509Certificate>)PEPPOL_AP_CA_CERTS);
        }

        @Nonnull
        public PeppolRevocationCheckBuilder validCAsPeppolSMP() {
            return (PeppolRevocationCheckBuilder)this.validCAs((Iterable<? extends X509Certificate>)PEPPOL_SMP_CA_CERTS);
        }
    }

    @ThreadSafe
    public static final class PeppolRevocationCache {
        private final ExpiringMap<String, Boolean> m_aCache;
        private final IToBooleanFunction<X509Certificate> m_aValueProvider;

        public PeppolRevocationCache(@Nonnull IToBooleanFunction<X509Certificate> iToBooleanFunction) {
            ValueEnforcer.notNull(iToBooleanFunction, (String)"ValueProvider");
            this.m_aCache = ExpiringMap.builder().expirationPolicy(ExpirationPolicy.CREATED).expiration(6L, TimeUnit.HOURS).build();
            this.m_aValueProvider = iToBooleanFunction;
        }

        @Nonnull
        private static String _getKey(@Nonnull X509Certificate x509Certificate) {
            return x509Certificate.getSubjectX500Principal().getName() + "-" + x509Certificate.getSerialNumber().toString();
        }

        public boolean isRevoked(@Nonnull X509Certificate x509Certificate) {
            String string2 = PeppolRevocationCache._getKey(x509Certificate);
            return (Boolean)this.m_aCache.computeIfAbsent((Object)string2, string -> this.m_aValueProvider.applyAsBoolean((Object)x509Certificate));
        }

        public void clearCache() {
            this.m_aCache.clear();
        }
    }
}

