package com.linkare.booleval;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

import com.linkare.booleval.utils.WildcardToRegexUtility;

/**
 * 
 * @author Paulo Zenida
 * 
 *         Date: 2006-04-02
 */
public final class ExpressionParser {

    private static final String EMPTY_STRING = "";

    private static final String ONE_SPACE = " ";

    private static final String EQUAL_CHAR = "=";

    private static final String XOR_CHAR = "^";

    private static final String FALSE = "false";

    private static final String TRUE = "true";

    private static final String NOT_CHAR = "!";

    private static final String AND_CHAR = "&";

    private static final String OR_CHAR = "|";

    private static final String CLOSE_PARENTHESIS = ")";

    private static final String OPEN_PARENTHESIS = "(";

    private static final Set<Character> OPERATORS = new HashSet<Character>();

    static {
	OPERATORS.add(EQUAL_CHAR.charAt(0));
	OPERATORS.add(XOR_CHAR.charAt(0));
	OPERATORS.add(NOT_CHAR.charAt(0));
	OPERATORS.add(AND_CHAR.charAt(0));
	OPERATORS.add(OR_CHAR.charAt(0));
    }

    private ExpressionParser() {
    }

    /**
     * It returns a boolean expression from the comparision between the permission and the expression. For each value in the expression, it simply verifies if
     * the element corresponds to the permission name. For example: If the permission is "abc" and the expression is "(abc && b) || (b &&
     * c)", the returning string is "( true && b ) || ( b && c )".
     * 
     * @param value
     *            The value to compare with the expression
     * @param expression
     *            The expression to be compared with the value.
     * @return Returns the boolean expression resulting from the comparision between the expression and the value.
     */
    private static synchronized String convertToBooleanExpressionAux(final String value, final String expression) {
	String element = EMPTY_STRING;
	String operator = EMPTY_STRING;
	String result = EMPTY_STRING;
	String booleanResult = EMPTY_STRING;
	for (int i = 0; i != expression.length(); i++) {
	    char c = expression.charAt(i);
	    if (c == ' ') {
		if (operator.length() > 0) {
		    booleanResult += operator + ONE_SPACE;
		    operator = EMPTY_STRING;
		} else if (element.length() > 0) {
		    booleanResult = appendBooleanMatch(value, element, booleanResult);
		    element = EMPTY_STRING;
		}
		continue;
	    }
	    if (isOpenParenthesis(c) || isCloseParenthesis(c)) {
		if (isOpenParenthesis(c)) { // Potential operator
		    // Not operator
		    if (element.length() > 0 && operator.length() == 0) {
			element += c;
		    } else {
			result += c;
			booleanResult += c + ONE_SPACE;
			continue;
		    }
		} else if (isCloseParenthesis(c)) {
		    if (element.length() > 0 && element.contains(OPEN_PARENTHESIS) && operator.length() == 0) { // Part
			// of
			// the
			// signature
			result += c;
			element += c;
			booleanResult = appendBooleanMatch(value, element, booleanResult);
			element = EMPTY_STRING;
			operator = EMPTY_STRING;
			continue;
		    } else {
			if (element.length() > 0) {
			    booleanResult = appendBooleanMatch(value, element, booleanResult);
			}
			element = EMPTY_STRING;
			result += c;
			booleanResult += c + ONE_SPACE;
		    }
		}
	    } else {
		if (isOperator(c)) {
		    if (c == '!') {
			booleanResult += c + ONE_SPACE;
		    } else {
			if (operator.length() > 0) {
			    if (operator.charAt(0) == c) {
				result += c;
				booleanResult += operator + c + ONE_SPACE;
				operator = EMPTY_STRING;
			    }
			} else {
			    operator += c;
			    result += c;
			}
		    }
		} else {
		    element += c;
		    result += c;
		}
	    }
	}
	if (element.length() > 0) {
	    booleanResult = appendBooleanMatch(value, element, booleanResult);
	}
	return booleanResult;
    }

    /**
     * 
     * @param value
     * @param element
     * @param booleanResult
     * @return
     */
    private static String appendBooleanMatch(final String value, final String element, String booleanResult) {
	final StringBuilder result = new StringBuilder(booleanResult);
	if (areExpressionsEquivalent(value, element)) {
	    result.append(true).append(ONE_SPACE);
	} else {
	    result.append(element).append(ONE_SPACE);
	}
	return result.toString();
    }

    /**
     * 
     * @param elements
     * @param expression
     * @return
     */
    public static String convertToBooleanExpression(final Collection<String> elements, final String expression) {
	final StringBuilder builder = new StringBuilder(expression);
	for (final String element : elements) {
	    builder.replace(0, builder.length(), convertToBooleanExpressionAux(element, builder.toString()));
	}
	final String[] booleanElements = builder.toString().split(ONE_SPACE);
	builder.delete(0, builder.length());
	for (final String be : booleanElements) {
	    if (!be.equals(OPEN_PARENTHESIS) && !be.equals(CLOSE_PARENTHESIS) && containsNoOperator(be) && !be.equalsIgnoreCase(TRUE)) {
		builder.append(FALSE);
	    } else {
		builder.append(be);
	    }
	}
	return builder.toString();
    }

    private static boolean containsNoOperator(final String element) {
	return !element.contains(OR_CHAR) && !element.contains(AND_CHAR) && !element.contains(NOT_CHAR) && !element.contains(XOR_CHAR);
    }

    /**
     * 
     * @param element
     * @return
     */
    private static boolean isOperator(char element) {
	return OPERATORS.contains(element);
    }

    /**
     * 
     * @param element
     * @return
     */
    private static boolean isOpenParenthesis(char element) {
	return element == OPEN_PARENTHESIS.charAt(0);
    }

    /**
     * 
     * @param element
     * @return
     */
    private static boolean isCloseParenthesis(char element) {
	return element == CLOSE_PARENTHESIS.charAt(0);
    }

    /**
     * This method checks for equivalence in two expressions. It's prepared to check for expressions containing wildcards, such as *, + or ?.
     * 
     * @see java.util.regex.Pattern
     * @param expression1
     * @param expression2
     * @return Returns true if two expressions are equivalent. Returns false otherwise.
     */
    public static boolean areExpressionsEquivalent(String expression1, String expression2) {
	return Pattern.matches(WildcardToRegexUtility.wildcardToRegex(expression1), expression2)
		|| Pattern.matches(WildcardToRegexUtility.wildcardToRegex(expression2), expression1);
    }
}