package com.linkare.commons.communications.smtp;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.activation.FileDataSource;
import javax.mail.Address;
import javax.mail.BodyPart;
import javax.mail.Message;
import javax.mail.Message.RecipientType;
import javax.mail.MessagingException;
import javax.mail.SendFailedException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimeUtility;

import org.apache.commons.validator.EmailValidator;

import com.linkare.commons.utils.PropertiesManager;

/**
 * Used to send e-mails through an smtp protocol and using the javax.mail API.
 * 
 * This class was inspired on the homonymous class of the fenix-tools project.
 * 
 * @author Ricardo Espírito Santo - Linkare TI
 * 
 */
public class EmailSender {

    private static final String MAIL_CONTENT_TYPE = "mail.content.type";

    private static final String MAIL_SENDER_MAX_RECIPIENTS = "mailSender.max.recipients";

    private static final String SMTP_PROPERTIES = "/smtp.properties";

    private static final String CONTENT_TYPE;

    private static final int MAX_MAIL_RECIPIENTS;

    private static final Session session;

    private static Logger Log;

    static {
	try {

	    Log = Logger.getLogger("EmailSender");

	    final Properties properties = new Properties();
	    PropertiesManager.loadProperties(properties, SMTP_PROPERTIES);

	    Authenticator authenticator = new Authenticator(properties);

	    session = Session.getInstance(properties, authenticator);
	    MAX_MAIL_RECIPIENTS = Integer.parseInt(properties.getProperty(MAIL_SENDER_MAX_RECIPIENTS));
	    CONTENT_TYPE = properties.getProperty(MAIL_CONTENT_TYPE);

	} catch (IOException e) {
	    Log.severe("error.initializing.EmailSender: " + e.toString());
	    throw new RuntimeException(e);
	}
    }

    /**
     * Forwards the e-mail message to the bccAddresses
     * 
     * @param message
     * @param bccAddressesToforward
     * 
     * @return a list of skipped addresses
     */
    public static Collection<String> forward(final MimeMessage message, final List<String> bccAddressesToforward) {
	if (message == null) {
	    throw new NullPointerException("error.message.cannot.be.null");
	}

	final ArrayList<String> unsent = new ArrayList<String>(0);

	final List<String> bccAddressesList = new ArrayList<String>(new HashSet<String>(bccAddressesToforward));
	for (int i = 0; i < bccAddressesList.size(); i += MAX_MAIL_RECIPIENTS) {
	    final List<String> subList = bccAddressesList.subList(i, Math.min(bccAddressesList.size(), i + MAX_MAIL_RECIPIENTS));
	    try {
		MimeMessage newMessage = new MimeMessage(message);
		newMessage.setRecipients(javax.mail.internet.MimeMessage.RecipientType.TO, new InternetAddress[] {});
		newMessage.setRecipients(javax.mail.internet.MimeMessage.RecipientType.CC, new InternetAddress[] {});
		newMessage.setRecipients(javax.mail.internet.MimeMessage.RecipientType.BCC, new InternetAddress[] {});
		addRecipients(newMessage, Message.RecipientType.BCC, subList, unsent);
		Transport.send(newMessage);
	    } catch (SendFailedException e) {
		registerInvalidAddresses(unsent, e, null, null, subList);
	    } catch (MessagingException e) {
		if (subList != null) {
		    unsent.addAll(subList);
		}

		e.printStackTrace();
	    }
	}

	return unsent;
    }

    private static String encode(final String string) {
	try {
	    return MimeUtility.encodeText(string);
	} catch (final UnsupportedEncodingException e) {
	    e.printStackTrace();
	    return string;
	}
    }

    /**
     * A constructor whose content type is predefined with whatever value is set in <code>build.properties</code>
     * 
     * @param fromName
     * @param fromAddress
     * @param replyTos
     * @param toAddresses
     * @param ccAddresses
     * @param bccAddresses
     * @param subject
     * @param body
     * 
     * @return a <code>Collection</code> of addresses that never got reached
     */
    public static Collection<String> send(final String fromName, final String fromAddress, final String[] replyTos, final Collection<String> toAddresses,
	    final Collection<String> ccAddresses, final Collection<String> bccAddresses, final String subject, final String body) {
	return send(fromName, fromAddress, replyTos, toAddresses, ccAddresses, bccAddresses, subject, body, CONTENT_TYPE, null);
    }

    /**
     * A constructor whose content type is predefined with whatever value is set in <code>build.properties</code>
     * 
     * @param fromName
     * @param fromAddress
     * @param replyTos
     * @param toAddresses
     * @param ccAddresses
     * @param bccAddresses
     * @param subject
     * @param body
     * @param attachments
     * 
     * @return a <code>Collection</code> of addresses that never got reached
     */
    public static Collection<String> send(final String fromName, final String fromAddress, final String[] replyTos, final Collection<String> toAddresses,
	    final Collection<String> ccAddresses, final Collection<String> bccAddresses, final String subject, final String body, final List<String> attachments) {
	return send(fromName, fromAddress, replyTos, toAddresses, ccAddresses, bccAddresses, subject, body, CONTENT_TYPE, attachments);
    }

    /**
     * A simpler form of sending an e-mail specify only the from the to the subject and the body No multiple lists no from name and the content type is given by
     * the property file.
     * 
     * @param from
     * @param to
     * @param subject
     * @param body
     */
    public static void send(final String fromName, final String fromAddress, final String to, final String subject, final String body) {
	List<String> toAddresses = new ArrayList<String>();
	toAddresses.add(to);
	List<String> emptyAddresses = new ArrayList<String>();

	String[] replyTo = { fromAddress };

	send(fromName, fromAddress, replyTo, toAddresses, emptyAddresses, emptyAddresses, subject, body, CONTENT_TYPE, null);
    }

    /**
     * Adds an attachment to the email based on the fileAttachment @param
     * 
     * @param multipart
     * @param fileAttachment
     * @throws MessagingException
     */
    private static void addAttachment(MimeMultipart multipart, final List<String> attachmentsList) throws MessagingException {
	for (String fileAttachment : attachmentsList) {
	    MimeBodyPart messageBodyPart = new MimeBodyPart();
	    DataSource source = new FileDataSource(fileAttachment);
	    messageBodyPart.setDataHandler(new DataHandler(source));
	    int indexOf = fileAttachment.lastIndexOf("/") == -1 ? fileAttachment.lastIndexOf("\\") : fileAttachment.lastIndexOf("/");
	    String newFileName = indexOf > 0 ? fileAttachment.substring(indexOf + 1) : fileAttachment;
	    messageBodyPart.setFileName(newFileName);
	    multipart.addBodyPart(messageBodyPart);
	}
    }

    /**
     * @param fromName
     * @param fromAddress
     * @param replyTos
     * @param toAddresses
     * @param ccAddresses
     * @param bccAddresses
     * @param subject
     * @param body
     * @param contentType
     * @return
     */
    public static Collection<String> send(final String fromName, final String fromAddress, final String[] replyTos, final Collection<String> toAddresses,
	    final Collection<String> ccAddresses, final Collection<String> bccAddresses, final String subject, final String body, final String contentType) {
	return send(fromName, fromAddress, replyTos, toAddresses, ccAddresses, bccAddresses, subject, body, contentType, null);
    }

    /**
     * 
     * A constructor that specifies the content type of the message to be sent
     * 
     * @param fromName
     * @param fromAddress
     * @param replyTos
     * @param toAddresses
     * @param ccAddresses
     * @param bccAddresses
     * @param subject
     * @param body
     * @param contentType
     * @param attachmentsList
     * 
     * @return a <code>Collection</code> of addresses that never got reached
     */
    public static Collection<String> send(final String fromName, final String fromAddress, final String[] replyTos, final Collection<String> toAddresses,
	    final Collection<String> ccAddresses, final Collection<String> bccAddresses, final String subject, final String body, final String contentType,
	    final List<String> attachmentsList) {

	if (fromAddress == null) {
	    throw new NullPointerException("error.from.address.cannot.be.null");
	}

	final Collection<String> unsentAddresses = new ArrayList<String>(0);
	final Address[] replyToAddresses = new Address[replyTos == null ? 0 : replyTos.length];
	if (replyTos != null) {
	    for (int i = 0; i < replyTos.length; i++) {
		try {
		    replyToAddresses[i] = new InternetAddress(encode(replyTos[i]));
		} catch (AddressException e) {
		    throw new Error("invalid.reply.to.address: " + replyTos[i]);
		}
	    }
	}

	final String from = constructFromString(encode(fromName), fromAddress);
	final boolean hasToAddresses = (toAddresses != null && !toAddresses.isEmpty()) ? true : false;
	final boolean hasCCAddresses = (ccAddresses != null && !ccAddresses.isEmpty()) ? true : false;
	final boolean hasAttachments = (attachmentsList != null && !attachmentsList.isEmpty()) ? true : false;
	if (hasToAddresses || hasCCAddresses) {
	    try {
		final MimeMessage mimeMessageTo = new MimeMessage(session);
		mimeMessageTo.setFrom(new InternetAddress(from));
		mimeMessageTo.setSubject(encode(subject));
		mimeMessageTo.setReplyTo(replyToAddresses);

		if (hasToAddresses) {
		    addRecipients(mimeMessageTo, Message.RecipientType.TO, toAddresses, unsentAddresses);
		}

		if (hasCCAddresses) {
		    addRecipients(mimeMessageTo, Message.RecipientType.CC, ccAddresses, unsentAddresses);
		}

		final BodyPart bodyPart = new MimeBodyPart();
		bodyPart.setContent(body, contentType);

		final MimeMultipart mimeMultipart = new MimeMultipart();
		mimeMultipart.addBodyPart(bodyPart);

		if (hasAttachments) {
		    addAttachment(mimeMultipart, attachmentsList);
		}
		mimeMessageTo.setContent(mimeMultipart);

		//Apparently this next lines aren't needed it 
		//		try {
		//		    mimeMessageTo.setDataHandler(new DataHandler(new ByteArrayDataSource(body, contentType)));
		//		} catch (IOException e) {
		//		    e.printStackTrace();
		//		}

		Transport.send(mimeMessageTo);
	    } catch (SendFailedException e) {
		registerInvalidAddresses(unsentAddresses, e, toAddresses, ccAddresses, null);
	    } catch (MessagingException e) {
		if (toAddresses != null) {
		    unsentAddresses.addAll(toAddresses);
		}

		if (ccAddresses != null) {
		    unsentAddresses.addAll(ccAddresses);
		}

		e.printStackTrace();
	    }
	}

	if (bccAddresses != null && !bccAddresses.isEmpty()) {
	    final List<String> bccAddressesList = new ArrayList<String>(new HashSet<String>(bccAddresses));
	    for (int i = 0; i < bccAddressesList.size(); i += MAX_MAIL_RECIPIENTS) {
		List<String> subList = null;
		try {
		    subList = bccAddressesList.subList(i, Math.min(bccAddressesList.size(), i + MAX_MAIL_RECIPIENTS));
		    final Message message = new MimeMessage(session);
		    message.setFrom(new InternetAddress(from));
		    message.setSubject(encode(subject));
		    message.setReplyTo(replyToAddresses);

		    addRecipients(message, Message.RecipientType.BCC, subList, unsentAddresses);

		    final BodyPart bodyPart = new MimeBodyPart();
		    bodyPart.setContent(body, contentType);

		    final MimeMultipart mimeMultipart = new MimeMultipart();
		    mimeMultipart.addBodyPart(bodyPart);

		    //Apparently this next lines aren't needed it	
		    //		    try {
		    //			message.setDataHandler(new DataHandler(new ByteArrayDataSource(body, contentType)));
		    //		    } catch (IOException e) {
		    //			e.printStackTrace();
		    //		    }

		    if (hasAttachments) {
			addAttachment(mimeMultipart, attachmentsList);
		    }

		    message.setContent(mimeMultipart);

		    Transport.send(message);

		} catch (SendFailedException e) {
		    registerInvalidAddresses(unsentAddresses, e, null, null, subList);
		} catch (MessagingException e) {
		    if (subList != null) {
			unsentAddresses.addAll(subList);
		    }

		    e.printStackTrace();
		}
	    }
	}

	return unsentAddresses;
    }

    /**
     * Register invalid e-mail addresses. These addresses have been attempted to reach at least once and failed
     * 
     * @param unsentAddresses
     * @param e
     *            the exception
     * @param toAddresses
     * @param ccAddresses
     * @param bccAddresses
     */
    private static void registerInvalidAddresses(final Collection<String> unsentAddresses, final SendFailedException e, final Collection<String> toAddresses,
	    final Collection<String> ccAddresses, final Collection<String> bccAddresses) {
	e.printStackTrace();
	if (e.getValidUnsentAddresses() != null) {
	    for (int i = 0; i < e.getValidUnsentAddresses().length; i++) {
		unsentAddresses.add(e.getValidUnsentAddresses()[i].toString());
	    }
	} else {
	    if (e.getValidSentAddresses() == null || e.getValidSentAddresses().length == 0) {
		if (toAddresses != null) {
		    unsentAddresses.addAll(toAddresses);
		}
		if (ccAddresses != null) {
		    unsentAddresses.addAll(ccAddresses);
		}
		if (bccAddresses != null) {
		    unsentAddresses.addAll(bccAddresses);
		}
	    }
	}
    }

    /**
     * Builds the from address to display on the e-mail that gets sent if we have a from name we display it else we only display the addess. So it either
     * becomes: "John Doe " <john@doe.com> OR john@doe.com if no name is provided
     * 
     * @param fromName
     *            The name on who's behalf is this message sent
     * @param fromAddress
     *            The e-mail address from which this message is sent
     * 
     * @return the "from" field full composed text
     */
    protected static String constructFromString(final String fromName, String fromAddress) {
	return (fromName == null || fromName.length() == 0) ? fromAddress : "\"" + fromName + "\" <" + fromAddress + ">";
    }

    /**
     * Adds the e-mail addresses to a list of receipients
     * 
     * @param message
     * @param recipientType
     * @param emailAddresses
     * @param unsentMails
     * 
     * @throws MessagingException
     */
    private static void addRecipients(final Message message, final RecipientType recipientType, final Collection<String> emailAddresses,
	    Collection<String> unsentMails) throws MessagingException {
	if (emailAddresses != null) {
	    for (final String emailAddress : emailAddresses) {
		try {
		    final EmailValidator validator = EmailValidator.getInstance();
		    if (validator.isValid(emailAddress)) {
			Log.fine("Sending to: " + emailAddress);
			message.addRecipient(recipientType, new InternetAddress(encode(emailAddress)));
		    } else {
			Log.fine("skipped: " + emailAddress);
			unsentMails.add(emailAddress);
		    }
		} catch (AddressException e) {
		    Log.fine("skipped due to address exception: " + emailAddress);
		    unsentMails.add(emailAddress);
		}
	    }
	}
    }
}
