package com.linkare.commons.dao;

import java.io.Serializable;
import java.util.List;

/**
 * This interface presents the common set of operations that any DAO should provide by default. Possible implementations of this interface are DAOs for JPA
 * access or JDBC.
 * 
 * @author Paulo Zenida - Linkare TI
 * 
 * @param <Entity>
 *            The entity type to which this generic dao presents an interface.
 * @param <ID>
 *            The identifier type of the <code>Entity</code> type.
 */
public interface IGenericDAO<Entity extends Identifiable<ID> & Deletable, ID extends Serializable> {

    /**
     * Provides the create/persist operation for the <code>entity<code> passed in.
     * 
     * @param entity
     *            the entity to create.
     */
    public void create(Entity entity);

    /**
     * Provides the edit/merge operation for the <code>entity<code> passed in.
     * 
     * @param entity
     *            the entity to edit.
     * @return Returns the entity after being merged/updated.
     */
    public Entity edit(Entity entity);

    /**
     * Provides the remove operation for the <code>entity</code> passed in.
     * 
     * @param entity
     *            the entity to remove.
     */
    public void remove(Entity entity);

    /**
     * Provides the find operation, based on the entity's <code>id</code>.
     * 
     * @param id
     *            the id to be found.
     * @return Returns the entity if found. It returns null otherwise.
     */
    public Entity find(ID id);

    /**
     * Provides the find operation, allowing pagination of results.
     * 
     * @param all
     *            flag to control if the method should look for all entities.
     * @param firstResult
     *            the number where the result query should start.
     * @param maxResults
     *            the maximum number of entities after <code>firstResult</code> that this method should return.
     * @return Returns the list of entities found.
     */
    public List<Entity> find(final boolean all, final int firstResult, final int maxResults);

    /**
     * Provides the find operation, according to a range of results.
     * 
     * @param range
     *            the range to look for in the resulting list.
     * @return Returns the list of entities found in the determined <code>range</code>.
     */
    public List<Entity> findRange(final int[] range);

    /**
     * Provides the find all operation.
     * 
     * @return Returns the full list of entities found.
     */
    public List<Entity> findAll();

    /**
     * Provides the count operation for the list of entities found.
     * 
     * @return Returns the number of entities found.
     */
    public int count();
}