package com.linkare.commons.utils;

import java.io.Serializable;

/**
 * Utility class to store a boolean result and a message to explain the reason why the result is either true, false or undetermined (null). The message may be
 * used, for instance, as a resource bundle key in which clients use it to supply details about any operations performed.
 * 
 * @author Paulo Zenida - Linkare TI
 * 
 */
public final class BooleanResult implements Serializable {

    private static final long serialVersionUID = 1475830919034616675L;

    private static final int HASH_CODE_MULTIPLIER = 29;

    private static final int HASH_CODE_INIT = 31;

    /**
     * The result of the boolean result type is immutable. Once created, it should not be possible to change it. This means a true result, for instance, will
     * always be a true result.
     */
    private final Boolean result;

    /**
     * The details message associated to this boolean result type. It should not be final to enable the possibility for this type's clients to add a details
     * message associated to the result of the evaluation in which this type was used.
     */
    private String message;

    /**
     * Utility constant to represent the true result without any detail message
     */
    public static final BooleanResult TRUE_RESULT = new BooleanResult(Boolean.TRUE);

    /**
     * Utility constant to represent the false result without any detail message
     */
    public static final BooleanResult FALSE_RESULT = new BooleanResult(Boolean.FALSE);

    /**
     * Utility constant to represent the undetermined result without any detail message
     */
    public static final BooleanResult UNDETERMINED_RESULT = new BooleanResult(null);

    /**
     * This constructor is private to "force" the client code from this class to use the constants.
     * 
     * @param result
     *            the result to set.
     */
    private BooleanResult(Boolean result) {
	this(result, null);
    }

    /**
     * 
     * @param result
     *            the result to set.
     * @param message
     *            the message to set.
     */
    public BooleanResult(Boolean result, String message) {
	super();
	this.result = result;
	this.message = message;
    }

    /**
     * @return the result
     */
    public Boolean getResult() {
	return result;
    }

    /**
     * @return the message
     */
    public String getMessage() {
	return message;
    }

    /**
     * @param message
     *            the message to set
     */
    public BooleanResult setMessage(String message) {
	this.message = message;
	return this;
    }

    @Override
    public boolean equals(final Object other) {
	if (!(other instanceof BooleanResult)) {
	    return false;
	}
	return equalsTo((BooleanResult) other);
    }

    @Override
    public int hashCode() {
	int hashCode = HASH_CODE_INIT;
	hashCode = HASH_CODE_MULTIPLIER * hashCode + (getResult() != null ? getResult().hashCode() : 0);
	hashCode = HASH_CODE_MULTIPLIER * hashCode + (getMessage() != null ? getMessage().hashCode() : 0);
	return hashCode;
    }

    private boolean equalsTo(final BooleanResult other) {
	return EqualityUtils.<Boolean> equals(getResult(), other.getResult()) && EqualityUtils.<String> equals(getMessage(), other.getMessage());
    }

    @Override
    public String toString() {
	return getResult() + (getMessage() == null ? "" : " " + getMessage());
    }
}