package com.linkare.commons.utils;

import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * Utility class that provides utility methods to return a localized string for a given enum type constant in the specified resource bundle.
 * 
 * @author Paulo Zenida - Linkare TI
 * 
 */
public final class EnumLocalizeNameHelper<EnumType extends Enum<?>> {

    private static final String MISSING_RESOURCE_DELIMITER = "???";

    /**
     * Hiding constructor so that this utility class cannot be instantiated from outside.
     */
    private EnumLocalizeNameHelper() {
    }

    /**
     * 
     * @param <EnumType>
     *            the type of the enum whose constant we are looking for.
     * @param enumType
     *            the enum constant we are looking for.
     * @param bundle
     *            the bundle where the <code>enumType</code> should be looked in.
     * @return Returns the localized name of the <code>enumType</code> in the <code>bundle</code> specified, if both the <code>enumType</code> and the
     *         <code>bundle</code> are not null. The bundle key format is <code>'<enumType.getClass.simpleName()>.<enumType.name()>'</code>. It returns false
     *         otherwise. E.g., for one enum type named Gender, containing MALE and FEMALE as its members:
     *         <table>
     *         <tr>
     *         <th>Enum type</th>
     *         <th>Searching key</th>
     *         </tr>
     *         <tr>
     *         <td>MALE</td>
     *         <td>Gender.MALE</td>
     *         </tr>
     *         <tr>
     *         <td>FEMALE</td>
     *         <td>Gender.FEMALE</td>
     *         </tr>
     *         </table>
     */
    public static <EnumType extends Enum<?>> String getLocalizedName(final EnumType enumType, final ResourceBundle bundle) {
	final String key = enumType.getDeclaringClass().getSimpleName() + "." + enumType.name();
	try {
	    return (enumType == null || bundle == null) ? null : bundle.getString(key);
	} catch (final MissingResourceException e) {
	    return MISSING_RESOURCE_DELIMITER + key + MISSING_RESOURCE_DELIMITER;
	}
    }
}