package com.linkare.commons.utils;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

/**
 * Utility class to help loading and looking for any property in a Java Properties map.
 * 
 * @author Paulo Zenida - Linkare TI
 * @since 0.0.1
 */
public final class PropertiesManager {

    private static final String DEFAULT_BUILD_PROPERTIES = "/build.properties";

    private static final PropertiesManager INSTANCE = new PropertiesManager();

    private static final Properties PROPERTIES = new Properties();

    /**
     * On class loading, try to load a PROPERTIES file in the classpath named build.properties
     */
    static {
	try {
	    loadProperties(PROPERTIES, DEFAULT_BUILD_PROPERTIES);
	} catch (IOException e) {
	    // ignore on purpose
	}
    }

    /**
     * Hiding constructor so that this utility class cannot be instantiated from outside.
     */
    private PropertiesManager() {
    }

    /**
     * It loads the properties in a file named <code>fileName</code> into the <code>properties</code> map.
     * 
     * @param properties
     *            the properties file to which the the properties in the file <code>fileName</code> should be loaded to.
     * @param fileName
     *            the name of the file from which there are properties to be loaded
     * @throws IOException
     *             if the input stream for the file <code>fileName</code> is null.
     */
    public static void loadProperties(final Properties properties, final String fileName) throws IOException {
	InputStream inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(fileName);
	inputStream = inputStream != null ? inputStream : INSTANCE.getClass().getResourceAsStream(fileName);
	if (inputStream != null) {
	    properties.load(inputStream);
	}
    }

    /**
     * It loads the properties in a file named <code>fileName</code> into the internal <code>PROPERTIES</code> map.
     * 
     * @param fileName
     *            the name of the file from which there are properties to be loaded
     * @throws IOException
     *             if the input stream for the file <code>fileName</code> is null.
     */
    public static void loadProperties(final String fileName) throws IOException {
	final InputStream inputStream = INSTANCE.getClass().getResourceAsStream(fileName);
	if (inputStream != null) {
	    PROPERTIES.load(inputStream);
	}
    }

    /**
     * 
     * @param key
     *            the key to be looked into the <code>PROPERTIES</code> map of this class.
     * @return the value for the <code>key</code> provided, if found. Returns null otherwise.
     */
    public static String getProperty(final String key) {
	return PROPERTIES.getProperty(key);
    }

    /**
     * 
     * @param key
     *            the key to be looked into the <code>PROPERTIES</code> map of this class.
     * @return the boolean value for the <code>key</code> provided, if found. Returns false when not found or the value is not a boolean type.
     */
    public static boolean getBooleanProperty(final String key) {
	return Boolean.parseBoolean(PROPERTIES.getProperty(key));
    }

    /**
     * 
     * @param key
     *            the key to be looked into the <code>PROPERTIES</code> map of this class.
     * @return the int value for the <code>key</code> provided, if found.
     * @throws NumberFormatException
     *             if the found value is not a parseable number.
     */
    public static Integer getIntegerProperty(final String key) {
	return Integer.valueOf(PROPERTIES.getProperty(key));
    }

    /**
     * 
     * @param key
     *            the key to be set into the <code>PROPERTIES</code> map of this class.
     * @param value
     *            the value to be set into the <code>PROPERTIES</code> map of this class, associated to the <code>key</code> specified.
     */
    public static void setProperty(final String key, final String value) {
	PROPERTIES.setProperty(key, value);
    }
}