package com.linkare.commons.utils;

import java.text.Normalizer;
import java.text.Normalizer.Form;

/**
 * Utility class providing some methods to normalize strings.
 * 
 * @author Paulo Zenida - Linkare TI
 * 
 * @since 0.0.1
 */
public final class StringNormalizer {

    /**
     * Hiding constructor so that this utility class cannot be instantiated from outside.
     */
    private StringNormalizer() {
    }

    /**
     * 
     * @param word
     *            the word to be normalized.
     * @param toLowerCase
     *            the flag to control if the <code>string</code> should be converted to lower case (when true) or if the casing should be preserved (when
     *            false).
     * @return Returns the <code>word</code> normalized in ASCII code.
     */
    private static String normalize(String word, boolean toLowerCase) {
	final String normalized = Normalizer.normalize(word, Form.NFD).replaceAll("[^\\p{ASCII}]", "");
	return (toLowerCase) ? normalized.toLowerCase() : normalized;
    }

    /**
     * 
     * @param string
     *            the string to be normalized.
     * @return Returns the <code>string</code> in a normalized format, in all lower case.
     * @see StringNormalizer#normalize(String, boolean)
     */
    public static String normalize(String string) {
	return normalize(string, true);
    }

    /**
     * @param word
     *            the word to be normalized.
     * @return Returns the <code>string</code> in a normalized format, preserving the capital letters.
     * @see StringNormalizer#normalize(String, boolean)
     */
    public static String normalizePreservingCapitalizedLetters(String word) {
	return normalize(word, false);
    }

    /**
     * It normalizes an array of words.
     * 
     * @param words
     *            the list of words to be normalized.
     * @see StringNormalizer#normalize(String)
     */
    public static void normalize(String[] words) {
	for (int i = 0; i < words.length; i++) {
	    words[i] = normalize(words[i]);
	}
    }

    /**
     * 
     * @param word
     *            the word to normalize.
     * @return Returns the <code>word</code> in a normalized format, without any lower case char.
     */
    public static String normalizeAndRemoveMinorChars(String word) {
	final String normalizedString = normalize(word);
	final StringBuilder stringBuilder = new StringBuilder();
	for (final char c : normalizedString.toCharArray()) {
	    if (!Character.isLowerCase(c)) {
		stringBuilder.append(c);
	    }
	}
	return stringBuilder.toString();
    }

    /**
     * 
     * @param string
     *            the string to be computed.
     * @return Returns the computational name of the passed in <code>string</code>. A computational name is the string without any empty spaces nor bars ('/'),
     *         which are both replaced with '-', and with no special characters like accents. E.g.,
     * 
     *         <table border="1">
     *         <tr>
     *         <th>string</th>
     *         <th>result</th>
     *         </tr>
     *         <tr>
     *         <td>This is just one example</td>
     *         <td>this-is-just-one-example</td>
     *         </tr>
     *         <tr>
     *         <td>Centro de comunicação</td>
     *         <td>centro-de-comunicacao</td>
     *         </tr>
     *         </table>
     * 
     * @see StringNormalizer#createComputationalName(String, char)
     */
    public static String createComputationalName(final String string) {
	return createComputationalName(string, '-');
    }

    /**
     * 
     * @param word
     * @return Returns the computational name of the passed in <code>word</code>. A computational name is the string without any empty spaces nor bars ('/'),
     *         which are both replaced with '-', and with no special characters like accents. E.g.,
     * 
     *         <table border="1">
     *         <tr>
     *         <th>string</th>
     *         <th>result</th>
     *         </tr>
     *         <tr>
     *         <td>This is just one example</td>
     *         <td>this'separator'is'separator'just'separator'an'separator'example</td>
     *         </tr>
     *         </table>
     * 
     * @see StringNormalizer#normalizeAndRemoveMinorChars(String)
     */
    public static String createComputationalName(final String word, final char separator) {
	if (StringUtils.isBlank(word)) {
	    return word;
	}
	return normalizeAndRemoveMinorChars(word).replace(' ', separator).replace('/', separator);
    }
}