package com.linkare.commons.utils;

/**
 * Utility class that groups commons operations related to strings.
 * 
 * @author Paulo Zenida - Linkare TI
 * 
 * @since 0.0.1
 */
public final class StringUtils {

    /**
     * The default line separator string
     */
    public static final String LINE_SEPARATOR = "\n";

    /**
     * The system property representing the line separator
     */
    public static final String SYSTEM_PROP_LINE_SEPARATOR = "line.separator";

    /**
     * Hiding constructor so that this utility class cannot be instantiated from outside.
     */
    private StringUtils() {
    }

    /**
     * Provides one utility method to check if a <code>word</code> is blank. One <code>word</code> is said to be blank when it's null or, after being trimmed,
     * it has a length of 0 (zero).
     * 
     * @param word
     *            the word to be checked
     * 
     * @return Returns true if the word is blank. Returns false otherwise.
     */
    public static boolean isBlank(final String word) {
	return word == null || word.trim().length() == 0;
    }

    /**
     * Provides one utility method to check if a <code>word</code> is not blank. One <code>word</code> is said to be not blank if it's not
     * <code>isBlank(word)</code>.
     * 
     * @param word
     *            the word to be checked
     * 
     * @return Returns true if the word is not blank. Returns false otherwise.
     * @see StringUtils#isBlank(String)
     */
    public static boolean isNotBlank(final String word) {
	return !isBlank(word);
    }

    /**
     * 
     * @return the new line string, according to the <code>SYSTEM_PROP_LINE_SEPARATOR</code> system property, if present. It returns the
     *         <code>LINE_SEPARATOR</code> string otherwise.
     */
    public static String newline() {
	String newline = System.getProperty(SYSTEM_PROP_LINE_SEPARATOR);
	if (newline == null) {
	    newline = LINE_SEPARATOR;
	}
	return newline;
    }

    /**
     * 
     * @param firstWord
     *            the first word to be the output of this method.
     * @param secondWord
     *            the second word to be appended to be appended.
     * @param otherWords
     *            an optional list of words to be appended to <code>firstWord</code> and <code>secondWord</code>.
     * @return the result of appending <code>firstWord</code> with <code>secondWord</code> and with <code>otherWords</code>.
     */
    public static final String append(final String firstWord, final String secondWord, final String... otherWords) {
	final StringBuilder builder = new StringBuilder();
	builder.append(firstWord);
	builder.append(secondWord);
	for (String s : otherWords) {
	    builder.append(s);
	}
	return builder.toString();
    }
}