/**
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 */
package com.microsoft.azure.management.network.implementation;

import com.microsoft.azure.CloudException;
import com.microsoft.azure.SubResource;
import com.microsoft.azure.management.apigeneration.LangDefinition;
import com.microsoft.azure.management.network.AddressSpace;
import com.microsoft.azure.management.network.DdosProtectionPlan;
import com.microsoft.azure.management.network.DhcpOptions;
import com.microsoft.azure.management.network.Network;
import com.microsoft.azure.management.network.NetworkPeerings;
import com.microsoft.azure.management.network.Subnet;
import com.microsoft.azure.management.network.model.GroupableParentResourceWithTagsImpl;

import com.microsoft.azure.management.resources.fluentcore.model.Creatable;
import com.microsoft.azure.management.resources.fluentcore.utils.SdkContext;
import com.microsoft.azure.management.resources.fluentcore.utils.Utils;
import rx.Observable;
import rx.functions.Func1;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

/**
 * Implementation for Network and its create and update interfaces.
 */
@LangDefinition
class NetworkImpl
    extends GroupableParentResourceWithTagsImpl<
        Network,
        VirtualNetworkInner,
        NetworkImpl,
        NetworkManager>
    implements
        Network,
        Network.Definition,
        Network.Update {

    private Map<String, Subnet> subnets;
    private NetworkPeeringsImpl peerings;
    private Creatable<DdosProtectionPlan> ddosProtectionPlanCreatable;

    NetworkImpl(String name,
            final VirtualNetworkInner innerModel,
            final NetworkManager networkManager) {
        super(name, innerModel, networkManager);
    }

    @Override
    protected void initializeChildrenFromInner() {
        // Initialize subnets
        this.subnets = new TreeMap<>();
        List<SubnetInner> inners = this.inner().subnets();
        if (inners != null) {
            for (SubnetInner inner : inners) {
                SubnetImpl subnet = new SubnetImpl(inner, this);
                this.subnets.put(inner.name(), subnet);
            }
        }

        this.peerings = new NetworkPeeringsImpl(this);
    }

    // Verbs

    @Override
    public Observable<Network> refreshAsync() {
        return super.refreshAsync().map(new Func1<Network, Network>() {
            @Override
            public Network call(Network network) {
                NetworkImpl impl = (NetworkImpl) network;
                impl.initializeChildrenFromInner();
                return impl;
            }
        });
    }

    @Override
    protected Observable<VirtualNetworkInner> getInnerAsync() {
        return this.manager().inner().virtualNetworks().getByResourceGroupAsync(this.resourceGroupName(), this.name());
    }

    @Override
    protected Observable<VirtualNetworkInner> applyTagsToInnerAsync() {
        return this.manager().inner().virtualNetworks().updateTagsAsync(resourceGroupName(), name(), inner().getTags());
    }

    @Override
    public boolean isPrivateIPAddressAvailable(String ipAddress) {
        IPAddressAvailabilityResultInner result = checkIPAvailability(ipAddress);
        return (result != null) ? result.available() : false;
    }

    @Override
    public boolean isPrivateIPAddressInNetwork(String ipAddress) {
        IPAddressAvailabilityResultInner result = checkIPAvailability(ipAddress);
        return (result != null) ? true : false;
    }

    // Helpers

    private IPAddressAvailabilityResultInner checkIPAvailability(String ipAddress) {
        if (ipAddress == null) {
            return null;
        }
        IPAddressAvailabilityResultInner result = null;
        try {
            result = this.manager().networks().inner().checkIPAddressAvailability(
                    this.resourceGroupName(),
                    this.name(),
                    ipAddress);
        } catch (CloudException e) {
            if (!e.body().code().equalsIgnoreCase("PrivateIPAddressNotInAnySubnet")) {
                throw e; // Rethrow if the exception reason is anything other than IP address not found
            }
        }
        return result;
    }

    NetworkImpl withSubnet(SubnetImpl subnet) {
        this.subnets.put(subnet.name(), subnet);
        return this;
    }

    // Setters (fluent)

    @Override
    public NetworkImpl withDnsServer(String ipAddress) {
        if (this.inner().dhcpOptions() == null) {
            this.inner().withDhcpOptions(new DhcpOptions());
        }

        if (this.inner().dhcpOptions().dnsServers() == null) {
            this.inner().dhcpOptions().withDnsServers(new ArrayList<String>());
        }

        this.inner().dhcpOptions().dnsServers().add(ipAddress);
        return this;
    }

    @Override
    public NetworkImpl withSubnet(String name, String cidr) {
        return this.defineSubnet(name)
            .withAddressPrefix(cidr)
            .attach();
    }

    @Override
    public NetworkImpl withSubnets(Map<String, String> nameCidrPairs) {
        this.subnets.clear();
        for (Entry<String, String> pair : nameCidrPairs.entrySet()) {
            this.withSubnet(pair.getKey(), pair.getValue());
        }
        return this;
    }

    @Override
    public NetworkImpl withoutSubnet(String name) {
        this.subnets.remove(name);
        return this;
    }

    @Override
    public NetworkImpl withAddressSpace(String cidr) {
        if (this.inner().addressSpace() == null) {
            this.inner().withAddressSpace(new AddressSpace());
        }

        if (this.inner().addressSpace().addressPrefixes() == null) {
            this.inner().addressSpace().withAddressPrefixes(new ArrayList<String>());
        }

        this.inner().addressSpace().addressPrefixes().add(cidr);
        return this;
    }

    @Override
    public SubnetImpl defineSubnet(String name) {
        SubnetInner inner = new SubnetInner()
                .withName(name);
        return new SubnetImpl(inner, this);
    }

    @Override
    public NetworkImpl withoutAddressSpace(String cidr) {
        if (cidr == null) {
            // Skip
        } else if (this.inner().addressSpace() == null) {
            // Skip
        } else if (this.inner().addressSpace().addressPrefixes() == null) {
            // Skip
        } else {
            this.inner().addressSpace().addressPrefixes().remove(cidr);
        }
        return this;
    }

    // Getters

    @Override
    public List<String> addressSpaces() {
        List<String> addressSpaces = new ArrayList<String>();
        if (this.inner().addressSpace() == null) {
            return Collections.unmodifiableList(addressSpaces);
        } else if (this.inner().addressSpace().addressPrefixes() == null) {
            return Collections.unmodifiableList(addressSpaces);
        } else {
            return Collections.unmodifiableList(this.inner().addressSpace().addressPrefixes());
        }
    }

    @Override
    public List<String> dnsServerIPs() {
        List<String> ips = new ArrayList<String>();
        if (this.inner().dhcpOptions() == null) {
            return Collections.unmodifiableList(ips);
        } else if (this.inner().dhcpOptions().dnsServers() == null) {
            return Collections.unmodifiableList(ips);
        } else {
            return this.inner().dhcpOptions().dnsServers();
        }
    }

    @Override
    public Map<String, Subnet> subnets() {
        return Collections.unmodifiableMap(this.subnets);
    }

    @Override
    protected void beforeCreating() {
        // Ensure address spaces
        if (this.addressSpaces().size() == 0) {
            this.withAddressSpace("10.0.0.0/16");
        }

        if (isInCreateMode()) {
            // Create a subnet as needed, covering the entire first address space
            if (this.subnets.size() == 0) {
                this.withSubnet("subnet1", this.addressSpaces().get(0));
            }
        }

        // Reset and update subnets
        this.inner().withSubnets(innersFromWrappers(this.subnets.values()));
    }

    @Override
    protected void afterCreating() {
        initializeChildrenFromInner();
    }

    @Override
    public SubnetImpl updateSubnet(String name) {
        return (SubnetImpl) this.subnets.get(name);
    }

    @Override
    protected Observable<VirtualNetworkInner> createInner() {
        if (ddosProtectionPlanCreatable != null && this.taskResult(ddosProtectionPlanCreatable.key()) != null) {
            DdosProtectionPlan ddosProtectionPlan = this.<DdosProtectionPlan>taskResult(ddosProtectionPlanCreatable.key());
            withExistingDdosProtectionPlan(ddosProtectionPlan.id());
        }
        return this.manager().inner().virtualNetworks().createOrUpdateAsync(this.resourceGroupName(), this.name(), this.inner())
                .map(new Func1<VirtualNetworkInner, VirtualNetworkInner>() {
                    @Override
                    public VirtualNetworkInner call(VirtualNetworkInner virtualNetworkInner) {
                        NetworkImpl.this.ddosProtectionPlanCreatable = null;
                        return virtualNetworkInner;
                    }
                });
    }

    @Override
    public NetworkPeerings peerings() {
        return this.peerings;
    }

    @Override
    public boolean isDdosProtectionEnabled() {
        return Utils.toPrimitiveBoolean(inner().enableDdosProtection());
    }

    @Override
    public boolean isVmProtectionEnabled() {
        return Utils.toPrimitiveBoolean(inner().enableVmProtection());
    }

    @Override
    public String ddosProtectionPlanId() {
        return inner().ddosProtectionPlan() == null ? null : inner().ddosProtectionPlan().id();
    }

    @Override
    public NetworkImpl withNewDdosProtectionPlan() {
        inner().withEnableDdosProtection(true);
        DdosProtectionPlan.DefinitionStages.WithGroup ddosProtectionPlanWithGroup = manager().ddosProtectionPlans()
                .define(SdkContext.randomResourceName(name(), 20))
                .withRegion(region());
        if (super.creatableGroup != null && isInCreateMode()) {
            ddosProtectionPlanCreatable = ddosProtectionPlanWithGroup.withNewResourceGroup(super.creatableGroup);
        } else {
            ddosProtectionPlanCreatable = ddosProtectionPlanWithGroup.withExistingResourceGroup(resourceGroupName());
        }
        this.addDependency(ddosProtectionPlanCreatable);
        return this;
    }

    @Override
    public NetworkImpl withExistingDdosProtectionPlan(String planId) {
        inner().withEnableDdosProtection(true).withDdosProtectionPlan(new SubResource().withId(planId));
        return this;
    }

    @Override
    public NetworkImpl withoutDdosProtectionPlan() {
        inner().withEnableDdosProtection(false).withDdosProtectionPlan(null);
        return this;
    }

    @Override
    public NetworkImpl withVmProtection() {
        inner().withEnableVmProtection(true);
        return this;
    }

    @Override
    public NetworkImpl withoutVmProtection() {
        inner().withEnableVmProtection(false);
        return this;
    }
}
