/**************************************************************************
 * copyright file="PhysicalAddressEntry.java" company="Microsoft"
 *     Copyright (c) Microsoft Corporation.  All rights reserved.
 * 
 * Defines the PhysicalAddressEntry.java.
 **************************************************************************/
package microsoft.exchange.webservices.data;

import java.util.ArrayList;
import java.util.List;

import javax.xml.stream.XMLStreamException;

/**
 * Represents an entry of an PhysicalAddressDictionary.
 */
public final class PhysicalAddressEntry extends
		DictionaryEntryProperty<PhysicalAddressKey> implements
		IPropertyBagChangedDelegate {

	/** The property bag. */
	private SimplePropertyBag<String> propertyBag;

	/**
	 * Initializes a new instance of PhysicalAddressEntry.
	 */
	public PhysicalAddressEntry() {
		super(PhysicalAddressKey.class);
		this.propertyBag = new SimplePropertyBag<String>();
		this.propertyBag.addOnChangeEvent(this);
	}

	/**
	 * * Property was changed.
	 * 
	 * @param simplePropertyBag
	 *            the simple property bag
	 */
	public void propertyBagChanged(SimplePropertyBag simplePropertyBag) {
		this.changed();
	}

	/**
	 * Gets the street.
	 * 
	 * @return the street
	 * @throws Exception
	 *             the exception
	 */
	public String getStreet() throws Exception {
		return (String)this.propertyBag
				.getSimplePropertyBag(PhysicalAddressSchema.Street);
	}

	/**
	 * Sets the street.
	 * 
	 * @param value
	 *            the new street
	 * @throws Exception
	 *             the exception
	 */
	public void setStreet(String value) throws Exception {
		this.propertyBag.setSimplePropertyBag(PhysicalAddressSchema.Street,
				value);

	}

	/**
	 * Gets the city.
	 * 
	 * @return the city
	 * @throws Exception
	 *             the exception
	 */
	public String getCity() throws Exception {
		return (String)this.propertyBag
				.getSimplePropertyBag(PhysicalAddressSchema.City);
	}

	/**
	 * Sets the city.
	 * 
	 * @param value
	 *            the new city
	 */
	public void setCity(String value) {
		this.propertyBag
				.setSimplePropertyBag(PhysicalAddressSchema.City, value);
	}

	/**
	 * Gets the state.
	 * 
	 * @return the state
	 * @throws Exception
	 *             the exception
	 */
	public String getState() throws Exception {
		return (String)this.propertyBag
				.getSimplePropertyBag(PhysicalAddressSchema.State);
	}

	/**
	 * Sets the state.
	 * 
	 * @param value
	 *            the new state
	 */
	public void setState(String value) {
		this.propertyBag.setSimplePropertyBag(PhysicalAddressSchema.State,
				value);
	}

	/**
	 * Gets the country or region.
	 * 
	 * @return the country or region
	 * @throws Exception
	 *             the exception
	 */
	public String getCountryOrRegion() throws Exception {
		return (String)this.propertyBag
				.getSimplePropertyBag(PhysicalAddressSchema.CountryOrRegion);
	}

	/**
	 * Sets the country or region.
	 * 
	 * @param value
	 *            the new country or region
	 */
	public void setCountryOrRegion(String value) {
		this.propertyBag.setSimplePropertyBag(
				PhysicalAddressSchema.CountryOrRegion, value);
	}

	/**
	 * Gets the postal code.
	 * 
	 * @return the postal code
	 */
	public String getPostalCode() {
		return (String)this.propertyBag
				.getSimplePropertyBag(PhysicalAddressSchema.PostalCode);
	}

	/**
	 * Sets the postal code.
	 * 
	 * @param value
	 *            the new postal code
	 */
	public void setPostalCode(String value) {
		this.propertyBag.setSimplePropertyBag(PhysicalAddressSchema.PostalCode,
				value);
	}

	/**
	 * Clears the change log.
	 */
	@Override
	protected void clearChangeLog() {
		this.propertyBag.clearChangeLog();
	}

	/**
	 * Writes elements to XML.
	 * 
	 * @param reader
	 *            the reader
	 * @return true, if successful
	 * @throws Exception
	 *             the exception
	 */
	@Override
	protected boolean tryReadElementFromXml(EwsServiceXmlReader reader)
			throws Exception {
		if (PhysicalAddressSchema.getXmlElementNames().contains(
				reader.getLocalName())) {
			this.propertyBag.setSimplePropertyBag(reader.getLocalName(), reader
					.readElementValue());
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Writes elements to XML.
	 * 
	 * @param writer
	 *            the writer
	 * @throws XMLStreamException
	 *             the xML stream exception
	 * @throws ServiceXmlSerializationException
	 *             the service xml serialization exception
	 */
	@Override
	protected void writeElementsToXml(EwsServiceXmlWriter writer)
			throws XMLStreamException, ServiceXmlSerializationException {
		for (String xmlElementName : PhysicalAddressSchema.getXmlElementNames()) {
			writer.writeElementValue(XmlNamespace.Types, xmlElementName,
					this.propertyBag.getSimplePropertyBag(xmlElementName));

		}
	}

	/**
	 * Writes the update to XML.
	 * 
	 * @param writer
	 *            the writer
	 * @param ewsObject
	 *            the ews object
	 * @param ownerDictionaryXmlElementName
	 *            the owner dictionary xml element name
	 * @return True if update XML was written.
	 * @throws XMLStreamException
	 *             the xML stream exception
	 * @throws ServiceXmlSerializationException
	 *             the service xml serialization exception
	 */
	@Override
	protected boolean writeSetUpdateToXml(EwsServiceXmlWriter writer,
			ServiceObject ewsObject, String ownerDictionaryXmlElementName)
			throws XMLStreamException, ServiceXmlSerializationException {
		List<String> fieldsToSet = new ArrayList<String>();

		for (String xmlElementName : this.propertyBag.getAddedItems()) {
			fieldsToSet.add(xmlElementName);
		}

		for (String xmlElementName : this.propertyBag.getModifiedItems()) {
			fieldsToSet.add(xmlElementName);
		}

		for (String xmlElementName : fieldsToSet) {
			writer.writeStartElement(XmlNamespace.Types, ewsObject
					.getSetFieldXmlElementName());

			writer.writeStartElement(XmlNamespace.Types,
					XmlElementNames.IndexedFieldURI);
			writer.writeAttributeValue(XmlAttributeNames.FieldURI,
					getFieldUri(xmlElementName));
			writer.writeAttributeValue(XmlAttributeNames.FieldIndex, this
					.getKey().toString());
			writer.writeEndElement(); // IndexedFieldURI

			writer.writeStartElement(XmlNamespace.Types, ewsObject
					.getXmlElementName());
			writer.writeStartElement(XmlNamespace.Types,
					ownerDictionaryXmlElementName);
			writer.writeStartElement(XmlNamespace.Types, XmlElementNames.Entry);
			this.writeAttributesToXml(writer);
			writer.writeElementValue(XmlNamespace.Types, xmlElementName,
					this.propertyBag.getSimplePropertyBag(xmlElementName));
			writer.writeEndElement(); // Entry
			writer.writeEndElement(); // ownerDictionaryXmlElementName
			writer.writeEndElement(); // ewsObject.GetXmlElementName()

			writer.writeEndElement(); // ewsObject.GetSetFieldXmlElementName()
		}

		for (String xmlElementName : this.propertyBag.getRemovedItems()) {
			this.internalWriteDeleteFieldToXml(writer, ewsObject,
					xmlElementName);
		}

		return true;
	}

	/**
	 * Writes the delete update to XML.
	 * 
	 * @param writer
	 *            the writer
	 * @param ewsObject
	 *            the ews object
	 * @return True if update XML was written.
	 * @throws XMLStreamException
	 *             the xML stream exception
	 * @throws ServiceXmlSerializationException
	 *             the service xml serialization exception
	 */
	@Override
	protected boolean writeDeleteUpdateToXml(EwsServiceXmlWriter writer,
			ServiceObject ewsObject) throws XMLStreamException,
			ServiceXmlSerializationException {
		for (String xmlElementName : PhysicalAddressSchema.getXmlElementNames()) {
			this.internalWriteDeleteFieldToXml(writer, ewsObject,
					xmlElementName);
		}
		return true;
	}

	/**
	 * Gets the field URI.
	 * 
	 * @param xmlElementName
	 *            the xml element name
	 * @return Field URI.
	 */
	private static String getFieldUri(String xmlElementName) {
		return "contacts:PhysicalAddress:" + xmlElementName;
	}

	/**
	 * Property bag was changed.
	 */
	private void propertyBagChanged() {
		this.changed();
	}

	/**
	 * Write field deletion to XML.
	 * 
	 * @param writer
	 *            the writer
	 * @param ewsObject
	 *            the ews object
	 * @param fieldXmlElementName
	 *            the field xml element name
	 * @throws XMLStreamException
	 *             the xML stream exception
	 * @throws ServiceXmlSerializationException
	 *             the service xml serialization exception
	 */
	private void internalWriteDeleteFieldToXml(EwsServiceXmlWriter writer,
			ServiceObject ewsObject, String fieldXmlElementName)
			throws XMLStreamException, ServiceXmlSerializationException {
		writer.writeStartElement(XmlNamespace.Types, ewsObject
				.getDeleteFieldXmlElementName());
		writer.writeStartElement(XmlNamespace.Types,
				XmlElementNames.IndexedFieldURI);
		writer.writeAttributeValue(XmlAttributeNames.FieldURI,
				getFieldUri(fieldXmlElementName));
		writer.writeAttributeValue(XmlAttributeNames.FieldIndex, this.getKey()
				.toString());
		writer.writeEndElement(); // IndexedFieldURI
		writer.writeEndElement(); // ewsObject.GetDeleteFieldXmlElementName()
	}

	/**
	 * Schema definition for PhysicalAddress.
	 */
	private static class PhysicalAddressSchema {

		/** The Constant Street. */
		public static final String Street = "Street";

		/** The Constant City. */
		public static final String City = "City";

		/** The Constant State. */
		public static final String State = "State";

		/** The Constant CountryOrRegion. */
		public static final String CountryOrRegion = "CountryOrRegion";

		/** The Constant PostalCode. */
		public static final String PostalCode = "PostalCode";

		/**
		 * List of XML element names.
		 */
		private static LazyMember<List<String>> xmlElementNames = 
			new LazyMember<List<String>>(

		new ILazyMember<List<String>>() {
			@Override
			public List<String> createInstance() {
				List<String> result = new ArrayList<String>();
				result.add(Street);
				result.add(City);
				result.add(State);
				result.add(CountryOrRegion);
				result.add(PostalCode);
				return result;
			}
		});

		/**
		 * Gets the XML element names.
		 * 
		 * @return The XML element names.
		 */
		public static List<String> getXmlElementNames() {
			return xmlElementNames.getMember();
		}
	}

}
