/*
 * Decompiled with CFR 0.152.
 */
package com.monitorjbl.xlsx;

import com.monitorjbl.xlsx.XmlUtils;
import com.monitorjbl.xlsx.exceptions.CloseException;
import com.monitorjbl.xlsx.exceptions.MissingSheetException;
import com.monitorjbl.xlsx.exceptions.OpenException;
import com.monitorjbl.xlsx.exceptions.ReadException;
import com.monitorjbl.xlsx.impl.StreamingCell;
import com.monitorjbl.xlsx.impl.StreamingRow;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.Characters;
import javax.xml.stream.events.EndElement;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.openxml4j.exceptions.OpenXML4JException;
import org.apache.poi.openxml4j.opc.OPCPackage;
import org.apache.poi.poifs.crypt.Decryptor;
import org.apache.poi.poifs.crypt.EncryptionInfo;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.apache.poi.ss.usermodel.BuiltinFormats;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DataFormatter;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.util.CellReference;
import org.apache.poi.xssf.eventusermodel.XSSFReader;
import org.apache.poi.xssf.model.SharedStringsTable;
import org.apache.poi.xssf.model.StylesTable;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFRichTextString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class StreamingReader
implements Iterable<Row>,
AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(StreamingReader.class);
    private final SharedStringsTable sst;
    private final StylesTable stylesTable;
    private final XMLEventReader parser;
    private final DataFormatter dataFormatter = new DataFormatter();
    private int rowCacheSize;
    private List<Row> rowCache = new ArrayList<Row>();
    private Iterator<Row> rowCacheIterator;
    private String lastContents;
    private StreamingRow currentRow;
    private StreamingCell currentCell;
    private File tmp;
    private OPCPackage pkg;

    private StreamingReader(OPCPackage pkg, SharedStringsTable sst, StylesTable stylesTable, XMLEventReader parser, int rowCacheSize) {
        this.pkg = pkg;
        this.sst = sst;
        this.stylesTable = stylesTable;
        this.parser = parser;
        this.rowCacheSize = rowCacheSize;
    }

    private boolean getRow() {
        try {
            this.rowCache.clear();
            while (this.rowCache.size() < this.rowCacheSize && this.parser.hasNext()) {
                this.handleEvent(this.parser.nextEvent());
            }
            this.rowCacheIterator = this.rowCache.iterator();
            return this.rowCacheIterator.hasNext();
        }
        catch (XMLStreamException | SAXException e) {
            log.debug("End of stream");
            return false;
        }
    }

    private void handleEvent(XMLEvent event) throws SAXException {
        if (event.getEventType() == 4) {
            Characters c = event.asCharacters();
            this.lastContents = this.lastContents + c.getData();
        } else if (event.getEventType() == 1) {
            StartElement startElement = event.asStartElement();
            String tagLocalName = startElement.getName().getLocalPart();
            if ("row".equals(tagLocalName)) {
                Attribute rowIndex = startElement.getAttributeByName(new QName("r"));
                this.currentRow = new StreamingRow(Integer.parseInt(rowIndex.getValue()) - 1);
            } else if ("c".equals(tagLocalName)) {
                Attribute ref = startElement.getAttributeByName(new QName("r"));
                String[] coord = ref.getValue().split("(?<=\\D)(?=\\d)|(?<=\\d)(?=\\D)");
                this.currentCell = new StreamingCell(CellReference.convertColStringToIndex((String)coord[0]), Integer.parseInt(coord[1]) - 1);
                this.setFormatString(startElement, this.currentCell);
                Attribute type = startElement.getAttributeByName(new QName("t"));
                if (type != null) {
                    this.currentCell.setType(type.getValue());
                } else {
                    this.currentCell.setType("n");
                }
                Attribute style = startElement.getAttributeByName(new QName("s"));
                if (style != null) {
                    String indexStr = style.getValue();
                    try {
                        int index = Integer.parseInt(indexStr);
                        this.currentCell.setCellStyle((CellStyle)this.stylesTable.getStyleAt(index));
                    }
                    catch (NumberFormatException nfe) {
                        log.warn("Ignoring invalid style index {}", (Object)indexStr);
                    }
                }
            }
            this.lastContents = "";
        } else if (event.getEventType() == 2) {
            EndElement endElement = event.asEndElement();
            String tagLocalName = endElement.getName().getLocalPart();
            if ("v".equals(tagLocalName) || "t".equals(tagLocalName)) {
                this.currentCell.setRawContents(this.unformattedContents());
                this.currentCell.setContents(this.formattedContents());
            } else if ("row".equals(tagLocalName) && this.currentRow != null) {
                this.rowCache.add(this.currentRow);
            } else if ("c".equals(tagLocalName)) {
                this.currentRow.getCellMap().put(this.currentCell.getColumnIndex(), this.currentCell);
            }
        }
    }

    void setFormatString(StartElement startElement, StreamingCell cell) {
        Attribute cellStyle = startElement.getAttributeByName(new QName("s"));
        String cellStyleString = cellStyle != null ? cellStyle.getValue() : null;
        XSSFCellStyle style = null;
        if (cellStyleString != null) {
            style = this.stylesTable.getStyleAt(Integer.parseInt(cellStyleString));
        } else if (this.stylesTable.getNumCellStyles() > 0) {
            style = this.stylesTable.getStyleAt(0);
        }
        if (style != null) {
            cell.setNumericFormatIndex(style.getDataFormat());
            String formatString = style.getDataFormatString();
            if (formatString != null) {
                cell.setNumericFormat(formatString);
            } else {
                cell.setNumericFormat(BuiltinFormats.getBuiltinFormat((int)cell.getNumericFormatIndex().shortValue()));
            }
        } else {
            cell.setNumericFormatIndex(null);
            cell.setNumericFormat(null);
        }
    }

    String formattedContents() {
        switch (this.currentCell.getType()) {
            case "s": {
                int idx = Integer.parseInt(this.lastContents);
                return new XSSFRichTextString(this.sst.getEntryAt(idx)).toString();
            }
            case "inlineStr": {
                return new XSSFRichTextString(this.lastContents).toString();
            }
            case "str": {
                return '\"' + this.lastContents + '\"';
            }
            case "e": {
                return "ERROR:  " + this.lastContents;
            }
            case "n": {
                if (this.currentCell.getNumericFormat() != null && this.lastContents.length() > 0) {
                    return this.dataFormatter.formatRawCellContents(Double.parseDouble(this.lastContents), (int)this.currentCell.getNumericFormatIndex().shortValue(), this.currentCell.getNumericFormat());
                }
                return this.lastContents;
            }
        }
        return this.lastContents;
    }

    String unformattedContents() {
        switch (this.currentCell.getType()) {
            case "s": {
                int idx = Integer.parseInt(this.lastContents);
                return new XSSFRichTextString(this.sst.getEntryAt(idx)).toString();
            }
            case "inlineStr": {
                return new XSSFRichTextString(this.lastContents).toString();
            }
        }
        return this.lastContents;
    }

    @Override
    public Iterator<Row> iterator() {
        return new StreamingIterator();
    }

    @Override
    public void close() {
        try {
            this.parser.close();
            this.pkg.revert();
        }
        catch (XMLStreamException e) {
            throw new CloseException(e);
        }
        if (this.tmp != null) {
            log.debug("Deleting tmp file [" + this.tmp.getAbsolutePath() + "]");
            this.tmp.delete();
        }
    }

    static File writeInputStreamToFile(InputStream is, int bufferSize) throws IOException {
        File f = Files.createTempFile("tmp-", ".xlsx", new FileAttribute[0]).toFile();
        try (FileOutputStream fos = new FileOutputStream(f);){
            int read;
            byte[] bytes = new byte[bufferSize];
            while ((read = is.read(bytes)) != -1) {
                fos.write(bytes, 0, read);
            }
            is.close();
            fos.close();
            File file = f;
            return file;
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    class StreamingIterator
    implements Iterator<Row> {
        public StreamingIterator() {
            if (StreamingReader.this.rowCacheIterator == null) {
                this.hasNext();
            }
        }

        @Override
        public boolean hasNext() {
            return StreamingReader.this.rowCacheIterator != null && StreamingReader.this.rowCacheIterator.hasNext() || StreamingReader.this.getRow();
        }

        @Override
        public Row next() {
            return (Row)StreamingReader.this.rowCacheIterator.next();
        }

        @Override
        public void remove() {
            throw new RuntimeException("NotSupported");
        }
    }

    public static class Builder {
        int rowCacheSize = 10;
        int bufferSize = 1024;
        int sheetIndex = 0;
        String sheetName;
        String password;

        public Builder rowCacheSize(int rowCacheSize) {
            this.rowCacheSize = rowCacheSize;
            return this;
        }

        public Builder bufferSize(int bufferSize) {
            this.bufferSize = bufferSize;
            return this;
        }

        public Builder sheetIndex(int sheetIndex) {
            this.sheetIndex = sheetIndex;
            return this;
        }

        public Builder sheetName(String sheetName) {
            this.sheetName = sheetName;
            return this;
        }

        public Builder password(String password) {
            this.password = password;
            return this;
        }

        public StreamingReader read(InputStream is) {
            File f = null;
            try {
                f = StreamingReader.writeInputStreamToFile(is, this.bufferSize);
                log.debug("Created temp file [" + f.getAbsolutePath() + "]");
                StreamingReader r = this.read(f);
                r.tmp = f;
                return r;
            }
            catch (IOException e) {
                throw new ReadException("Unable to read input stream", e);
            }
            catch (RuntimeException e) {
                f.delete();
                throw e;
            }
        }

        public StreamingReader read(File f) {
            try {
                OPCPackage pkg;
                if (this.password != null) {
                    POIFSFileSystem poifs = new POIFSFileSystem(f);
                    EncryptionInfo info = new EncryptionInfo(poifs);
                    Decryptor d = Decryptor.getInstance((EncryptionInfo)info);
                    d.verifyPassword(this.password);
                    pkg = OPCPackage.open((InputStream)d.getDataStream(poifs));
                } else {
                    pkg = OPCPackage.open((File)f);
                }
                XSSFReader reader = new XSSFReader(pkg);
                SharedStringsTable sst = reader.getSharedStringsTable();
                StylesTable styles = reader.getStylesTable();
                InputStream sheet = this.findSheet(reader);
                if (sheet == null) {
                    throw new MissingSheetException("Unable to find sheet at index [" + this.sheetIndex + "]");
                }
                XMLEventReader parser = XMLInputFactory.newInstance().createXMLEventReader(sheet);
                return new StreamingReader(pkg, sst, styles, parser, this.rowCacheSize);
            }
            catch (IOException e) {
                throw new OpenException("Failed to open file", e);
            }
            catch (XMLStreamException | OpenXML4JException e) {
                throw new ReadException("Unable to read workbook", (Exception)e);
            }
            catch (GeneralSecurityException e) {
                throw new ReadException("Unable to read workbook - Decryption failed", e);
            }
        }

        InputStream findSheet(XSSFReader reader) throws IOException, InvalidFormatException {
            int index = this.sheetIndex;
            if (this.sheetName != null) {
                index = -1;
                NodeList nl = XmlUtils.searchForNodeList(XmlUtils.document(reader.getWorkbookData()), "/workbook/sheets/sheet");
                for (int i = 0; i < nl.getLength(); ++i) {
                    if (!Objects.equals(nl.item(i).getAttributes().getNamedItem("name").getTextContent(), this.sheetName)) continue;
                    index = i;
                }
                if (index < 0) {
                    return null;
                }
            }
            Iterator iter = reader.getSheetsData();
            InputStream sheet = null;
            int i = 0;
            while (iter.hasNext()) {
                InputStream is = (InputStream)iter.next();
                if (i++ != index) continue;
                sheet = is;
                log.debug("Found sheet at index [" + this.sheetIndex + "]");
                break;
            }
            return sheet;
        }
    }
}

