/*
 * Decompiled with CFR 0.152.
 */
package com.newrelic.agent.errors;

import com.newrelic.agent.Agent;
import com.newrelic.agent.IRPMService;
import com.newrelic.agent.Transaction;
import com.newrelic.agent.TransactionData;
import com.newrelic.agent.TransactionListener;
import com.newrelic.agent.config.IAgentConfig;
import com.newrelic.agent.config.IAgentConfigListener;
import com.newrelic.agent.config.IErrorCollectorConfig;
import com.newrelic.agent.errors.ExceptionHandlerPointCut;
import com.newrelic.agent.errors.ExceptionHandlerSignature;
import com.newrelic.agent.errors.HttpTracedError;
import com.newrelic.agent.errors.ThrowableError;
import com.newrelic.agent.errors.TracedError;
import com.newrelic.agent.instrumentation.PointCut;
import com.newrelic.agent.instrumentation.methodmatchers.InvalidMethodDescriptor;
import com.newrelic.agent.instrumentation.yaml.PointCutFactory;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.stats.StatsEngine;
import com.newrelic.agent.stats.TransactionStats;
import com.newrelic.agent.tracers.ClassMethodSignature;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.logging.Level;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class ErrorService {
    public static final int ERROR_LIMIT_PER_REPORTING_PERIOD = 20;
    private static final Set<String> IGNORE_ERRORS;
    private final AtomicInteger errorCountThisHarvest = new AtomicInteger();
    private final AtomicInteger errorCount = new AtomicInteger();
    private final AtomicLong totalErrorCount = new AtomicLong();
    private final AtomicReferenceArray<TracedError> tracedErrors;
    private volatile IErrorCollectorConfig errorCollectorConfig;
    private final boolean shouldRecordErrorCount;
    private final String appName;

    public ErrorService(String appName) {
        this.appName = appName;
        this.errorCollectorConfig = ServiceFactory.getConfigService().getErrorCollectorConfig(appName);
        this.tracedErrors = new AtomicReferenceArray(20);
        ServiceFactory.getTransactionService().addTransactionListener(new MyTransactionListener());
        ServiceFactory.getConfigService().addIAgentConfigListener(new MyConfigListener());
        this.shouldRecordErrorCount = !Boolean.getBoolean("com.newrelic.agent.errors.no_error_metric");
    }

    protected void refreshErrorCollectorConfig(IAgentConfig agentConfig) {
        IErrorCollectorConfig oldErrorConfig = this.errorCollectorConfig;
        this.errorCollectorConfig = agentConfig.getErrorCollectorConfig();
        if (this.errorCollectorConfig.isEnabled() == oldErrorConfig.isEnabled()) {
            return;
        }
        String msg = MessageFormat.format("Errors will{0} be sent to New Relic for {1}", this.errorCollectorConfig.isEnabled() ? "" : " not", this.appName);
        Agent.LOG.info(msg);
    }

    public void reportError(TracedError error) {
        if (error == null) {
            return;
        }
        if (error.incrementsErrorMetric()) {
            this.errorCountThisHarvest.incrementAndGet();
        }
        if (!this.errorCollectorConfig.isEnabled()) {
            return;
        }
        if (this.errorCount.get() >= 20) {
            Agent.LOG.finer(MessageFormat.format("Error limit exceeded for {0}: {1}", this.appName, error));
            return;
        }
        int index = (int)this.totalErrorCount.getAndIncrement() % 20;
        if (this.tracedErrors.compareAndSet(index, null, error)) {
            this.errorCount.getAndIncrement();
            if (Agent.LOG.isLoggable(Level.FINER)) {
                String msg = MessageFormat.format("Recording error for {0} : {1}", this.appName, error);
                Agent.LOG.finer(msg);
            }
        }
    }

    public void reportErrors(TracedError ... errors) {
        for (TracedError error : errors) {
            this.reportError(error);
        }
    }

    public List<TracedError> getTracedErrors() {
        ArrayList<TracedError> errors = new ArrayList<TracedError>(20);
        for (int i = 0; i < this.tracedErrors.length(); ++i) {
            TracedError error = this.tracedErrors.getAndSet(i, null);
            if (error == null) continue;
            this.errorCount.getAndDecrement();
            errors.add(error);
        }
        return errors;
    }

    public List<TracedError> harvest(IRPMService rpmService, StatsEngine statsEngine) {
        if (!this.errorCollectorConfig.isEnabled()) {
            return Collections.emptyList();
        }
        this.recordMetrics(statsEngine);
        if (rpmService.isConnected()) {
            return this.getTracedErrors();
        }
        return Collections.emptyList();
    }

    private void recordMetrics(StatsEngine statsEngine) {
        int errorCount = this.errorCountThisHarvest.getAndSet(0);
        if (this.shouldRecordErrorCount) {
            statsEngine.getStats("Errors/all").incrementCallCount(errorCount);
        }
    }

    private void noticeTransaction(TransactionData td, TransactionStats transactionStats) {
        if (!this.appName.equals(td.getApplicationName())) {
            return;
        }
        String statusMessage = td.getStatusMessage();
        int responseStatus = td.getResponseStatus();
        Throwable throwable = td.getThrowable();
        boolean customError = false;
        if (throwable instanceof ReportableException) {
            throwable = throwable.getCause();
            if (responseStatus < 400) {
                responseStatus = 500;
            }
            customError = true;
        } else if (throwable instanceof ReportableError) {
            statusMessage = throwable.getMessage();
            throwable = null;
            responseStatus = 500;
            customError = true;
        } else if (throwable != null && responseStatus < 400) {
            responseStatus = 500;
        }
        if (responseStatus >= 400) {
            if (!this.errorCollectorConfig.isEnabled()) {
                return;
            }
            if (!customError && this.isIgnoredError(td)) {
                if (Agent.LOG.isLoggable(Level.FINER)) {
                    String errorString = throwable == null ? "" : throwable.getClass().getName();
                    String msg = MessageFormat.format("Ignoring error {0} for {1} {2} ({3})", errorString, td.getRequestUri(), this.appName, responseStatus);
                    Agent.LOG.finer(msg);
                }
                return;
            }
            TracedError error = throwable != null ? new ThrowableError(this.appName, td.getBlameOrRootMetricName(), throwable, td.getRequestUri(), td.getParameters(), td.getStartTime()) : new HttpTracedError(this.appName, td.getBlameOrRootMetricName(), responseStatus, statusMessage, td.getRequestUri(), td.getParameters(), td.getRootTracer().getStartTime());
            if (this.shouldRecordErrorCount && error.incrementsErrorMetric()) {
                this.recordErrorCount(td, transactionStats);
            }
            if (this.errorCount.get() < 20) {
                this.reportError(error);
            } else if (error.incrementsErrorMetric()) {
                this.errorCountThisHarvest.incrementAndGet();
            }
        }
    }

    private void recordErrorCount(TransactionData td, TransactionStats transactionStats) {
        String metricName = this.getErrorCountMetricName(td);
        if (metricName != null) {
            transactionStats.getUnscopedStats().getStats(metricName).incrementCallCount();
        }
        String metricNameAll = td.isWebTransaction() ? "Errors/allWeb" : "Errors/allOther";
        transactionStats.getUnscopedStats().getStats(metricNameAll).incrementCallCount();
    }

    private String getErrorCountMetricName(TransactionData td) {
        String blameMetricName = td.getBlameMetricName();
        if (blameMetricName != null) {
            return "Errors/" + blameMetricName;
        }
        return null;
    }

    public boolean isIgnoredError(int responseStatus, Throwable throwable) {
        if (this.errorCollectorConfig.getIgnoreStatusCodes().contains(responseStatus)) {
            return true;
        }
        while (throwable != null) {
            String name = throwable.getClass().getName();
            if (this.errorCollectorConfig.getIgnoreErrors().contains(name)) {
                return true;
            }
            if (IGNORE_ERRORS.contains(name)) {
                return true;
            }
            throwable = throwable.getCause();
        }
        return false;
    }

    public boolean isIgnoredError(TransactionData transactionData) {
        return this.isIgnoredError(transactionData.getResponseStatus(), transactionData.getThrowable());
    }

    public static void reportException(Throwable throwable, Map<String, String> params) {
        Transaction transaction = Transaction.getTransaction();
        if (transaction.getLastTracer() == null) {
            String uri = '/' + Thread.currentThread().getName();
            ThrowableError error = new ThrowableError(null, "OtherTransaction" + uri, throwable, uri, params, System.currentTimeMillis());
            ServiceFactory.getRPMService().getErrorService().reportError(error);
        } else {
            if (params != null) {
                transaction.getParameters().putAll(params);
            }
            transaction.setThrowable(new ReportableException(throwable));
        }
    }

    public static void reportError(String message, Map<String, String> params) {
        Transaction transaction = Transaction.getTransaction();
        if (transaction.getLastTracer() == null) {
            String uri = '/' + Thread.currentThread().getName();
            HttpTracedError error = new HttpTracedError(null, "OtherTransaction" + uri, 500, message, uri, params, System.currentTimeMillis());
            ServiceFactory.getRPMService().getErrorService().reportError(error);
        } else {
            if (params != null) {
                transaction.getParameters().putAll(params);
            }
            transaction.setThrowable(new ReportableError(message));
        }
    }

    public static Collection<? extends PointCut> getErrorHandlerPointCuts() {
        IAgentConfig config = ServiceFactory.getConfigService().getDefaultAgentConfig();
        Object exceptionHandlers = config.getErrorCollectorConfig().getProperty("exception_handlers");
        if (exceptionHandlers == null) {
            return Collections.emptyList();
        }
        ArrayList<ExceptionHandlerPointCut> pointcuts = new ArrayList<ExceptionHandlerPointCut>();
        if (exceptionHandlers instanceof Collection) {
            for (Object sigObject : (Collection)exceptionHandlers) {
                ExceptionHandlerPointCut pc;
                if (sigObject instanceof ExceptionHandlerSignature) {
                    ExceptionHandlerSignature exHandlerSig = (ExceptionHandlerSignature)sigObject;
                    String msg = MessageFormat.format("Instrumenting exception handler signature {0}", exHandlerSig.toString());
                    Agent.LOG.finer(msg);
                    pc = new ExceptionHandlerPointCut(exHandlerSig);
                    pointcuts.add(pc);
                    continue;
                }
                if (sigObject instanceof String) {
                    ClassMethodSignature signature = PointCutFactory.parseClassMethodSignature(sigObject.toString());
                    try {
                        ExceptionHandlerSignature exHandlerSig = new ExceptionHandlerSignature(signature);
                        Agent.LOG.info(MessageFormat.format("Instrumenting exception handler signature {0}", exHandlerSig.toString()));
                        pc = new ExceptionHandlerPointCut(exHandlerSig);
                        pointcuts.add(pc);
                    }
                    catch (InvalidMethodDescriptor e) {
                        Agent.LOG.severe(MessageFormat.format("Unable to instrument exception handler {0} : {1}", sigObject.toString(), e.toString()));
                    }
                    continue;
                }
                if (!(sigObject instanceof Exception)) continue;
                Agent.LOG.severe(MessageFormat.format("Unable to instrument exception handler : {0}", sigObject.toString()));
            }
        }
        return pointcuts;
    }

    static {
        HashSet<String> ignoreErrors = new HashSet<String>(1);
        ignoreErrors.add("org.eclipse.jetty.continuation.ContinuationThrowable");
        ignoreErrors.add("org.mortbay.jetty.RetryRequest");
        IGNORE_ERRORS = Collections.unmodifiableSet(ignoreErrors);
    }

    private class MyConfigListener
    implements IAgentConfigListener {
        private MyConfigListener() {
        }

        public void configChanged(String appName, IAgentConfig agentConfig) {
            if (ErrorService.this.appName.equals(appName)) {
                Agent.LOG.fine(MessageFormat.format("Error service received configuration change notification for {0}", appName));
                ErrorService.this.refreshErrorCollectorConfig(agentConfig);
            }
        }
    }

    private class MyTransactionListener
    implements TransactionListener {
        private MyTransactionListener() {
        }

        public void dispatcherTransactionFinished(TransactionData transactionData, TransactionStats transactionStats) {
            ErrorService.this.noticeTransaction(transactionData, transactionStats);
        }
    }

    private static class ReportableException
    extends Throwable {
        private static final long serialVersionUID = -6926170846485250498L;

        ReportableException(Throwable cause) {
            super(cause);
        }
    }

    private static class ReportableError
    extends Throwable {
        private static final long serialVersionUID = 3472056044517410355L;

        public ReportableError(String message) {
            super(message);
        }
    }
}

