package com.newrelic.api.agent;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * If you annotate a method with the Trace annotation, it will be automatically measured by New Relic, with the
 * following metrics - call count - calls per minute - average call time - standard deviation call time - min and max
 * call time
 * 
 * Also these metrics will be traced inside the call scope of a named http transaction (e.g. "/servlets/myservlet" so
 * that New Relic can "break out" the response time of a given transaction by specific called methods.
 * 
 * Be mindful when using this attribute. When placed on relatively heavyweight operations such as database access
 * webservice invocation, its overhead will be negligible. On the other hand, if you placed it on a tight, frequently
 * called method (e.g. an accessor that is called thousands of times per second), then the tracer will introduce higher
 * overhead to your application.
 * 
 * @author cirne
 */
@Target(ElementType.METHOD)
@Retention(RetentionPolicy.RUNTIME)
public @interface Trace {
    public static final String NULL = "";

    String metricName() default NULL;

    boolean dispatcher() default false;

    boolean nameTransaction() default false;

    String tracerFactoryName() default NULL;

    boolean skipTransactionTrace() default false;
}