/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2007, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.security;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.spec.DESKeySpec;
import javax.crypto.SecretKeyFactory;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import oracle.toplink.essentials.internal.helper.Helper;
import oracle.toplink.essentials.exceptions.ValidationException;
import oracle.toplink.essentials.exceptions.ConversionException;

/**
 * TopLink reference implementation for password encryption.
 *
 * @author Guy Pelletier
 */
public class JCEEncryptor implements Securable {
    private Cipher m_cipher;
    private final String m_algorithm = "DES";
    private final String m_padding = "DES/ECB/PKCS5Padding";

    public JCEEncryptor() throws Exception {

        /*
         * We want to force the initialization of the cipher here. This is a fix
         * for bug #2696486.
         * JDev with JDK 1.3 in some cases will allow a JCE object to be created
         * when it shouldn't. That is, JDev includes an incompletely configured JCE
         * library for JDK 1.3, meaning JCE will not run properly in the VM. So, JDev
         * allows you to create a JCEEncryptor object, but eventually throw's
         * errors when trying to make JCE library calls from encryptPassword.
         *
         * Confusing??? Well, don't move this code before talking to Guy first!
         */
        m_cipher = Cipher.getInstance(m_padding);
    }

    /**
     * Encrypts a string. Will throw a validation exception.
     */
    public synchronized String encryptPassword(String password) {
        try {
            m_cipher.init(Cipher.ENCRYPT_MODE, Synergizer.getMultitasker(m_algorithm));

            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            CipherOutputStream cos = new CipherOutputStream(baos, m_cipher);
            ObjectOutputStream oos = new ObjectOutputStream(cos);

            oos.writeObject(password);
            oos.flush();
            oos.close();

            return Helper.buildHexStringFromBytes(baos.toByteArray());
        } catch (Exception e) {
            throw ValidationException.errorEncryptingPassword(e);
        }
    }

    /**
     * Decrypts a string. Will throw a validation exception.
     * Handles backwards compatability for older encrypted strings.
     */
    public synchronized String decryptPassword(String encryptedPswd) {
        String password = "";

        try {
            m_cipher.init(Cipher.DECRYPT_MODE, Synergizer.getMultitasker(m_algorithm));

            byte[] bytePassword = Helper.buildBytesFromHexString(encryptedPswd);

            ByteArrayInputStream bais = new ByteArrayInputStream(bytePassword);
            CipherInputStream cis = new CipherInputStream(bais, m_cipher);
            ObjectInputStream ois = new ObjectInputStream(cis);

            password = (String)ois.readObject();
            ois.close();
        } catch (IOException e) {
            // JCE 1.2.2 couldn't decrypt it, assume clear text
            password = encryptedPswd;
        } catch (ArrayIndexOutOfBoundsException e) {
            // JCE 1.2.1 couldn't decrypt it, assume clear text
            password = encryptedPswd;
        } catch (ConversionException e) {
            // Never prepared (buildBytesFromHexString failed), assume clear text
            password = encryptedPswd;
        } catch (Exception e) {
            throw ValidationException.errorDecryptingPassword(e);
        }

        return password;
    }

    private static class Synergizer {
        private static String multitasker = "E60B80C7AEC78038";

        static public SecretKey getMultitasker(String algorithm) throws Exception {
            SecretKeyFactory skf = SecretKeyFactory.getInstance(algorithm);
            return skf.generateSecret(new DESKeySpec(Helper.buildBytesFromHexString(multitasker)));
        }
    }
}
