/*
 * Copyright 2004 Sun Microsystems, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package com.rometools.rome.feed.module;

import java.io.Serializable;

import com.rometools.rome.feed.impl.ObjectBean;

/**
 * Base class for modules describing Metadata of feeds, default implementations. Examples of such
 * modules are the Dublin Core and Syndication modules.
 * <p>
 *
 * @author Alejandro Abdelnur
 *
 */
public abstract class ModuleImpl implements Cloneable, Serializable, Module {
    private static final long serialVersionUID = 1L;
    private final ObjectBean objBean;
    private final String uri;

    /**
     * Constructor.
     * <p>
     *
     * @param uri URI of the module.
     *
     */
    protected ModuleImpl(final Class<?> beanClass, final String uri) {
        objBean = new ObjectBean(beanClass, this);
        this.uri = uri;
    }

    /**
     * Creates a deep 'bean' clone of the object.
     * <p>
     *
     * @return a clone of the object.
     * @throws CloneNotSupportedException thrown if an element of the object cannot be cloned.
     *
     */
    @Override
    public Object clone() throws CloneNotSupportedException {
        return objBean.clone();
    }

    /**
     * Indicates whether some other object is "equal to" this one as defined by the Object equals()
     * method.
     * <p>
     *
     * @param other he reference object with which to compare.
     * @return <b>true</b> if 'this' object is equal to the 'other' object.
     *
     */
    @Override
    public boolean equals(final Object other) {
        if (!(other instanceof ModuleImpl)) {
            return false;
        }
        return objBean.equals(other);
    }

    /**
     * Returns a hashcode value for the object.
     * <p>
     * It follows the contract defined by the Object hashCode() method.
     * <p>
     *
     * @return the hashcode of the bean object.
     *
     */
    @Override
    public int hashCode() {
        return objBean.hashCode();
    }

    /**
     * Returns the String representation for the object.
     * <p>
     *
     * @return String representation for the object.
     *
     */
    @Override
    public String toString() {
        return objBean.toString();
    }

    /**
     * Returns the URI of the module.
     * <p>
     *
     * @return URI of the module.
     *
     */
    @Override
    public String getUri() {
        return uri;
    }

}
