/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.sap.cloud.sdk.datamodel.odata.client.request;

import java.math.BigDecimal;
import java.util.function.Consumer;

import javax.annotation.Nonnull;

import com.google.gson.GsonBuilder;
import com.google.gson.internal.bind.CustomMapTypeAdapterFactory;
import com.google.gson.stream.JsonToken;
import com.sap.cloud.sdk.datamodel.odata.client.adapter.BigDecimalAdapter;

import lombok.RequiredArgsConstructor;

/**
 * Number deserialization strategy to determine behavior for JSON numbers without target type references.
 */
@RequiredArgsConstructor
public enum NumberDeserializationStrategy
{

    /**
     * Double strategy to deserialize JSON numbers to double, if no target type references are specified.
     */
    DOUBLE(gsonBuilder -> {
        // default behavior of GSON
    }),

    /**
     * BigDecimal strategy to deserialize JSON numbers to BigDecimal, if no target type references are specified.
     */
    BIG_DECIMAL(gsonBuilder -> {
        final CustomMapTypeAdapterFactory mapTypeAdapter = new CustomMapTypeAdapterFactory();
        mapTypeAdapter.registerAdapter(JsonToken.NUMBER, BigDecimal.class::isInstance, new BigDecimalAdapter());
        gsonBuilder.registerTypeAdapterFactory(mapTypeAdapter);
    });

    private final Consumer<GsonBuilder> adapter;

    /**
     * Adjust the deserialization strategy for untyped numbers.
     * 
     * @param gsonBuilder
     *            The GsonBuilder to change.
     */
    void decorate( @Nonnull final GsonBuilder gsonBuilder )
    {
        adapter.accept(gsonBuilder);
    }
}
